package ru.yandex.direct.core.entity.retargeting.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.IntStream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.bids.container.BidSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.markupcondition.repository.MarkupConditionRepository;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageGeoTree;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingSelection;
import ru.yandex.direct.core.entity.retargeting.container.SwitchRetargeting;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.core.entity.retargeting.service.validation2.AddRetargetingValidationService;
import ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingSetBidsValidationService;
import ru.yandex.direct.core.entity.retargeting.service.validation2.SwitchRetargetingOnBannersValidationService2;
import ru.yandex.direct.core.entity.retargeting.service.validation2.UpdateRetargetingValidationService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ValidationUtils;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.util.GuavaCollectors.toMultimap;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.convertRetargetingsToTargetInterests;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.fillAbsentAutobudgetPriority;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.warningDuplicatedRetargetingId;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.operation.Applicability.isFull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.PathHelper.path;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;
import static ru.yandex.direct.validation.result.ValidationResult.mergeSuccessfulAndInvalidItems;

@ParametersAreNonnullByDefault
@Service
public class RetargetingService implements EntityService<Retargeting, Long> {

    private static final Logger logger = LoggerFactory.getLogger(RetargetingService.class);
    private static final Set<String> ALLOWED_ERROR_FIELDS_ON_SUSPEND_CHANGE_VALIDATION =
            new HashSet<>(asList(Retargeting.ID.name(), Retargeting.IS_SUSPENDED.name()));

    private final ShardHelper shardHelper;
    private final RetargetingRepository retargetingRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final PricePackageRepository pricePackageRepository;
    private final PricePackageGeoTree pricePackageGeoTree;
    private final MarkupConditionRepository markupConditionRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final AddRetargetingValidationService addValidationService2;
    private final DeleteRetargetingValidationService deleteRetargetingValidationService;
    private final ClientService clientService;
    private final ClientGeoService clientGeoService;
    private final RbacService rbacService;
    private final LogPriceService logPriceService;
    private final MailNotificationEventService mailNotificationEventService;
    private final SwitchRetargetingOnBannersValidationService2 switchRetargetingOnBannersValidationService2;
    private final UpdateRetargetingValidationService updateRetargetingValidationService;
    private final RetargetingSetBidsValidationService setBidsValidationService;
    private final AdGroupService adGroupService;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final AddTargetInterestService targetInterestService;
    private final RetargetingDeleteService retargetingDeleteService;

    @Autowired
    public RetargetingService(
            ShardHelper shardHelper,
            RetargetingRepository retargetingRepository,
            RetargetingConditionRepository retargetingConditionRepository,
            AdGroupRepository adGroupRepository,
            CampaignRepository campaignRepository,
            CampaignTypedRepository campaignTypedRepository,
            PricePackageRepository pricePackageRepository,
            PricePackageGeoTree pricePackageGeoTree,
            MarkupConditionRepository markupConditionRepository,
            AggregatedStatusesRepository aggregatedStatusesRepository,
            AddRetargetingValidationService addValidationService2,
            DeleteRetargetingValidationService deleteRetargetingValidationService,
            ClientService clientService,
            ClientGeoService clientGeoService,
            RbacService rbacService,
            LogPriceService logPriceService,
            MailNotificationEventService mailNotificationEventService,
            SwitchRetargetingOnBannersValidationService2 switchRetargetingOnBannersValidationService2,
            UpdateRetargetingValidationService updateRetargetingValidationService,
            RetargetingSetBidsValidationService setBidsValidationService,
            AdGroupService adGroupService,
            CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
            AddTargetInterestService targetInterestService,
            RetargetingDeleteService retargetingDeleteService) {
        this.shardHelper = shardHelper;
        this.retargetingRepository = retargetingRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.pricePackageGeoTree = pricePackageGeoTree;
        this.markupConditionRepository = markupConditionRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.addValidationService2 = addValidationService2;
        this.deleteRetargetingValidationService = deleteRetargetingValidationService;
        this.clientService = clientService;
        this.clientGeoService = clientGeoService;
        this.rbacService = rbacService;
        this.logPriceService = logPriceService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.switchRetargetingOnBannersValidationService2 = switchRetargetingOnBannersValidationService2;
        this.updateRetargetingValidationService = updateRetargetingValidationService;
        this.setBidsValidationService = setBidsValidationService;
        this.adGroupService = adGroupService;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.targetInterestService = targetInterestService;
        this.retargetingDeleteService = retargetingDeleteService;
    }

    /**
     * Получаем список всех {@link RetargetingCondition}, которые являются ссылками на категории интересов.
     */
    public List<InterestLink> getExistingInterest(int shard, ClientId clientId) {
        return retargetingConditionRepository.getExistingInterest(shard, clientId);
    }

    @Override
    public List<Retargeting> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        var targetInterests = getRetargetings(new RetargetingSelection().withIds(List.copyOf(ids)),
                clientId, operatorUid,
                new LimitOffset(Integer.MAX_VALUE, 0));
        return StreamEx.of(targetInterests).map(ti -> (Retargeting) ti).toList();
    }

    public List<TargetInterest> getRetargetings(RetargetingSelection selection,
                                                ClientId clientId, Long operatorUid, LimitOffset limitOffset) {
        int shard = shardFor(clientId);

        List<InterestLink> existingInterests = retargetingConditionRepository.getExistingInterest(shard, clientId);

        // получаем retId ретаргетингов с кампаниями, соответствующих условиями в selection
        Map<Long, Long> retIdToCid =
                retargetingRepository.getRetIdWithCidWithoutLimit(shard, selection, existingInterests);

        // оставляем только те retId, кампании которых видны оператору
        Collection<Long> cids = retIdToCid.values();
        Set<Long> visibleCids = rbacService.getVisibleCampaigns(operatorUid, cids);

        List<Long> retIds = retIdToCid.entrySet().stream()
                .filter(v -> visibleCids.contains(v.getValue()))
                .map(Map.Entry::getKey).collect(toList());

        List<Retargeting> retargetings = retargetingRepository.getRetargetingsByIds(shard, retIds, limitOffset);
        fillAbsentAutobudgetPriority(retargetings);

        return RetargetingUtils.convertRetargetingsToTargetInterests(retargetings, existingInterests);
    }

    public List<Retargeting> getRetargetingByAdGroupIds(Collection<Long> adGroupIds, ClientId clientId) {
        // достает все ретаргетинги из указанных групп
        return retargetingRepository.getRetargetingsByAdGroups(shardFor(clientId), adGroupIds);
    }

    public List<TargetInterest> getTargetInterestsWithInterestByAdGroupIds(
            Collection<Long> adGroupIds, ClientId clientId, int shard) {
        // достает все ретаргетинги из указанных групп
        List<Retargeting> retargetings = retargetingRepository.getRetargetingsByAdGroups(shard, adGroupIds);
        fillAbsentAutobudgetPriority(retargetings);
        // достает из них retCondId
        List<Long> retCondIds = mapList(retargetings, Retargeting::getRetargetingConditionId);
        // по переданным id условий ретаргетинга достает все интересы, а те условия ретаргетинга,
        // которые не являются хранилищами ссылки на интерес, просто игнорирует
        List<InterestLink> interestLinks = retargetingConditionRepository.getInterestByIds(shard, clientId, retCondIds);
        // превращает все ретаргетинги в TargetInterest, но по внутреннему состоянию
        // они становятся либо ретаргетингами, либо таргетингами по интересам
        return convertRetargetingsToTargetInterests(retargetings, interestLinks);
    }

    public MassResult<Long> deleteRetargetings(List<Long> retargetingIds, ClientId clientId, long operatorUid) {
        Objects.requireNonNull(retargetingIds, "retargetingIds");

        long clientUid = rbacService.getChiefByClientId(clientId);

        return createDeleteOperation(Applicability.PARTIAL, retargetingIds, operatorUid, clientId, clientUid)
                .prepareAndApply();
    }

    public MassResult<Long> suspendRetargetings(List<Long> retargetingIds, ClientId clientId, Long operatorUid) {
        return prepareAndExecuteSuspendResumeOperation(retargetingIds, clientId, operatorUid, true);
    }

    @Override
    public MassResult<Long> add(
            ClientId clientId, Long operatorUid, List<Retargeting> entities, Applicability applicability) {
        var targetInterests = StreamEx.of(entities).select(TargetInterest.class).toList();
        return createAddOperation(applicability, targetInterests, operatorUid, clientId, operatorUid)
                .prepareAndApply();
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer, List<Retargeting> entities,
                                 Applicability applicability) {
        var targetInterests = StreamEx.of(entities).select(TargetInterest.class).toList();
        ShowConditionFixedAutoPrices fixedAutoPrices = copyContainer.getShowConditionFixedAutoPrices();
        return createAddOperation(
                applicability,
                false,
                false,
                targetInterests,
                true,
                fixedAutoPrices,
                copyContainer.getShardTo(),
                copyContainer.getOperatorUid(),
                copyContainer.getClientIdTo(),
                copyContainer.getOperatorUid(),
                true)
                .prepareAndApply();
    }

    /**
     * Создание операции добавления TargetInterests с настройкой параметров.
     *
     * @param adGroupsNonexistentOnPrepare Флаг, что группа, в которую добавляют TargetInterest,
     *                                     на данный момент не существует
     * @param autoPrices                   включает режим {@code autoPrices}, см. коммент к классу
     *                                     {@link AddRetargetingsOperation}
     * @param fixedAutoPrices              контейнер с фиксированными ставками для ретаргетингов, где те отсутствуют.
     *                                     Должен быть не {@code null}, если {@code autoPrices == true}
     */
    public AddRetargetingsOperation createAddOperation(Applicability applicability,
                                                       boolean adGroupsNonexistentOnPrepare,
                                                       boolean retargetingConditionsNonexistentOnPrepare,
                                                       List<TargetInterest> targetInterests,
                                                       boolean autoPrices,
                                                       @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                                       long operatorUid, ClientId clientId, long clientUid) {
        int shard = shardFor(clientId);

        return createAddOperation(
                applicability,
                adGroupsNonexistentOnPrepare,
                retargetingConditionsNonexistentOnPrepare,
                targetInterests,
                autoPrices,
                fixedAutoPrices,
                shard,
                operatorUid,
                clientId,
                clientUid,
                false);
    }

    /**
     * Создание операции добавления TargetInterests с настройкой параметров.
     *
     * @param adGroupsNonexistentOnPrepare Флаг, что группа, в которую добавляют TargetInterest,
     *                                     на данный момент не существует
     * @param autoPrices                   включает режим {@code autoPrices}, см. коммент к классу
     *                                     {@link AddRetargetingsOperation}
     * @param fixedAutoPrices              контейнер с фиксированными ставками для ретаргетингов, где те отсутствуют.
     *                                     Должен быть не {@code null}, если {@code autoPrices == true}
     */
    private AddRetargetingsOperation createAddOperation(
            Applicability applicability,
            boolean adGroupsNonexistentOnPrepare,
            boolean retargetingConditionsNonexistentOnPrepare,
            List<TargetInterest> targetInterests,
            boolean autoPrices,
            @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
            int shard,
            long operatorUid,
            ClientId clientId,
            long clientUid,
            boolean isCopy) {
        return new AddRetargetingsOperation(applicability, adGroupsNonexistentOnPrepare,
                retargetingConditionsNonexistentOnPrepare,
                targetInterests,
                this, addValidationService2,
                clientService,
                clientGeoService,
                retargetingConditionRepository,
                adGroupRepository,
                campaignRepository,
                campaignTypedRepository,
                pricePackageRepository,
                pricePackageGeoTree,
                markupConditionRepository,
                cpmYndxFrontpageCurrencyService,
                targetInterestService, autoPrices, fixedAutoPrices,
                operatorUid, clientId, clientUid, shard, isCopy);
    }

    /**
     * Создание операции добавления TargetInterest в существующую группу.
     * Пустые ставки в сети будут заменены на минимальные ставки в клиентской валюте.
     */
    public AddRetargetingsOperation createAddOperation(Applicability applicability,
                                                       List<TargetInterest> targetInterests,
                                                       long operatorUid, ClientId clientId, long clientUid) {
        return createAddOperation(applicability, false, false, targetInterests,
                false, null, operatorUid, clientId, clientUid);
    }

    /**
     * Возвращает все ссылки на интересы клиента с созданием недостающих. Используется в РМП
     */
    public List<InterestLink> getAllExistingInterestLinksWithCreationMissing(List<TargetInterest> targetInterests,
                                                                             int shard, ClientId clientId) {
        return targetInterestService.createMissedInterestsWithRetargetingGoals(targetInterests, clientId, shard);
    }

    public RetargetingUpdateOperation createUpdateOperation(Applicability applicability,
                                                            List<ModelChanges<Retargeting>> modelChanges,
                                                            long operatorUid, ClientId clientId, long clientUid) {
        return createUpdateOperation(applicability, modelChanges, false, null, false, operatorUid, clientId, clientUid);
    }

    public RetargetingUpdateOperation createUpdateOperation(Applicability applicability,
                                                            List<ModelChanges<Retargeting>> modelChanges,
                                                            boolean autoPrices,
                                                            @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                                            boolean partOfComplexOperation,
                                                            long operatorUid, ClientId clientId, long clientUid) {
        int shard = shardFor(clientId);
        LocalDateTime updateBefore = LocalDateTime.now();

        return new RetargetingUpdateOperation(applicability,
                modelChanges,
                updateRetargetingValidationService,
                retargetingRepository,
                retargetingConditionRepository,
                markupConditionRepository,
                adGroupRepository,
                campaignRepository,
                campaignTypedRepository,
                pricePackageRepository,
                pricePackageGeoTree,
                aggregatedStatusesRepository,
                logPriceService,
                mailNotificationEventService, clientService, clientGeoService,
                cpmYndxFrontpageCurrencyService,
                autoPrices, fixedAutoPrices, partOfComplexOperation,
                updateBefore,
                operatorUid, clientId, clientUid, shard);
    }

    public RetargetingDeleteOperation createDeleteOperation(Applicability applicability, List<Long> retargetingIds,
                                                            long operatorUid, ClientId clientId, long clientUid) {
        int shard = shardFor(clientId);

        return new RetargetingDeleteOperation(applicability, retargetingIds,
                retargetingRepository,
                retargetingDeleteService,
                deleteRetargetingValidationService,
                operatorUid, clientId, clientUid, shard);
    }

    /**
     * Подготовка и вызов операции обновления статуса isSuspend для списка ретаргетингов
     *
     * @param retargetingIds список обновляемых ретаргетингов
     * @param clientId       клиент
     * @param operatorUid    оператор
     * @param isSuspend      новое значение поля
     * @return результат валидации операции
     */
    private MassResult<Long> prepareAndExecuteSuspendResumeOperation(List<Long> retargetingIds, ClientId clientId,
                                                                     Long operatorUid, boolean isSuspend) {
        long clientUid = rbacService.getChiefByClientId(clientId);

        Objects.requireNonNull(retargetingIds, "retargetingIds");

        List<ModelChanges<Retargeting>> modelChanges = retargetingIds.stream()
                .distinct()
                .map(id -> ModelChanges.build(id, Retargeting.class, Retargeting.IS_SUSPENDED, isSuspend))
                .collect(toList());

        MassResult<Long> massResult =
                createUpdateOperation(Applicability.PARTIAL, modelChanges, operatorUid, clientId, clientUid)
                        .prepareAndApply();

        @SuppressWarnings("unchecked")
        ValidationResult<List<Retargeting>, Defect> modelsMassValidation =
                (ValidationResult<List<Retargeting>, Defect>) massResult.getValidationResult();
        checkState(modelsMassValidation != null);

        Predicate<DefectInfo> isAllowedErrorForSuspendOperation = d -> d.getPath() == null || d.getPath().equals(path())
                || ALLOWED_ERROR_FIELDS_ON_SUSPEND_CHANGE_VALIDATION.contains(d.getPath().getFieldName());
        checkState(!modelsMassValidation.hasAnyErrors()
                        || modelsMassValidation.flattenErrors().stream().allMatch(isAllowedErrorForSuspendOperation),
                "допустимы только определенные поля при валидации suspend/resume");

        ValidationResult<List<Long>, Defect> flattenUniqueValidationResult =
                ValidationUtils.flattenValidationResult(modelsMassValidation, Retargeting::getId);
        ValidationResult<List<Long>, Defect> validationResult =
                extendValidationResultWithNonUniqueValues(flattenUniqueValidationResult, retargetingIds);

        List<Long> validIds = getValidItems(validationResult);
        List<Long> resultIds = mergeSuccessfulAndInvalidItems(validationResult, validIds, identity());

        if (validationResult.hasErrors() || validIds.isEmpty()) {
            return MassResult.brokenMassAction(validationResult.getValue(), validationResult);
        }


        return MassResult.successfulMassAction(resultIds, validationResult);
    }

    /**
     * Обработка результата валидации: склеивание с возможными дупликатами
     *
     * @param validationResult результат валидации до склейки с уникальными объектами
     * @param retargetingIds   первоначальный список объектов c возможными дублями
     * @return расширенный результат валидации с добавленными варнингами на дупликатах
     */
    private ValidationResult<List<Long>, Defect> extendValidationResultWithNonUniqueValues(
            ValidationResult<List<Long>, Defect> validationResult, List<Long> retargetingIds) {
        Set<Long> uniqueIds = new LinkedHashSet<>();
        Set<Long> duplicateIds = retargetingIds.stream()
                .filter(id -> !uniqueIds.add(id))
                .collect(toSet());

        if (duplicateIds.isEmpty()) {
            return validationResult;
        }

        ValidationResult<List<Long>, Defect> extendedValidationResult =
                new ValidationResult<>(retargetingIds);
        validationResult.getErrors().forEach(extendedValidationResult::addError);
        validationResult.getWarnings().forEach(extendedValidationResult::addWarning);

        Iterator<Long> iteratorUniqueIds = uniqueIds.iterator();
        Map<Long, Integer> uniqueIdToIndexMap = IntStream.range(0, uniqueIds.size())
                .boxed()
                .collect(toMap(i -> iteratorUniqueIds.next(), identity()));

        for (int nonUniqueIndex = 0; nonUniqueIndex < retargetingIds.size(); nonUniqueIndex++) {
            Long value = retargetingIds.get(nonUniqueIndex);

            PathNode nonUniqueCollectionPathNode = index(nonUniqueIndex);
            ValidationResult<Long, Defect> extendedSubVr =
                    extendedValidationResult.getOrCreateSubValidationResult(nonUniqueCollectionPathNode, value);

            if (duplicateIds.contains(value)) {
                extendedSubVr.addWarning(warningDuplicatedRetargetingId());
            }

            int uniqueIndex = uniqueIdToIndexMap.get(value);
            PathNode uniqueCollectionPathNode = index(uniqueIndex);
            @SuppressWarnings("unchecked")
            ValidationResult<Long, Defect> uniqueSubVr = (ValidationResult<Long, Defect>)
                    validationResult.getSubResults().get(uniqueCollectionPathNode);
            if (uniqueSubVr != null) {
                uniqueSubVr.getErrors().forEach(extendedSubVr::addError);
                uniqueSubVr.getWarnings().forEach(extendedSubVr::addWarning);
            }
        }

        return extendedValidationResult;
    }

    /**
     * Включает/выключает у всех объявлений переданные условия ретаргетинга
     *
     * @param switchRetargetings {@link List} id'ов условий ретаргетингов со статусами приостановки
     * @param clientId           id клиента
     */
    public MassResult<SwitchRetargeting> switchRetargetingConditions(List<SwitchRetargeting> switchRetargetings,
                                                                     ClientId clientId,
                                                                     Applicability applicability) {
        ValidationResult<List<SwitchRetargeting>, Defect> massValidation =
                switchRetargetingOnBannersValidationService2.validate(switchRetargetings, clientId);

        if (massValidation.hasErrors()) {
            logger.debug("can not switch retargeting conditions: selection criteria contains errors");
            return MassResult.brokenMassAction(switchRetargetings, massValidation);
        }

        List<SwitchRetargeting> validSwitchRetargetings = getValidItems(massValidation);

        if (isFull(applicability) && validSwitchRetargetings.size() != switchRetargetings.size()) {
            logger.debug("can not switch retargeting conditions: there are invalid ids");
            return MassResult.brokenMassAction(switchRetargetings, massValidation);
        }
        int shard = shardFor(clientId);

        List<Retargeting> retargetings = retargetingRepository.getRetargetingsByRetCondIds(
                shard, mapList(validSwitchRetargetings, SwitchRetargeting::getRetCondId));

        Map<Long, Boolean> retCondIdToSuspended = StreamEx.of(switchRetargetings)
                .mapToEntry(SwitchRetargeting::getRetCondId, SwitchRetargeting::getSuspended)
                .toMap();

        LocalDateTime now = LocalDateTime.now();

        List<AppliedChanges<Retargeting>> changes = StreamEx.of(retargetings)
                .map(retargeting -> retargetingModelChanges(retargeting.getId())
                        .process(retCondIdToSuspended.get(retargeting.getRetargetingConditionId()),
                                Retargeting.IS_SUSPENDED)
                        .process(now, Retargeting.LAST_CHANGE_TIME)
                        .applyTo(retargeting)
                ).toList();
        retargetingRepository.setSuspended(shard, changes);
        List<SwitchRetargeting> resultIds = mergeSuccessfulAndInvalidItems(massValidation, validSwitchRetargetings,
                rId -> rId);

        Set<Long> adGroupIds = new HashSet<>(retargetingRepository
                .getAdGroupIdByRetargetingIds(shard, mapList(retargetings, Retargeting::getId))
                .values());

        adGroupRepository.updateStatusBsSynced(shard, adGroupIds, StatusBsSynced.NO);

        return MassResult.successfulMassAction(resultIds, massValidation);
    }

    public MassResult<Long> resumeRetargetings(List<Long> retargetingIds, ClientId clientId, Long operatorUid) {
        return prepareAndExecuteSuspendResumeOperation(retargetingIds, clientId, operatorUid, false);
    }

    /**
     * Для использования из API. Для остальных мест нужно вызывать {@link #setBids(List, ClientId, Long)}.
     */
    public MassResult<SetBidItem> setBidsApi(List<SetBidItem> setBidItemList, ClientId clientId, Long operatorUid,
                                             Set<AdGroupType> allowedAdGroupTypes) {
        return setBids(setBidItemList, clientId, operatorUid, true, allowedAdGroupTypes);
    }

    public MassResult<SetBidItem> setBids(List<SetBidItem> setBidItemList, ClientId clientId, Long operatorUid) {
        return setBids(setBidItemList, clientId, operatorUid, false, emptySet());
    }

    private MassResult<SetBidItem> setBids(List<SetBidItem> setBidItemList,
                                           ClientId clientId,
                                           Long operatorUid,
                                           boolean fromApi,
                                           Set<AdGroupType> allowedAdGroupTypes) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        long clientUid = rbacService.getChiefByClientId(clientId);

        ValidationResult<List<SetBidItem>, Defect> validation =
                setBidsValidationService.compatiblePreValidation(setBidItemList);
        List<SetBidItem> filteredSetBids = getValidItems(validation);
        if (filteredSetBids.isEmpty()) {
            return MassResult.brokenMassAction(setBidItemList, validation);
        }

        BidSelectionCriteria bidCriteria = filteredSetBids.stream().findAny().orElseGet(SetBidItem::new);
        RequestSetBidType requestType = RequestSetBidType.getByBidSelectionCriteria(bidCriteria);
        Function<BidSelectionCriteria, Long> bidCriteriaSupplier = requestType.getBidCriteriaSupplier();

        List<Retargeting> retargetings = getRetargetingsByBids(filteredSetBids, requestType, shard);

        ValidationResult<List<SetBidItem>, Defect> compatibleModelsValidation =
                setBidsValidationService.compatibleValidation(filteredSetBids, retargetings, operatorUid, shard);

        Map<SetBidItem, ValidationResult<?, Defect>> validationResultBySetBidItem = new HashMap<>();
        for (int i = 0; i < setBidItemList.size(); i++) {
            ValidationResult<?, Defect> vr = validation.getSubResults().get(index(i));
            validationResultBySetBidItem.put(setBidItemList.get(i), vr);
        }

        validation.getErrors().addAll(compatibleModelsValidation.getErrors());
        validation.getWarnings().addAll(compatibleModelsValidation.getWarnings());
        compatibleModelsValidation.getSubResults().forEach((pathNode, subResultFrom) -> {
            SetBidItem setBidItem = (SetBidItem) subResultFrom.getValue();
            ValidationResult<?, Defect> subResultTo = validationResultBySetBidItem.get(setBidItem);
            ValidationUtils.transferIssuesFromValidationToValidation(subResultFrom, subResultTo);
        });

        filteredSetBids = getValidItems(validation);
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(clientId,
                StreamEx.of(retargetings)
                        .map(Retargeting::getAdGroupId)
                        .filter(Objects::nonNull)
                        .toSet());
        retargetings.removeIf(item -> fromApi
                && adGroupTypes.get(item.getAdGroupId()) != null
                && !allowedAdGroupTypes.contains(adGroupTypes.get(item.getAdGroupId())));

        if (filteredSetBids.isEmpty()) {
            return MassResult.brokenMassAction(setBidItemList, validation);
        }

        List<ModelChanges<Retargeting>> modelChanges = buildModelChangesForSetBids(filteredSetBids, retargetings,
                bidCriteriaSupplier);
        MassResult<Long> result =
                createUpdateOperation(Applicability.PARTIAL, modelChanges, operatorUid, clientId, clientUid)
                        .prepareAndApply();

        @SuppressWarnings("unchecked")
        ValidationResult<List<Retargeting>, Defect> retargetingValidationResult =
                (ValidationResult<List<Retargeting>, Defect>) result.getValidationResult();
        checkState(retargetingValidationResult != null);

        //Если отбор был по id ретаргетингов, то нужна переиндексация для дальнейшего правильного мержа результатов
        // валидации
        ValidationResult<List<Retargeting>, Defect> reindexedRetargetingValidationResult =
                requestType == RequestSetBidType.ID
                ? getReindexedRetargetingValidationResult(retargetingValidationResult, validation)
                : retargetingValidationResult;

        ValidationUtils.transferIssuesFromValidationToValidationWithNewValue(reindexedRetargetingValidationResult,
                validation);

        filteredSetBids = getValidItems(validation);
        if (validation.hasErrors() || filteredSetBids.isEmpty()) {
            return MassResult.brokenMassAction(setBidItemList, validation);
        }

        int validItemsSizeBySetBids = ((int) filteredSetBids.stream()
                .map(bidCriteriaSupplier)
                .distinct()
                .count());

        return MassResult.successfulMassAction(validation.getValue(), validation)
                .withCounts(result.getSuccessfulCount(), setBidItemList.size() - validItemsSizeBySetBids);
    }

    //Установка в результат валидации исходных индексов для правильного дальнейшего мержа
    private ValidationResult<List<Retargeting>, Defect> getReindexedRetargetingValidationResult(
            ValidationResult<List<Retargeting>, Defect> retargetingValidationResult,
            ValidationResult<List<SetBidItem>, Defect> validation) {
        Map<Long, Integer> showCondIdToIndexId = validation.getSubResults()
                .entrySet()
                .stream()
                .collect(toMap(e -> ((SetBidItem) e.getValue().getValue()).getId(),
                        e -> ((PathNode.Index) e.getKey()).getIndex()));

        Map<PathNode, ValidationResult<?, Defect>> reindexedSubresults =
                listToMap(retargetingValidationResult.getSubResults().values(),
                        vr -> createIndex(vr, showCondIdToIndexId), identity());

        return new ValidationResult<>(retargetingValidationResult.getValue(),
                retargetingValidationResult.getErrors(),
                retargetingValidationResult.getWarnings(),
                reindexedSubresults
        );
    }

    private PathNode.Index createIndex(ValidationResult<?, Defect> vr, Map<Long, Integer> showCondIdToIndexId) {
        Long id = ((Retargeting) vr.getValue()).getId();
        return new PathNode.Index(showCondIdToIndexId.get(id));
    }

    /**
     * @param filteredSetBidItems уже отфильтрованный список SetBidItem по валидному идентификатору запроса,
     *                            одинаковому для всех объектов
     * @return список изменений Retargeting для RetargetingUpdateOperation
     */
    private List<ModelChanges<Retargeting>> buildModelChangesForSetBids(List<SetBidItem> filteredSetBidItems,
                                                                        List<Retargeting> existingRetargetings,
                                                                        Function<BidSelectionCriteria, Long> bidCriteriaSupplier) {
        Multimap<Long, Long> retargetingIdsByBidCriteria = existingRetargetings.stream()
                .collect(toMultimap(bidCriteriaSupplier, Retargeting::getId));

        BiFunction<Long, SetBidItem, ModelChanges<Retargeting>> modelChangesCreator =
                (id, setBid) -> retargetingModelChanges(id)
                        .processNotNull(setBid.getAutobudgetPriority(), Retargeting.AUTOBUDGET_PRIORITY)
                        .processNotNull(setBid.getPriceContext(), Retargeting.PRICE_CONTEXT);

        return filteredSetBidItems.stream()
                .map(setBid -> {
                    Long bidSelectionId = bidCriteriaSupplier.apply(setBid);
                    Collection<Long> retargetingIds = retargetingIdsByBidCriteria.get(bidSelectionId);

                    if (retargetingIds.isEmpty()) {
                        // add null instead of bad not found retargeting id by bidSelectionCriteria
                        retargetingIds = singletonList(null);
                    }

                    return mapList(retargetingIds, id -> modelChangesCreator.apply(id, setBid));
                })
                .flatMap(List::stream)
                .collect(toList());
    }

    private static ModelChanges<Retargeting> retargetingModelChanges(long id) {
        return new ModelChanges<>(id, Retargeting.class);
    }

    private List<Retargeting> getRetargetingsByBids(List<SetBidItem> filteredSetBidItems,
                                                    RequestSetBidType requestType, int shard) {
        List<Long> ids = mapList(filteredSetBidItems, requestType.getBidCriteriaSupplier());

        if (requestType == RequestSetBidType.ID) {
            return retargetingRepository.getRetargetingsByIds(shard, ids, limited(filteredSetBidItems.size()));
        } else if (requestType == RequestSetBidType.ADGROUP_ID) {
            return retargetingRepository.getRetargetingsByAdGroups(shard, ids);
        } else if (requestType == RequestSetBidType.CAMPAIGN_ID) {
            return retargetingRepository.getRetargetingsByCampaigns(shard, ids);
        }

        return emptyList();
    }

    private int shardFor(ClientId clientId) {
        return shardHelper.getShardByClientIdStrictly(clientId);
    }
}
