package ru.yandex.direct.core.entity.retargeting.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupName;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.mailnotification.model.RetargetingEvent;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.markupcondition.repository.MarkupConditionRepository;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageGeoTree;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.core.entity.retargeting.service.validation2.UpdateRetargetingValidationService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.mailnotification.model.RetargetingEvent.changeRetargetingConditionEvent;
import static ru.yandex.direct.core.entity.mailnotification.model.RetargetingEvent.changeRetargetingPriceContextEvent;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.fillAbsentAutobudgetPriority;
import static ru.yandex.direct.core.entity.retargeting.service.RetargetingUtils.getPackagePriceFunctionByCampaignId;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class RetargetingUpdateOperation extends SimpleAbstractUpdateOperation<Retargeting, Long> {

    private static final Logger logger = LoggerFactory.getLogger(RetargetingUpdateOperation.class);

    private static final Set<ModelProperty> IGNORE_CHANGES_TO_ADGROUP_UPDATE_STATUS_BS_SYNCED =
            new HashSet<>(asList(
                    Retargeting.STATUS_BS_SYNCED,
                    Retargeting.LAST_CHANGE_TIME,
                    Retargeting.PRICE_CONTEXT,
                    Retargeting.AUTOBUDGET_PRIORITY));

    private final UpdateRetargetingValidationService updateRetargetingValidationService;
    private final RetargetingRepository retargetingRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final MarkupConditionRepository markupConditionRepository;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final PricePackageRepository pricePackageRepository;
    private final PricePackageGeoTree pricePackageGeoTree;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final LogPriceService logPriceService;
    private final MailNotificationEventService mailNotificationEventService;
    private final ClientService clientService;
    private final ClientGeoService clientGeoService;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;

    private final boolean partOfComplexOperation;
    private final boolean autoPrices;
    private final LocalDateTime updateBefore;
    @Nullable
    private final ShowConditionFixedAutoPrices fixedAutoPrices;

    private Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> existentFrontpageAdGroupsPriceRestrictions;
    private Map<Long, AdGroupSimple> existentAdGroupsInfo;
    private Map<Long, AdGroup> preparedAdGroupModels;
    private Map<Long, CampaignType> campaignsType;
    private Map<Long, BigDecimal> packagePriceByAdGroupId;

    private long operatorUid;
    private ClientId clientId;
    private long clientUid;
    private int shard;

    private int actualChangedItemsSize = 0;

    public RetargetingUpdateOperation(Applicability applicability,
                                      List<ModelChanges<Retargeting>> modelChanges,
                                      UpdateRetargetingValidationService updateRetargetingValidationService,
                                      RetargetingRepository retargetingRepository,
                                      RetargetingConditionRepository retargetingConditionRepository,
                                      MarkupConditionRepository markupConditionRepository,
                                      AdGroupRepository adGroupRepository,
                                      CampaignRepository campaignRepository,
                                      CampaignTypedRepository campaignTypedRepository,
                                      PricePackageRepository pricePackageRepository,
                                      PricePackageGeoTree pricePackageGeoTree,
                                      AggregatedStatusesRepository aggregatedStatusesRepository,
                                      LogPriceService logPriceService,
                                      MailNotificationEventService mailNotificationEventService,
                                      ClientService clientService,
                                      ClientGeoService clientGeoService,
                                      CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
                                      boolean autoPrices, @Nullable ShowConditionFixedAutoPrices fixedAutoPrices,
                                      boolean partOfComplexOperation,
                                      LocalDateTime updateBefore,
                                      long operatorUid, ClientId clientId, long clientUid, int shard) {
        super(applicability, modelChanges, id -> new Retargeting().withId(id));

        this.updateRetargetingValidationService = updateRetargetingValidationService;
        this.retargetingRepository = retargetingRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.markupConditionRepository = markupConditionRepository;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.pricePackageRepository = pricePackageRepository;
        this.pricePackageGeoTree = pricePackageGeoTree;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.logPriceService = logPriceService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.clientService = clientService;
        this.clientGeoService = clientGeoService;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;

        checkArgument(!autoPrices || fixedAutoPrices != null, "fixed price should be not null in autoPrice mode");
        this.autoPrices = autoPrices;
        this.fixedAutoPrices = fixedAutoPrices;
        this.partOfComplexOperation = partOfComplexOperation;
        this.updateBefore = updateBefore;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.shard = shard;
    }

    @Override
    protected ValidationResult<List<ModelChanges<Retargeting>>, Defect> validateModelChanges(
            List<ModelChanges<Retargeting>> modelChanges) {
        return updateRetargetingValidationService.preValidate(modelChanges, clientId, operatorUid, shard);
    }

    @Override
    protected Collection<Retargeting> getModels(Collection<Long> ids) {
        List<Retargeting> models = retargetingRepository.getRetargetingsByIds(shard, new ArrayList<>(ids), maxLimited());
        fillAbsentAutobudgetPriority(models);
        return models;
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<Retargeting> changesAppliedStep) {
        initCampaignsTypeAndPricePackagesPrice(changesAppliedStep);
        fillPriceContext(changesAppliedStep.getAppliedChangesForValidModelChanges());
        initExistentAdGroupsInfoAndFrontpagePriceRestrictions(changesAppliedStep);
    }

    private void initCampaignsTypeAndPricePackagesPrice(ChangesAppliedStep<Retargeting> changesAppliedStep) {
        Set<Long> campaignIds = changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                .map(AppliedChanges::getModel)
                .map(Retargeting::getCampaignId)
                .collect(toSet());
        campaignsType = campaignRepository.getCampaignsTypeMap(shard, campaignIds);

        var packagePriceFunctionByCampaignId =
                getPackagePriceFunctionByCampaignId(shard, campaignTypedRepository,
                        pricePackageRepository, markupConditionRepository, campaignsType);

        Set<Long> priceAdGroupIds = changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                .map(AppliedChanges::getModel)
                .filter(r -> packagePriceFunctionByCampaignId.containsKey(r.getCampaignId()))
                .map(Retargeting::getAdGroupId)
                .collect(toSet());

        Map<Long, List<RetargetingCondition>> retConditionsByAdGroupIds =
                retargetingConditionRepository.getRetConditionsByAdGroupIds(shard, clientId, priceAdGroupIds);

        Map<Long, List<Goal>> retargetingGoalsByAdGroupId = EntryStream.of(retConditionsByAdGroupIds)
                .mapValues(conditions -> conditions.stream()
                        .map(RetargetingCondition::collectGoals)
                        .flatMap(Collection::stream)
                        .distinct()
                        .collect(toList()))
                .toMap();

        GeoTree geoTree = pricePackageGeoTree.getGeoTree();
        packagePriceByAdGroupId = changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                .map(AppliedChanges::getModel)
                .filter(r -> priceAdGroupIds.contains(r.getAdGroupId()))
                .collect(toMap(Retargeting::getAdGroupId,
                        r -> {
                            AdGroup adGroup = preparedAdGroupModels.get(r.getAdGroupId());
                            List<Long> geo = clientGeoService.convertForSave(adGroup.getGeo(), geoTree);
                            List<Long> goalIds = mapList(retargetingGoalsByAdGroupId.get(r.getAdGroupId()), ModelWithId::getId);
                            List<Long> projectParamConditions = adGroup.getProjectParamConditions();
                            var priceFunction = packagePriceFunctionByCampaignId.get(r.getCampaignId());
                            BigDecimal price = priceFunction.apply(geo, goalIds, projectParamConditions).getPrice();
                            return price;
                        }));
    }

    private void fillPriceContext(Collection<AppliedChanges<Retargeting>> appliedChanges) {
        appliedChanges.forEach(ac -> {
            if (ac.changed(Retargeting.PRICE_CONTEXT)) {
                return;
            }
            CampaignType campaignType = campaignsType.get(ac.getModel().getCampaignId());
            Long adGroupId = ac.getModel().getAdGroupId();
            if (campaignType == CampaignType.CPM_PRICE) {
                BigDecimal pricePackagePrice = packagePriceByAdGroupId.get(adGroupId);
                ac.modify(Retargeting.PRICE_CONTEXT, pricePackagePrice);
            } else {
                if (autoPrices) {
                    setFixedAutoPrice(ac);
                }
            }
        });
    }

    private void initExistentAdGroupsInfoAndFrontpagePriceRestrictions(
            ChangesAppliedStep<Retargeting> changesAppliedStep) {
        if (existentAdGroupsInfo == null) {
            Set<Long> adGroupIds = StreamEx.of(changesAppliedStep.getAppliedChangesForValidModelChanges())
                    .map(AppliedChanges::getModel)
                    .map(Retargeting::getAdGroupId)
                    .toSet();
            existentAdGroupsInfo = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);
        }
        if (existentFrontpageAdGroupsPriceRestrictions == null) {
            existentFrontpageAdGroupsPriceRestrictions = cpmYndxFrontpageCurrencyService
                    .getAdGroupIdsToPriceDataMapByAdGroups(existentAdGroupsInfo.values(), shard, clientId);
        }
    }

    @Override
    protected ValidationResult<List<Retargeting>, Defect> validateAppliedChanges(
            ValidationResult<List<Retargeting>, Defect> validationResult) {
        boolean skipInterconnectionsWithAdsValidation = partOfComplexOperation;
        existentFrontpageAdGroupsPriceRestrictions = nvl(existentFrontpageAdGroupsPriceRestrictions, emptyMap());
        return updateRetargetingValidationService
                .validate(validationResult, existentAdGroupsInfo, existentFrontpageAdGroupsPriceRestrictions,
                        clientId, shard, skipInterconnectionsWithAdsValidation, packagePriceByAdGroupId);
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<Retargeting> appliedChangesValidatedStep) {
        appliedChangesValidatedStep.getValidAppliedChanges()
                .forEach(this::forCpmPriceCampaignOverwritePriceContextByDbValue);
    }

    private void forCpmPriceCampaignOverwritePriceContextByDbValue(AppliedChanges<Retargeting> appliedChanges) {
        Long campaignId = appliedChanges.getModel().getCampaignId();
        Long adGroupId = appliedChanges.getModel().getAdGroupId();
        CampaignType campaignType = campaignsType.get(campaignId);
        if (campaignType != CampaignType.CPM_PRICE) {
            return;
        }
        BigDecimal pricePackagePrice = packagePriceByAdGroupId.get(adGroupId);
        checkState(pricePackagePrice != null,
                String.format("pricePackagePrice not found for CPM_PRICE campaign with id = %d", campaignId));
        if (appliedChanges.changed(Retargeting.PRICE_CONTEXT)) {
            logger.error(String.format("received modelChange for priceContext for price campaign with id %s, " +
                    "but this field must not be changed", campaignId));
        }
        BigDecimal oldPriceContext = appliedChanges.getOldValue(Retargeting.PRICE_CONTEXT);
        if (oldPriceContext == null || pricePackagePrice.compareTo(oldPriceContext) != 0) {
            logger.error(String.format("old/database values for priceContext for price campaign with id %s diverged." +
                    " old value = %s, database value = %s", campaignId, oldPriceContext, pricePackagePrice));
        }
        // всегда перезаписываем PRICE_CONTEXT значением из пакета, на всякий случай
        // вообще мы и так провалидировали, что PRICE_CONTEXT правильный, это в каком-то смысле "двойная защита"
        appliedChanges.modify(Retargeting.PRICE_CONTEXT, pricePackagePrice);
    }

    @Override
    protected void beforeExecution(ExecutionStep<Retargeting> executionStep) {
        Collection<AppliedChanges<Retargeting>> appliedChangesCollection =
                filterList(executionStep.getAppliedChangesForExecution(), AppliedChanges::hasActuallyChangedProps);

        LocalDateTime now = LocalDateTime.now();
        appliedChangesCollection.stream()
                .filter(AppliedChanges::hasActuallyChangedProps)
                .forEach(ac -> {
                    if (!ac.getActuallyChangedProps().stream().allMatch(c -> c.equals(Retargeting.IS_SUSPENDED))) {
                        ac.modify(Retargeting.STATUS_BS_SYNCED, StatusBsSynced.NO);
                    }
                    ac.modify(Retargeting.LAST_CHANGE_TIME, now);
                });

        // ставка пересчитывается, поскольку в ComplexCpmAdGroupUpdateOperation::afterAdGroupsApply
        // могло поменяться условие ретаргетинга, потенциально влияющее на наценку в прайсовых кампаниях
        initCampaignsTypeAndPricePackagesPrice(executionStep);
        fillPriceContext(executionStep.getAppliedChangesForValidModelChanges());
        executionStep.getValidAppliedChanges().forEach(this::forCpmPriceCampaignOverwritePriceContextByDbValue);
    }

    public void setExistentAdGroupsInfoAndFrontpagePriceRestrictions(
            Map<Long, AdGroupSimple> existentAdGroupsInfo,
            Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> existentFrontpageAdGroupsPriceRestrictions,
            Map<Long, AdGroup> preparedAdGroupModels) {
        this.existentAdGroupsInfo = existentAdGroupsInfo;
        this.preparedAdGroupModels = preparedAdGroupModels;
        this.existentFrontpageAdGroupsPriceRestrictions = existentFrontpageAdGroupsPriceRestrictions;
    }

    /**
     * Если у ретаргетинга не выставлена ставка, пытаемся выставить ее на основе информации из {@link #fixedAutoPrices}
     */
    private void setFixedAutoPrice(AppliedChanges<Retargeting> ac) {
        checkNotNull(fixedAutoPrices);
        Long adGroupId = ac.getModel().getAdGroupId();

        checkState(!ac.changed(Retargeting.PRICE_CONTEXT));

        BigDecimal fixedPrice = null;
        if (fixedAutoPrices.hasGlobalFixedPrice()) {
            fixedPrice = fixedAutoPrices.getGlobalFixedPrice();
        } else if (fixedAutoPrices.hasAdGroupFixedPrice(adGroupId)) {
            fixedPrice = fixedAutoPrices.getAdGroupFixedPrice(adGroupId);
        }

        if (fixedPrice != null) {
            ac.modify(Retargeting.PRICE_CONTEXT, fixedPrice);
        }
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<Retargeting>> appliedChanges) {
        retargetingRepository.updateRetargetings(shard, appliedChanges);

        actualChangedItemsSize = (int) appliedChanges.stream()
                .filter(AppliedChanges::hasActuallyChangedProps)
                .count();

        return mapList(appliedChanges, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<Retargeting> executionStep) {
        Collection<AppliedChanges<Retargeting>> appliedChanges = executionStep.getAppliedChangesForExecution();

        updateAdGroups(appliedChanges);
        updateAggregatedStatuses(appliedChanges);

        logPriceChanges(appliedChanges);
        sendMailNotifications(appliedChanges);
    }

    @Override
    protected void postProcessResult(MassResult<Long> result) {
        /*
         * ручное выставление кол-ва успешно изменённых элементов,
         * для правильного подсчёта баллов в API
         */
        result.withCounts(actualChangedItemsSize, result.getUnsuccessfulObjectsCount());
    }

    /**
     * Актуализация AdGroup при изменении Retargeting'ов.
     * <p>
     * Обновление adGroup.lastChange для всех затронутых AdGroup.
     * Обновление adGroup.statusBsSync = 'No' для тех, кто не в черновиках (adGroup.statusModerate != 'New').
     */
    private void updateAdGroups(Collection<AppliedChanges<Retargeting>> appliedChanges) {
        List<Long> adGroupIdsToChange = StreamEx.of(appliedChanges)
                .remove(ac ->
                        IGNORE_CHANGES_TO_ADGROUP_UPDATE_STATUS_BS_SYNCED.containsAll(ac.getActuallyChangedProps()))
                .map(ac -> ac.getModel().getAdGroupId())
                .toList();

        adGroupRepository.updateStatusBsSyncedExceptNew(shard, adGroupIdsToChange, StatusBsSynced.NO);
        adGroupRepository.updateLastChange(shard, adGroupIdsToChange);
    }

    /**
     * Обновление aggr_statuses_retargetings.is_obsolete
     */
    private void updateAggregatedStatuses(Collection<AppliedChanges<Retargeting>> appliedChanges) {
        List<Long> retargetingIds = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(Retargeting.IS_SUSPENDED))
                .map(ac -> ac.getModel().getId())
                .toList();
        aggregatedStatusesRepository.markRetargetingStatusesAsObsolete(shard, updateBefore, retargetingIds);
    }

    /**
     * Логгирование изменения цен
     */
    private void logPriceChanges(Collection<AppliedChanges<Retargeting>> appliedChanges) {
        Currency currency = clientService.getWorkCurrency(clientId);

        Function<Retargeting, LogPriceData> retargetingToLog = r -> new LogPriceData(
                r.getCampaignId(),
                r.getAdGroupId(),
                r.getId(),
                nvl(r.getPriceContext(), BigDecimal.ZERO).doubleValue(),
                null,
                currency.getCode(),
                LogPriceData.OperationType.RET_UPDATE);

        List<LogPriceData> logPriceList = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(Retargeting.PRICE_CONTEXT))
                .map(AppliedChanges::getModel)
                .map(retargetingToLog)
                .toList();

        logPriceService.logPrice(logPriceList, operatorUid);
    }

    private void sendMailNotifications(Collection<AppliedChanges<Retargeting>> appliedChangesForExecution) {
        List<AppliedChanges<Retargeting>> affectedChanges = filterList(appliedChangesForExecution,
                ac -> ac.changed(Retargeting.PRICE_CONTEXT) || ac.changed(Retargeting.RETARGETING_CONDITION_ID));

        Set<Long> affectedAdGroupIds = listToSet(affectedChanges, ac -> ac.getModel().getAdGroupId());
        Map<Long, AdGroupName> adGroupIdToAdGroup =
                adGroupRepository.getAdGroupNames(shard, clientId, affectedAdGroupIds);

        checkState(affectedAdGroupIds.size() == adGroupIdToAdGroup.keySet().size(),
                "All ad groups of updated retargetings must exists.");

        List<RetargetingEvent> events = new ArrayList<>();

        affectedChanges.forEach(change -> {
            Long adGroupId = change.getModel().getAdGroupId();
            long campaignId = adGroupIdToAdGroup.get(adGroupId).getCampaignId();
            String adGroupName = adGroupIdToAdGroup.get(adGroupId).getName();

            if (change.changed(Retargeting.PRICE_CONTEXT)
                    && change.getOldValue(Retargeting.PRICE_CONTEXT) != null &&
                    change.getNewValue(Retargeting.PRICE_CONTEXT) != null) {
                BigDecimal oldValue = change.getOldValue(Retargeting.PRICE_CONTEXT);
                BigDecimal newValue = change.getNewValue(Retargeting.PRICE_CONTEXT);

                @SuppressWarnings("ConstantConditions")
                RetargetingEvent event = changeRetargetingPriceContextEvent(operatorUid, clientUid, campaignId,
                        adGroupId, adGroupName, oldValue, newValue);

                events.add(event);
            }

            if (change.changed(Retargeting.RETARGETING_CONDITION_ID)
                    && change.getOldValue(Retargeting.RETARGETING_CONDITION_ID) != null
                    && change.getNewValue(Retargeting.RETARGETING_CONDITION_ID) != null) {
                Long oldRetargetingCondtionId = change.getOldValue(Retargeting.RETARGETING_CONDITION_ID);
                Long newRetargetingConditionId = change.getNewValue(Retargeting.RETARGETING_CONDITION_ID);

                @SuppressWarnings("ConstantConditions")
                RetargetingEvent event =
                        changeRetargetingConditionEvent(operatorUid, clientUid, campaignId, adGroupId, adGroupName,
                                oldRetargetingCondtionId, newRetargetingConditionId);

                events.add(event);
            }

        });

        mailNotificationEventService.queueEvents(operatorUid, clientId, events);
    }
}
