package ru.yandex.direct.core.entity.retargeting.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.pricerecalculation.PriceCalculator;
import ru.yandex.direct.core.entity.markupcondition.repository.MarkupConditionRepository;
import ru.yandex.direct.core.entity.pricepackage.model.MarkupCondition;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingMarkup;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.retargeting.Constants;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCampaignInfo;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingDeleteRequest;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingDeleteRequestImpl;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.model.ModelWithId;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.springframework.util.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.CPM_PRICE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.LAL_SEGMENT;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class RetargetingUtils {

    public static List<Retargeting> convertTargetInterestsToRetargetings(List<TargetInterest> targetInterests,
                                                                         List<InterestLink> interestLinks) {
        Map<Long, Long> retCondIdsByInterestIds = StreamEx.of(interestLinks)
                .mapToEntry(InterestLink::getInterestId, InterestLink::getRetargetingConditionId)
                .toMap();

        return mapList(targetInterests, ti -> {
            Long interestId = ti.getInterestId();
            Long retCondId = interestId != null ? retCondIdsByInterestIds.get(interestId) : null;
            return targetInterestToRetargeting(ti, retCondId);
        });
    }

    private static Retargeting targetInterestToRetargeting(TargetInterest targetInterest, Long retargetingConditionId) {
        Retargeting retargeting = new Retargeting()
                .withId(targetInterest.getId())
                .withAdGroupId(targetInterest.getAdGroupId())
                .withPriceContext(targetInterest.getPriceContext())
                .withAutobudgetPriority(targetInterest.getAutobudgetPriority())
                .withIsSuspended(targetInterest.getIsSuspended())
                .withCampaignId(targetInterest.getCampaignId())
                .withLastChangeTime(targetInterest.getLastChangeTime())
                .withStatusBsSynced(targetInterest.getStatusBsSynced());

        if (targetInterest.getInterestId() != null) {
            checkArgument(retargetingConditionId != null,
                    "actual retargetingConditionId is not defined for TargetInterest");
            retargeting.setRetargetingConditionId(retargetingConditionId);
        } else {
            checkArgument(targetInterest.getRetargetingConditionId() != null,
                    "Retargeting.retargetingConditionId is not defined");
            retargeting.setRetargetingConditionId(targetInterest.getRetargetingConditionId());
        }
        return retargeting;
    }


    public static List<TargetInterest> convertRetargetingsToTargetInterests(List<Retargeting> retargetings,
                                                                            List<InterestLink> interestLinks) {
        Map<Long, Long> interestIdsByRetCondIds = StreamEx.of(interestLinks)
                .mapToEntry(InterestLink::getRetargetingConditionId, InterestLink::getInterestId)
                .toMap();

        return mapList(retargetings, retargeting -> {
            Long retCondId = retargeting.getRetargetingConditionId();
            Long interestId = interestIdsByRetCondIds.get(retCondId);
            return retargetingToTargetInterest(retargeting, interestId);
        });
    }

    /**
     * Конвертация ретаргетинга в более общий класс - TargetInterest,
     * который по сути может быть ретаргетингом, а может быть таргетингом по интересам.
     * Если {@code TargetInterest#getInterestId != null}, то это таргетинг по интересам,
     * а если {@code TargetInterest#getRetargetingConditionId != null}, то это ретаргетинг.
     */
    private static TargetInterest retargetingToTargetInterest(Retargeting retargeting, Long interestId) {
        TargetInterest targetInterest = new TargetInterest()
                .withId(retargeting.getId())
                .withAdGroupId(retargeting.getAdGroupId())
                .withPriceContext(retargeting.getPriceContext())
                .withAutobudgetPriority(retargeting.getAutobudgetPriority())
                .withIsSuspended(retargeting.getIsSuspended())
                .withCampaignId(retargeting.getCampaignId())
                .withLastChangeTime(retargeting.getLastChangeTime())
                .withStatusBsSynced(retargeting.getStatusBsSynced());

        if (interestId != null) {
            targetInterest.setInterestId(interestId);
        } else {
            targetInterest.setRetargetingConditionId(retargeting.getRetargetingConditionId());
        }

        return targetInterest;
    }

    public static void fillAbsentAutobudgetPriority(List<Retargeting> retargetings) {
        retargetings.forEach(retargeting -> retargeting
                .withAutobudgetPriority(
                        nvl(retargeting.getAutobudgetPriority(), Constants.DEFAULT_AUTOBUDGET_PRIORITY)));
    }

    /**
     * Стремность типа выходного параметра обусловлена
     * необходимостью вычислять итоговый CPM для еще не существующих групп (групп без ID),
     * поскольку в add-операции добыча таргетингов для несуществующих групп происходит по индексу в процессе итерации по списку моделей
     */
    public static Map<Long, PriceCalculator> getPackagePriceFunctionByCampaignId(
            int shard,
            CampaignTypedRepository campaignTypedRepository,
            PricePackageRepository pricePackageRepository,
            MarkupConditionRepository markupConditionRepository,
            Map<Long, CampaignType> campaignsType) {

        List<Long> priceCampaignIds = EntryStream.of(campaignsType)
                .filterValues(campaignType -> campaignType == CPM_PRICE)
                .keys()
                .toList();
        var priceCampaigns = campaignTypedRepository.getTypedCampaigns(shard, priceCampaignIds);
        Map<Long, CpmPriceCampaign> cpmPriceCampaigns = priceCampaigns.stream()
                .map(CpmPriceCampaign.class::cast)
                .collect(toMap(ModelWithId::getId, identity()));
        Map<Long, Long> pricePackageIdByCid = cpmPriceCampaigns.values().stream()
                .collect(toMap(CpmPriceCampaign::getId, CpmPriceCampaign::getPricePackageId));
        Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(pricePackageIdByCid.values());
        Set<Long> markupConditionIds = pricePackages.values().stream()
                .map(PricePackage::getTargetingMarkups)
                .filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .map(TargetingMarkup::getConditionId)
                .collect(toSet());
        Map<Long, MarkupCondition> markupConditionMap = markupConditionRepository.getAllMarkupConditions().stream()
                .filter(mc -> markupConditionIds.contains(mc.getId()) && !mc.getIsArchived())
                .collect(toMap(MarkupCondition::getId, identity()));
        return EntryStream.of(cpmPriceCampaigns)
                .mapValues(c -> new PriceCalculator(c, pricePackages, markupConditionMap))
                .toMap();
    }

    public static RetargetingDeleteRequest convertToRetargetingDeleteRequest(
            RetargetingCampaignInfo retargetingCampaignInfo) {
        return retargetingCampaignInfo;
    }

    public static RetargetingDeleteRequest convertToRetargetingDeleteRequest(Retargeting retargeting) {
        return new RetargetingDeleteRequestImpl()
                .withRetargetingId(checkNotNull(retargeting.getId()))
                .withRetargetingConditionId(checkNotNull(retargeting.getRetargetingConditionId()))
                .withAdGroupId(checkNotNull(retargeting.getAdGroupId()))
                .withCampaignId(checkNotNull(retargeting.getCampaignId()));
    }

    public static List<Long> extractLalSegmentIds(List<RetargetingCondition> retargetingConditions) {
        return StreamEx.of(retargetingConditions)
                .map(RetargetingCondition::collectGoals)
                .flatMap(List::stream)
                .filter(goal -> goal.getType() == LAL_SEGMENT)
                .map(GoalBase::getId)
                .toList();
    }

    public static TargetInterest getTargetInterestWithInterestsCondition(Set<Long> interestsRetCondIds,
                                                                         Long adGroupId,
                                                                         @Nullable List<TargetInterest> targetInterests) {
        List<TargetInterest> targetInterestsWithInterestsCondition = ifNotNull(targetInterests,
                ti -> filterList(ti, ret -> interestsRetCondIds.contains(ret.getRetargetingConditionId())));

        if (isEmpty(targetInterestsWithInterestsCondition)) {
            return new TargetInterest();
        }

        checkState(
                targetInterestsWithInterestsCondition.size() == 1,
                "There should be exactly one target for group %s, found %s",
                adGroupId, targetInterestsWithInterestsCondition.size()
        );

        return targetInterestsWithInterestsCondition.get(0);
    }
}
