package ru.yandex.direct.core.entity.retargeting.service.common;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.container.AllowedRetargetingComponentsInUserProfile;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.metrika.client.MetrikaClient;
import ru.yandex.direct.metrika.client.MetrikaHelper;
import ru.yandex.direct.metrika.client.model.request.GetGoalsRequest;
import ru.yandex.direct.rbac.RbacService;

import static com.google.common.collect.ImmutableSet.toImmutableSet;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType.short_term;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.AB_SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.AUDIENCE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.AUDIO_GENRES;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.BEHAVIORS;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.CDP_SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.CONTENT_CATEGORY;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.CONTENT_GENRE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.ECOMMERCE;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.FAMILY;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.GOAL;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.INTERESTS;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.INTERNAL;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.LAL_SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.SOCIAL_DEMO;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
public class GoalUtilsService {

    private static final Logger logger = LoggerFactory.getLogger(GoalUtilsService.class);

    private static final Map<Long, Set<Long>> MUTUALLY_EXCLUSIVE_GOALS_MAP = new HashMap<>();

    private static final long CRYPTA_GOAL_TV_NOT_SO_MUCH = 2_499_000_201L;

    private static final long CRYPTA_GOAL_TV_TOO_MUCH = 2_499_000_202L;

    private static final long CRYPTA_GOAL_MARRIED = 2_499_000_101L;

    private static final long CRYPTA_GOAL_NOT_MARRIED = 2_499_000_102L;

    static {
        addMutuallyExclusiveGoal();
        // DIRECT-80726: Телевидение/ТВ смотрение
        addMutuallyExclusiveGoal(CRYPTA_GOAL_TV_NOT_SO_MUCH, CRYPTA_GOAL_TV_TOO_MUCH);
        // Состоят в браке или нет
        addMutuallyExclusiveGoal(CRYPTA_GOAL_MARRIED, CRYPTA_GOAL_NOT_MARRIED);
    }

    private final MetrikaHelper metrikaHelper;
    private final RbacService rbacService;

    @Autowired
    public GoalUtilsService(
            MetrikaHelper metrikaHelper,
            RbacService rbacService) {
        this.metrikaHelper = metrikaHelper;
        this.rbacService = rbacService;
    }

    /**
     * Добавляет набор взаимоисключающих целей в {@link GoalUtilsService#MUTUALLY_EXCLUSIVE_GOALS_MAP}.
     * <p>
     * Наборы добавляются для всех ключей, то есть для списка A, B, C будут добавлены записи A:[B,C], B:[A,C] и C:[A,B]
     */
    private static void addMutuallyExclusiveGoal(Long... goals) {
        for (Long goal : goals) {
            Set<Long> exclusiveGoals = StreamEx.of(goals).filter(g -> !g.equals(goal)).collect(toImmutableSet());
            MUTUALLY_EXCLUSIVE_GOALS_MAP.put(goal, exclusiveGoals);
        }
    }

    /**
     * Меняет название цели типа "ECOMMERCE"
     * Сделано аналогично перлу, где название цели типа "ECOMMERCE" выводится в виде
     * "eCommerce: Покупка (счетчик № 12345)", когда в базе хранится "12345"
     * Номер счетчика берется из исходного названия
     */
    public static Goal changeEcommerceGoalName(Goal goal) {
        return (Goal) goal.withName(getEcommerceGoalName(goal.getType() == ECOMMERCE, goal));
    }

    public static String getEcommerceGoalName(boolean isEcommerce, Goal goal) {
        return isEcommerce ? String.format("eCommerce: Покупка (счетчик № %s)", goal.getName())
                : goal.getName();
    }

    /**
     * Собирает имя цели как "Название родительской цели; название самой цели" и собирает название для ecommerce-цели
     * (у таких родительских целей нет)
     */
    public static String composeGoalName(Goal goal, @Nullable Goal parentGoal) {
        return (parentGoal == null
                ? ""
                : parentGoal.getName() + "; ") + getEcommerceGoalName(goal.getType() == ECOMMERCE, goal);
    }

    /**
     * Собирает название для ecommerce-цели, если надо
     */
    public static String composeGoalSimpleName(Goal goal) {
        return getEcommerceGoalName(goal.getType() == ECOMMERCE, goal);
    }

    /**
     * Возвращает Map взаимоисключающих целей
     */
    public Map<Long, Set<Long>> getMutuallyExclusiveGoalsMap() {
        return Collections.unmodifiableMap(MUTUALLY_EXCLUSIVE_GOALS_MAP);
    }

    public static Map<Long, Set<Long>> getMutuallyExclusiveGoals() {
        return MUTUALLY_EXCLUSIVE_GOALS_MAP;
    }

    public Set<Long> getAvailableMetrikaGoalIds(ClientId clientId, Collection<Long> goalIds) {
        return listToSet(getAvailableMetrikaGoals(clientId, goalIds), g -> g.getId());
    }

    public List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> getAvailableMetrikaGoals(
            ClientId clientId, Collection<Long> goalIds) {
        var uids = rbacService.getClientRepresentativesUids(clientId);
        var goalTypeToConditions = StreamEx.of(goalIds)
                .mapToEntry(Goal::computeType, Function.identity())
                .nonNullKeys()
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .mapToValue((type, ids) -> getConditions(metrikaHelper.getMetrikaClient(), uids, type, ids))
                .toMap();

        Set<ru.yandex.direct.metrika.client.model.response.RetargetingCondition> result = new HashSet<>();
        goalTypeToConditions.forEach((type, uidToGoals) -> {
            var conditions = EntryStream.of(uidToGoals)
                    .values()
                    .flatMap(Collection::stream)
                    .toSet();
            result.addAll(conditions);
        });
        return StreamEx.of(result)
                .toList();
    }

    public static Map<Long, List<ru.yandex.direct.metrika.client.model.response.RetargetingCondition>> getConditions(
            MetrikaClient metrikaClient,
            Collection<Long> clientRepresentativesUids,
            ru.yandex.direct.core.entity.retargeting.model.GoalType goalType,
            Collection<Long> goalIds) {
        var metrikaGoalType = toMetrikaGoalType(goalType);
        if (metrikaGoalType.isEmpty()) {
            logger.info("cannot match internal and metrika goal type {}", goalType);
            return emptyMap();
        }
        return metrikaClient.getGoals(new GetGoalsRequest()
                .withUids(clientRepresentativesUids)
                .withGoalType(metrikaGoalType.get())
                .withIds(goalIds))
                .getUidToConditions();
    }

    public static Optional<ru.yandex.direct.metrika.client.model.request.GoalType> toMetrikaGoalType(
            ru.yandex.direct.core.entity.retargeting.model.GoalType goalType) {
        if (goalType == null) {
            return Optional.empty();
        }
        switch (goalType) {
            case AB_SEGMENT:
                return Optional.of(ru.yandex.direct.metrika.client.model.request.GoalType.AB_SEGMENT);
            case ECOMMERCE:
                return Optional.of(ru.yandex.direct.metrika.client.model.request.GoalType.ECOMMERCE);
            case GOAL:
                return Optional.of(ru.yandex.direct.metrika.client.model.request.GoalType.GOAL);
            case SEGMENT:
                return Optional.of(ru.yandex.direct.metrika.client.model.request.GoalType.SEGMENT);
            case CDP_SEGMENT:
                return Optional.of(ru.yandex.direct.metrika.client.model.request.GoalType.CDP_SEGMENT);
            case AUDIENCE:
                return Optional.of(ru.yandex.direct.metrika.client.model.request.GoalType.AUDIENCE);
        }
        return Optional.empty();
    }

    public static boolean isRetargetingConditionAllowedForTextAdGroup(RetargetingCondition retargetingCondition,
                                                                      AllowedRetargetingComponentsInUserProfile allowedComponents) {

        boolean allowedTermInterests = allowedComponents.isAllInterestsInTextEnabled()
                || allowedComponents.isCustomAudienceEnabled()
                || nvl(retargetingCondition.getRules(), Collections.<Rule>emptyList())
                        .stream()
                        .filter(r -> nvl(r.getGoals(), Collections.<Goal>emptyList()).stream().anyMatch(g -> g.getType() == INTERESTS))
                        .allMatch(r -> r.getInterestType() == short_term);

        Set<GoalType> allowedGoalTypes = new HashSet<>();
        // Интересы всегда разрешены
        allowedGoalTypes.add(INTERESTS);
        if (allowedComponents.isSocialDemoInTextEnabled()) {
            allowedGoalTypes.add(SOCIAL_DEMO);
        }
        if (allowedComponents.isFamilyAndBehaviorsInTextEnabled()) {
            allowedGoalTypes.addAll(Set.of(FAMILY, BEHAVIORS));
        }
        if (allowedComponents.isMetrikaInTextEnabled()) {
            allowedGoalTypes.addAll(Set.of(GOAL, SEGMENT, AUDIENCE, LAL_SEGMENT, ECOMMERCE, AB_SEGMENT, CDP_SEGMENT));
        }
        if (allowedComponents.isCustomAudienceEnabled()) {
            allowedGoalTypes.addAll(
                    Set.of(SEGMENT, FAMILY, INTERNAL, BEHAVIORS, AUDIENCE, LAL_SEGMENT, AUDIO_GENRES, CONTENT_CATEGORY,
                            CONTENT_GENRE)
            );
        }
        if (allowedComponents.isNewCustomAudienceEnabled()) {
            allowedGoalTypes.add(GoalType.HOST);
        }
        boolean onlyAllowedGoalTypes = retargetingCondition.collectGoals()
                .stream()
                .allMatch(g -> allowedGoalTypes.contains(g.getType()));

        return onlyAllowedGoalTypes && allowedTermInterests;
    }
}
