package ru.yandex.direct.core.entity.retargeting.service.helper;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.Rule;

/**
 * Класс, позволяющий обрабатывать условия ретаргентинга с учетом lal-сегментов
 */
@Component
@ParametersAreNonnullByDefault
public class RetargetingConditionWithLalSegmentHelper {
    private final LalSegmentRepository lalSegmentRepository;

    @Autowired
    public RetargetingConditionWithLalSegmentHelper(LalSegmentRepository lalSegmentRepository) {
        this.lalSegmentRepository = lalSegmentRepository;
    }

    /**
     * Если у lal-сегмента отсутствует поле union_with_id, а его родительская цель присутствует в правиле,
     * то union_with_id нужно принудительно выставить, иначе данное условие ретаргентинга в любом случае не будет
     * иметь смысла.
     */
    public void updateLalSegmentsOnRules(List<Rule> rules) {
        List<Long> lalSegmentsWithNullUnionWithIdField = rules.stream()
                .filter(Objects::nonNull)
                .map(Rule::getGoals)
                .filter(Objects::nonNull)
                .flatMap(Collection::stream)
                .filter(Objects::nonNull)
                .filter(goal -> goal.getType() == GoalType.LAL_SEGMENT)
                .filter(goal -> goal.getUnionWithId() == null)
                .map(GoalBase::getId)
                .collect(Collectors.toList());

        if (!lalSegmentsWithNullUnionWithIdField.isEmpty()) {
            List<Goal> lalSegments = lalSegmentRepository.getLalSegmentsByIds(lalSegmentsWithNullUnionWithIdField);

            Map<Long, Long> lalSegmentToParentId = lalSegments.stream()
                    .collect(Collectors.toMap(GoalBase::getId, GoalBase::getParentId));
            rules.forEach(rule -> {
                Set<Long> goalIdsInRule = rule.getGoals().stream()
                        .filter(Objects::nonNull)
                        .map(GoalBase::getId)
                        .collect(Collectors.toSet());
                rule.getGoals().stream()
                        .filter(Objects::nonNull)
                        .filter(goal -> lalSegmentToParentId.containsKey(goal.getId()))
                        .filter(goal -> goalIdsInRule.contains(lalSegmentToParentId.get(goal.getId())))
                        .forEach(goal -> goal.setUnionWithId(lalSegmentToParentId.get(goal.getId())));
            });
        }
    }
}
