package ru.yandex.direct.core.entity.retargeting.service.uc;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.metrika.service.MetrikaSegmentService;
import ru.yandex.direct.core.entity.metrika.service.validation.MetrikaGoalsValidationService;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.result.DefectInfo;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.campaign.converter.CampaignConverter.toMetrikaCounterSource;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOAL_TIME;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_AUTORETARGETING_ENABLED;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_BASE_AUTORETARGETING_ENABLED;
import static ru.yandex.direct.feature.FeatureName.UNIVERSAL_CAMPAIGNS_LAL_AUTORETARGETING_ENABLED;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class UcRetargetingConditionService {

    private static final Logger logger = LoggerFactory.getLogger(UcRetargetingConditionService.class);
    private static final Set<FeatureName> AUTO_RETARGETING_FEATURES = Set.of(
            UNIVERSAL_CAMPAIGNS_AUTORETARGETING_ENABLED,
            UNIVERSAL_CAMPAIGNS_LAL_AUTORETARGETING_ENABLED,
            UNIVERSAL_CAMPAIGNS_BASE_AUTORETARGETING_ENABLED);
    private static final Set<FeatureName> AUTO_RETARGETING_WITH_NOT_BOUNCE_FEATURES = Set.of(
            UNIVERSAL_CAMPAIGNS_LAL_AUTORETARGETING_ENABLED,
            UNIVERSAL_CAMPAIGNS_BASE_AUTORETARGETING_ENABLED);

    private final MetrikaSegmentService metrikaSegmentService;
    private final MetrikaGoalsValidationService metrikaGoalsValidationService;
    private final RetargetingConditionService retargetingConditionService;
    private final FeatureService featureService;
    private final LalSegmentRepository lalSegmentRepository;
    private final NetAcl netAcl;
    private final CampMetrikaCountersService campMetrikaCountersService;

    @Autowired
    public UcRetargetingConditionService(MetrikaSegmentService metrikaSegmentService,
                                         MetrikaGoalsValidationService metrikaGoalsValidationService,
                                         RetargetingConditionService retargetingConditionService,
                                         FeatureService featureService,
                                         LalSegmentRepository lalSegmentRepository,
                                         NetAcl netAcl,
                                         CampMetrikaCountersService campMetrikaCountersService) {
        this.metrikaSegmentService = metrikaSegmentService;
        this.metrikaGoalsValidationService = metrikaGoalsValidationService;
        this.retargetingConditionService = retargetingConditionService;
        this.featureService = featureService;
        this.lalSegmentRepository = lalSegmentRepository;
        this.netAcl = netAcl;
        this.campMetrikaCountersService = campMetrikaCountersService;
    }

    @Nullable
    public RetargetingCondition getAutoRetargetingCondition(ClientId clientId,
                                                            @Nullable List<Integer> counterIds,
                                                            @Nullable List<Long> goalIds,
                                                            @Nullable Long adGroupId) {
        try {
            return doGetAutoRetargetingCondition(clientId, counterIds, goalIds, adGroupId);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.error("Got an exception when querying for metrika for client: " + clientId, e);
            return null;
        }
    }

    @Nullable
    private RetargetingCondition doGetAutoRetargetingCondition(ClientId clientId,
                                                               @Nullable List<Integer> counterIds,
                                                               @Nullable List<Long> goalIds,
                                                               @Nullable Long adGroupId) {
        boolean autoRetargetingEnabled = featureService.anyEnabled(clientId, AUTO_RETARGETING_FEATURES);
        if (!autoRetargetingEnabled) {
            return null;
        }

        counterIds = nvl(counterIds, emptyList());
        goalIds = nvl(goalIds, emptyList());

        counterIds = filterCounterIdsForAutoRetargeting(clientId, counterIds);
        if (isEmpty(counterIds) && isEmpty(goalIds)) {
            return null;
        }

        Set<Long> notBounceGoalIds = Set.of();
        if (featureService.anyEnabled(clientId, AUTO_RETARGETING_WITH_NOT_BOUNCE_FEATURES)) {
            // важно создать not bounce сегменты до получения доступных к ретаргетингу целей
            notBounceGoalIds = metrikaSegmentService.getOrCreateNotBounceSegmentIds(clientId, counterIds);
        }

        List<Goal> availableGoals = retargetingConditionService.getMetrikaGoalsForRetargeting(clientId);
        if (!allGoalsAreAvailable(clientId)) {
            Set<Long> availableGoalIds = listToSet(availableGoals, Goal::getId);
            goalIds = filterList(goalIds, availableGoalIds::contains);
            if (isEmpty(counterIds) && isEmpty(goalIds)) {
                logger.info("No retargeting goals available for clientId {}", clientId);
                return null;
            }
        }

        List<Goal> autoRetargetingGoals;
        if (featureService.isEnabledForClientId(clientId, UNIVERSAL_CAMPAIGNS_BASE_AUTORETARGETING_ENABLED)) {
            autoRetargetingGoals = getBaseAutoRetargetingGoals(notBounceGoalIds, goalIds);
            logger.info("Base autoretargeting goals: {}", mapList(autoRetargetingGoals, Goal::getId));
        } else if (featureService.isEnabledForClientId(clientId, UNIVERSAL_CAMPAIGNS_LAL_AUTORETARGETING_ENABLED)) {
            autoRetargetingGoals = getLalAutoRetargetingGoals(clientId, notBounceGoalIds, goalIds, availableGoals);
            logger.info("Lal autoretargeting goals: {}", mapList(autoRetargetingGoals, Goal::getId));
        } else {
            autoRetargetingGoals = getAutoRetargetingGoals(clientId, counterIds, goalIds, availableGoals);
            logger.info("Autoretargeting goals: {}", mapList(autoRetargetingGoals, Goal::getId));
        }
        return getOrCreateAutoRetargeting(clientId, adGroupId, autoRetargetingGoals);
    }

    private List<Integer> filterCounterIdsForAutoRetargeting(ClientId clientId, List<Integer> counterIds) {
        Set<Long> uniqueCounterIds = StreamEx.of(counterIds)
                .nonNull()
                .map(Integer::longValue)
                .toSet();
        List<GetExistentCountersResponseItem> existentCounters =
                campMetrikaCountersService.getExistentCounters(uniqueCounterIds);
        Set<Integer> existentCounterIds = listToSet(existentCounters, counter -> counter.getCounterId().intValue());
        Map<Integer, String> counterSourceById = StreamEx.of(existentCounters)
                .mapToEntry(counter -> counter.getCounterId().intValue(), GetExistentCountersResponseItem::getCounterSource)
                .nonNullValues()
                .toMap();

        // Фильтруем счетчики без доступа, так как не хотим создавать
        // по ним сегменты и использовать цели "посещение сайта"
        if (!allGoalsAreAvailable(clientId)) {
            Set<Long> clientAvailableCounters = metrikaGoalsValidationService
                    .getAvailableCounterIds(clientId, mapList(existentCounterIds, Integer::longValue));
            existentCounterIds =
                    filterToSet(existentCounterIds, counter -> clientAvailableCounters.contains(counter.longValue()));
        }

        return StreamEx.of(counterIds)
                .nonNull()
                .filter(existentCounterIds::contains)
                .filter(counterId -> toMetrikaCounterSource(counterSourceById.get(counterId)) != MetrikaCounterSource.SPRAV)
                .toList();
    }

    /**
     * сегменты вида:
     * <li>только lal на «неотказную аудиторию сайта»</li>
     * <li>только lal на аудиторию, достигавшую целей из настроек кампании</li>
     */
    private List<Goal> getLalAutoRetargetingGoals(ClientId clientId,
                                                  Set<Long> notBounceGoalIds,
                                                  List<Long> goalIds,
                                                  List<Goal> availableGoals) {
        Set<Long> baseGoalIds = StreamEx.of(notBounceGoalIds)
                .append(goalIds)
                .nonNull()
                .toSet();
        return getOrCreateLalRetargetingGoals(clientId, baseGoalIds, availableGoals, false);
    }

    /**
     * сегменты вида:
     * <li>только «неотказная аудитория сайта»</li>
     * <li>только аудитория, достигавшая целей из настроек кампании</li>
     */
    private List<Goal> getBaseAutoRetargetingGoals(Set<Long> notBounceGoalIds,
                                                   List<Long> goalIds) {
        return StreamEx.of(notBounceGoalIds)
                .append(goalIds)
                .nonNull()
                .distinct()
                .map(goalId -> composeRetargetingGoal(goalId, null))
                .toList();
    }

    /**
     * сегменты вида:
     * <li>«аудитория сайта»</li>
     * <li>lal на «аудиторию сайта»</li>
     * <li>аудитория, достигавшая целей из настроек кампании</li>
     * <li>lal на аудиторию, достигавшую целей из настроек кампании</li>
     */
    private List<Goal> getAutoRetargetingGoals(ClientId clientId,
                                               List<Integer> counterIds,
                                               List<Long> goalIds,
                                               List<Goal> availableGoals) {
        Set<Long> autoRetargetingGoalIds = StreamEx.of(counterIds)
                .map(counterId -> Goal.METRIKA_COUNTER_LOWER_BOUND + counterId)
                .append(goalIds)
                .nonNull()
                .toSet();
        List<Goal> lalRetargetingGoals =
                getOrCreateLalRetargetingGoals(clientId, autoRetargetingGoalIds, availableGoals, true);

        return StreamEx.of(autoRetargetingGoalIds)
                .map(goalId -> composeRetargetingGoal(goalId, null))
                .append(lalRetargetingGoals)
                .toList();
    }

    public List<Goal> getOrCreateLalRetargetingGoals(ClientId clientId, Set<Long> baseGoalIds,
                                                     List<Goal> availableGoals) {
        List<Goal> existingLalSegments = lalSegmentRepository.getLalSegmentsByParentIds(baseGoalIds);

        Set<Long> goalIdsWithLalSegment = listToSet(existingLalSegments, Goal::getParentId);
        List<Long> goalIdsWithoutLalSegment = StreamEx.of(baseGoalIds)
                .remove(goalIdsWithLalSegment::contains)
                .toList();

        if (!goalsForLalSegmentCreationAreValid(clientId, goalIdsWithoutLalSegment, availableGoals)) {
            return emptyList();
        }

        List<Goal> createdLalSegments = lalSegmentRepository.createLalSegments(goalIdsWithoutLalSegment);
        return StreamEx.of(existingLalSegments)
                .append(createdLalSegments)
                .toList();
    }

    private List<Goal> getOrCreateLalRetargetingGoals(ClientId clientId, Set<Long> baseGoalIds,
                                                      List<Goal> availableGoals, boolean unitWithBase) {
        List<Goal> lalSegments = getOrCreateLalRetargetingGoals(clientId, baseGoalIds, availableGoals);
        return StreamEx.of(lalSegments)
                .map(lal -> composeRetargetingGoal(lal.getId(), unitWithBase ? lal.getParentId() : null))
                .toList();
    }

    private boolean goalsForLalSegmentCreationAreValid(ClientId clientId,
                                                       List<Long> goalIdsWithoutLalSegment,
                                                       List<Goal> availableGoals) {
        if (isEmpty(goalIdsWithoutLalSegment) || allGoalsAreAvailable(clientId)) {
            return true;
        }

        var vr = metrikaGoalsValidationService.validateGoalsForLalSegmentCreation(
                goalIdsWithoutLalSegment, availableGoals);
        if (vr.hasAnyErrors()) {
            String vrErrors = vr.flattenErrors().stream()
                    .map(DefectInfo::toString)
                    .collect(Collectors.joining("; "));
            logger.error("Validation error for LAL-segment creation: {}", vrErrors);
            return false;
        }

        return true;
    }

    private boolean allGoalsAreAvailable(ClientId clientId) {
        boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                .map(netAcl::isInternalIp)
                .orElse(false);
        return requestFromInternalNetwork
                && !featureService.isEnabledForClientId(clientId, FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED);
    }

    @Nullable
    private RetargetingCondition getOrCreateAutoRetargeting(ClientId clientId,
                                                            @Nullable Long adGroupId,
                                                            @Nullable List<Goal> autoRetargetingGoals) {
        if (isEmpty(autoRetargetingGoals)) {
            return null;
        }

        Rule conditionRule = new Rule()
                .withType(RuleType.OR)
                .withGoals(autoRetargetingGoals);

        return Optional.ofNullable(getExistingAutoRetargeting(clientId, conditionRule, adGroupId))
                .orElseGet(() -> getNewAutoRetargeting(clientId, conditionRule));
    }

    @Nullable
    private RetargetingCondition getExistingAutoRetargeting(ClientId clientId,
                                                            Rule conditionRule,
                                                            @Nullable Long adGroupId) {
        List<RetargetingCondition> existingRetConditions = retargetingConditionService.getRetargetingConditions(clientId,
                null, List.of(RetargetingConditionsRetargetingConditionsType.metrika_goals), LimitOffset.maxLimited());

        return StreamEx.of(existingRetConditions)
                .filter(rc -> rc.getAutoRetargeting() == Boolean.TRUE)
                .filter(rc -> rc.getRules().get(0).equals(conditionRule))
                .findFirst()
                .map(retCond ->
                        (RetargetingCondition) retCond.withTargetInterest(getTargetInterestForExisting(retCond, adGroupId)))
                .orElse(null);
    }

    private RetargetingCondition getNewAutoRetargeting(ClientId clientId,
                                                       Rule conditionRule) {
        var retargetingCondition = new RetargetingCondition();
        retargetingCondition.withName(String.format("UC autoretargeting %d %s",
                clientId.asLong(), LocalDateTime.now()))
                .withClientId(clientId.asLong())
                .withType(ConditionType.metrika_goals)
                .withAutoRetargeting(true)
                .withRules(List.of(conditionRule))
                // это означает что также создаем новый таргет (таблица bids_retargeting)
                .withTargetInterest(new TargetInterest().withAdGroupId(0L));
        return retargetingCondition;
    }

    private TargetInterest getTargetInterestForExisting(RetargetingCondition retargetingCondition,
                                                        @Nullable Long adGroupId) {
        return new TargetInterest()
                .withRetargetingConditionId(retargetingCondition.getId())
                .withAdGroupId(adGroupId);
    }

    private static Goal composeRetargetingGoal(Long id, @Nullable Long unionWithId) {
        return (Goal) new Goal()
                .withId(id)
                .withUnionWithId(unionWithId)
                .withTime(MAX_GOAL_TIME);
    }
}
