package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.metrika.service.MobileGoalsService;
import ru.yandex.direct.core.entity.retargeting.Constants;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingConditionValidationData;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesCache;
import ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.retargeting.service.validation2.ExistingRetargetingConditionsInfo.fromRetargetingConditionsValidationData;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingConditionsValidator.retConditionsIsValid;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.defect.CollectionDefects.maxElementsExceeded;

@Service
public class AddRetargetingConditionValidationService2 {

    private final RetargetingConditionRepository retConditionRepository;
    private final RetargetingConditionCryptaSegmentsProvider retargetingConditionCryptaSegmentsProvider;
    private final LalSegmentRepository lalSegmentRepository;
    private final ShardHelper shardHelper;
    private final GoalUtilsService goalUtilsService;
    private final FeatureService featureService;
    private final RbacService rbacService;
    private final TargetingCategoriesCache targetingCategoriesCache;
    private final MobileGoalsService mobileGoalsService;
    private final NetAcl netAcl;

    @Autowired
    public AddRetargetingConditionValidationService2(
            RetargetingConditionRepository retConditionRepository,
            RetargetingConditionCryptaSegmentsProvider retargetingConditionCryptaSegmentsProvider,
            LalSegmentRepository lalSegmentRepository,
            ShardHelper shardHelper,
            GoalUtilsService goalUtilsService,
            TargetingCategoriesCache targetingCategoriesCache,
            FeatureService featureService,
            RbacService rbacService,
            MobileGoalsService mobileGoalsService,
            NetAcl netAcl) {
        this.retConditionRepository = retConditionRepository;
        this.retargetingConditionCryptaSegmentsProvider = retargetingConditionCryptaSegmentsProvider;
        this.lalSegmentRepository = lalSegmentRepository;
        this.shardHelper = shardHelper;
        this.goalUtilsService = goalUtilsService;
        this.featureService = featureService;
        this.rbacService = rbacService;
        this.targetingCategoriesCache = targetingCategoriesCache;
        this.mobileGoalsService = mobileGoalsService;
        this.netAcl = netAcl;
    }

    private static Constraint<List<RetargetingCondition>, Defect> maxConditionsPerClient(
            int existingConditionsSize) {
        final int maxRetConditionsPerClient = Constants.MAX_RET_CONDITIONS_PER_CLIENT;

        Predicate<List<RetargetingCondition>> predicate = rcList -> {
            List<RetargetingCondition> retargetingConditionsToCount = StreamEx.of(rcList)
                    .filter(rc -> !nvl(rc.getAutoRetargeting(), false))
                    .toList();
            return retargetingConditionsToCount.size() + existingConditionsSize <= maxRetConditionsPerClient;
        };
        return Constraint.fromPredicate(predicate, maxElementsExceeded(maxRetConditionsPerClient));
    }

    public ValidationResult<List<RetargetingCondition>, Defect> validate(
            List<RetargetingCondition> retConditionList, ClientId clientId) {
        return validate(retConditionList, clientId, false);
    }

    public ValidationResult<List<RetargetingCondition>, Defect> validate(
            List<RetargetingCondition> retConditionList, ClientId clientId, boolean forceSkipGoalExistenceCheck) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<RetargetingConditionValidationData> existingConditionsData =
                retConditionRepository.getValidationData(shard, clientId);
        ExistingRetargetingConditionsInfo existingRetargetingConditionsInfo =
                fromRetargetingConditionsValidationData(existingConditionsData);
        return validateWithOptionalForceSkipGoalExistenceCheck(
                retConditionList, clientId, shard, existingRetargetingConditionsInfo, forceSkipGoalExistenceCheck);
    }

    public ValidationResult<List<RetargetingCondition>, Defect> validateWithOptionalForceSkipGoalExistenceCheck(
            List<RetargetingCondition> retConditionList,
            ClientId clientId,
            int shard,
            ExistingRetargetingConditionsInfo existingRetargetingConditionsInfo,
            boolean forceSkipGoalExistenceCheck) {
        boolean skipGoalExistenceCheck = forceSkipGoalExistenceCheck ||
                featureService.isEnabledForClientId(clientId, FeatureName.SKIP_GOAL_EXISTENCE_FOR_AGENCY);
        if (!skipGoalExistenceCheck) {
            boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                    .map(netAcl::isInternalIp)
                    .orElse(false);
            skipGoalExistenceCheck = requestFromInternalNetwork
                    && !featureService.isEnabledForClientId(clientId, FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED);
        }
        return validate(retConditionList, clientId, shard, existingRetargetingConditionsInfo, skipGoalExistenceCheck);
    }

    public ValidationResult<List<RetargetingCondition>, Defect> validate(
            List<RetargetingCondition> retConditionList,
            ClientId clientId,
            int shard,
            ExistingRetargetingConditionsInfo existingRetargetingConditions,
            boolean skipGoalExistenceCheck) {
        Set<Long> metrikaGoalIds = new HashSet<>();
        if (!skipGoalExistenceCheck) {
            var lalGoalIds = StreamEx.of(retConditionList)
                    .map(RetargetingCondition::collectGoalsSafe)
                    .flatMap(Collection::stream)
                    .filter(t -> t.getType() == GoalType.LAL_SEGMENT)
                    .map(GoalBase::getId)
                    .nonNull()
                    .toSet();
            List<Goal> lalSegments = lalSegmentRepository.getLalSegmentsByIds(lalGoalIds);
            var goalIds = StreamEx.of(retConditionList)
                    .map(RetargetingCondition::collectGoalsSafe)
                    .flatMap(Collection::stream)
                    .map(GoalBase::getId)
                    .append(mapList(lalSegments, GoalBase::getParentId))
                    .nonNull()
                    .toSet();
            metrikaGoalIds.addAll(goalUtilsService.getAvailableMetrikaGoalIds(clientId, goalIds));
            List<Long> mobileGoalIds = mapList(mobileGoalsService.getAllAvailableInAppMobileGoals(clientId),
                    GoalBase::getId);
            metrikaGoalIds.addAll(mobileGoalIds);
        }

        var isCustomAudienceEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.CUSTOM_AUDIENCE_ENABLED);
        var isNewCustomAudienceEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.NEW_CUSTOM_AUDIENCE_ENABLED);
        boolean isInternalAd = rbacService.isInternalAdProduct(clientId);

        Map<Long, Goal> allCryptaGoals = retargetingConditionCryptaSegmentsProvider.getAllowedCryptaSegments(
                isInternalAd, isNewCustomAudienceEnabled, retConditionList
        );

        List<Goal> lalSegments = lalSegmentRepository.getLalSegmentsByParentIds(metrikaGoalIds);

        Map<Long, Set<Long>> mutuallyExclusiveGoals = goalUtilsService.getMutuallyExclusiveGoalsMap();

        List<InterestLink> existingTargetingByInterests = retConditionRepository.getExistingInterest(shard, clientId);
        Set<Long> existingInterestTargetingIds = listToSet(existingTargetingByInterests, InterestLink::getGoalId);

        return ListValidationBuilder.<RetargetingCondition, Defect>of(retConditionList)
                .checkBy(conditions -> validateOperation(conditions, existingRetargetingConditions.getExistingCount()))
                .checkBy(
                        retConditionsIsValid(
                                metrikaGoalIds,
                                lalSegments,
                                allCryptaGoals,
                                existingRetargetingConditions.getExistingNames(),
                                existingRetargetingConditions.getExistingRules(),
                                existingInterestTargetingIds,
                                targetingCategoriesCache.getTargetingCategories(),
                                mutuallyExclusiveGoals,
                                skipGoalExistenceCheck,
                                isInternalAd,
                                isCustomAudienceEnabled,
                                isNewCustomAudienceEnabled),
                        When.isValid())
                .getResult();
    }

    // todo remove api-level checks
    private ValidationResult<List<RetargetingCondition>, Defect> validateOperation(
            List<RetargetingCondition> conditions, int existingConditionsSize) {
        return ListValidationBuilder.<RetargetingCondition, Defect>of(conditions)
                .check(maxConditionsPerClient(existingConditionsSize))
                .getResult();
    }

}
