package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.container.AllowedRetargetingComponentsInUserProfile;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingAdGroupInfo;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesCache;
import ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionIdCpmPriceValidator;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionsCpmPriceValidationData;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionsCpmPriceValidationDataFactory;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@ParametersAreNonnullByDefault
@Service
public class AddRetargetingValidationService {
    private final ClientService clientService;
    private final RbacService rbacService;
    private final CampaignRepository campaignRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final AdGroupRepository adGroupRepository;
    private final TargetingCategoriesCache targetingCategoriesCache;
    private final RetargetingsWithAdsValidator retargetingsWithAdsValidator;
    private final FeatureService featureService;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final RetargetingConditionsCpmPriceValidationDataFactory cpmPriceValidationDataFactory;

    @Autowired
    public AddRetargetingValidationService(
            ClientService clientService,
            RbacService rbacService,
            CampaignRepository campaignRepository,
            RetargetingConditionRepository retargetingConditionRepository,
            AdGroupRepository adGroupRepository,
            TargetingCategoriesCache targetingCategoriesCache,
            RetargetingsWithAdsValidator retargetingsWithAdsValidator,
            FeatureService featureService,
            CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
            RetargetingConditionsCpmPriceValidationDataFactory cpmPriceValidationDataFactory) {
        this.clientService = clientService;
        this.rbacService = rbacService;
        this.campaignRepository = campaignRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.adGroupRepository = adGroupRepository;
        this.targetingCategoriesCache = targetingCategoriesCache;
        this.retargetingsWithAdsValidator = retargetingsWithAdsValidator;
        this.featureService = featureService;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.cpmPriceValidationDataFactory = cpmPriceValidationDataFactory;
    }

    public ValidationResult<List<TargetInterest>, Defect> validate(
            ValidationResult<List<TargetInterest>, Defect> validationResult,
            List<TargetInterest> existingTargetInterests, Map<Long, AdGroupSimple> clientAdGroupsFromRetargetingsMap,
            long operatorUid, ClientId clientId, int shard) {
        return validate(validationResult, existingTargetInterests, clientAdGroupsFromRetargetingsMap,
                operatorUid, clientId, shard, false);
    }

    public ValidationResult<List<TargetInterest>, Defect> validate(
            ValidationResult<List<TargetInterest>, Defect> validationResult,
            List<TargetInterest> existingTargetInterests, Map<Long, AdGroupSimple> clientAdGroupsFromRetargetingsMap,
            long operatorUid, ClientId clientId, int shard, boolean doNotCheckRegionalCpmYndxFrontpagePrice) {

        Map<Long, TargetInterest> existingTargetInterestsById =
                listToMap(existingTargetInterests, TargetInterest::getId);

        List<TargetInterest> newTargetInterests = getValidItems(validationResult);
        Collection<TargetInterest> allTargetInterests = CollectionUtils.union(newTargetInterests,
                existingTargetInterestsById.values());
        Map<Long, RetargetingCondition> ownRetConditionsById = getOwnRetConditions(allTargetInterests, clientId, shard);

        Set<Long> clientCampaignIds =
                listToSet(clientAdGroupsFromRetargetingsMap.values(), AdGroupSimple::getCampaignId);
        Set<Long> writableCampaignIds = rbacService.getWritableCampaigns(operatorUid, clientCampaignIds);
        Set<Long> visibleCampaignIds = rbacService.getVisibleCampaigns(operatorUid, clientCampaignIds);

        Map<Long, CampaignSimple> clientCampaigns = campaignRepository.getCampaignsSimple(shard, clientCampaignIds);
        List<TargetingCategory> allTargetingCategories = targetingCategoriesCache.getTargetingCategories();

        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        Set<String> clientFeatures = featureService.getEnabledForClientId(clientId);
        Boolean retargetingsAllowedForCpmYndxFrontpage = clientFeatures.contains(
                FeatureName.CPM_YNDX_FRONTPAGE_PROFILE.getName());

        List<Long> cpmAdGroupIds =
                mapList(filterList(clientAdGroupsFromRetargetingsMap.values(),
                        adGroup -> adGroup.getType() == AdGroupType.CPM_BANNER
                                || adGroup.getType() == AdGroupType.CPM_GEOPRODUCT
                                || adGroup.getType() == AdGroupType.CPM_GEO_PIN
                                || adGroup.getType() == AdGroupType.CPM_VIDEO
                                || adGroup.getType() == AdGroupType.CPM_AUDIO
                                || adGroup.getType() == AdGroupType.CPM_YNDX_FRONTPAGE),
                        AdGroupSimple::getId);
        Map<Long, CriterionType> cpmAdGroupsWithCriterionType =
                adGroupRepository.getCriterionTypeByAdGroupIds(shard, cpmAdGroupIds);
        Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageAdGroupRestrictionsMap =
                cpmYndxFrontpageCurrencyService.getAdGroupIdsToPriceDataMapByAdGroups(clientAdGroupsFromRetargetingsMap.values(),
                        shard, clientCurrency);

        boolean textBannerInterestsRetCondEnabled = clientFeatures.contains(
                FeatureName.TEXT_BANNER_INTERESTS_RET_COND_ENABLED.getName());

        AllowedRetargetingComponentsInUserProfile allowedComponents = new AllowedRetargetingComponentsInUserProfile()
                .withSocialDemoInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_SOCIAL_DEMO_IN_USER_PROFILE.getName()))
                .withFamilyAndBehaviorsInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_FAMILY_AND_BEHAVIORS_IN_USER_PROFILE.getName()))
                .withAllInterestsInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_ALL_INTERESTS_IN_USER_PROFILE.getName()))
                .withMetrikaInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_METRIKA_AND_AUDIENCE_IN_USER_PROFILE.getName()))
                .withCustomAudienceEnabled(clientFeatures.contains(
                        FeatureName.CUSTOM_AUDIENCE_ENABLED.getName()))
                .withNewCustomAudienceEnabled(clientFeatures.contains(
                        FeatureName.NEW_CUSTOM_AUDIENCE_ENABLED.getName()));

        RetargetingConditionsCpmPriceValidationData cpmPriceValidationData = cpmPriceValidationDataFactory
                .createForAddRetargetingsWithExistentAdGroups(shard, newTargetInterests);

        Validator<List<TargetInterest>, Defect> validator =
                new RetargetingsWithAdGroupsValidator(clientAdGroupsFromRetargetingsMap,
                        ownRetConditionsById, existingTargetInterestsById, clientCampaigns, clientCurrency,
                        writableCampaignIds, visibleCampaignIds, allTargetingCategories, cpmAdGroupsWithCriterionType,
                        cpmYndxFrontpageAdGroupRestrictionsMap, retargetingsAllowedForCpmYndxFrontpage,
                        textBannerInterestsRetCondEnabled,
                        allowedComponents,
                        cpmPriceValidationData,
                        doNotCheckRegionalCpmYndxFrontpagePrice);

        return new ListValidationBuilder<>(validationResult)
                .check(eachNotNull(), When.isValid())
                .checkBy(validator, When.isValid())
                //проверка на то, что не поломали ничего в баннерах групп, связанных с данными ретаргетингами
                .checkBy(itemList -> retargetingsWithAdsValidator
                                .validateInterconnectionsWithAds(shard, clientId, itemList, cpmAdGroupIds),
                        When.isValid())
                .getResult();
    }
    public ValidationResult<List<TargetInterest>, Defect> validateWithNonExistentAdGroups(
            ValidationResult<List<TargetInterest>, Defect> validationResult,
            Map<Integer, RetargetingAdGroupInfo> retargetingsInfo, List<RetargetingCondition> retargetingConditions,
            boolean retargetingConditionsNonexistent, ClientId clientId, int shard) {
        return validateWithNonExistentAdGroups(validationResult, retargetingsInfo, retargetingConditions,
                retargetingConditionsNonexistent, clientId, shard, false);
    }

    public ValidationResult<List<TargetInterest>, Defect> validateWithNonExistentAdGroups(
            ValidationResult<List<TargetInterest>, Defect> validationResult,
            Map<Integer, RetargetingAdGroupInfo> retargetingsInfo, List<RetargetingCondition> retargetingConditions,
            boolean retargetingConditionsNonexistent, ClientId clientId, int shard,
            boolean doNotCheckRegionalCpmYndxFrontpagePrice) {
        Map<Long, RetargetingCondition> ownRetConditionsById =
                listToMap(retargetingConditions, RetargetingCondition::getId);

        Map<Long, List<RetargetingAdGroupInfo>> retargetingsInfoByRetConditionId = EntryStream.of(retargetingsInfo)
                .mapKeys(i -> validationResult.getValue().get(i).getRetargetingConditionId())
                .nonNullKeys()
                .grouping();

        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        boolean retargetingsAllowedForCpmYndxFrontpage =
                featureService.isEnabledForClientId(clientId, FeatureName.CPM_YNDX_FRONTPAGE_PROFILE);

        boolean textBannerInterestsRetCondEnabled =
                featureService.isEnabledForClientId(clientId, FeatureName.TEXT_BANNER_INTERESTS_RET_COND_ENABLED);

        List<TargetingCategory> allTargetingCategories = targetingCategoriesCache.getTargetingCategories();

        Map<Long, AdGroupType> adGroupTypesByIds = StreamEx.of(retargetingsInfo.values())
                .toMap(RetargetingAdGroupInfo::getAdGroupFakeId, RetargetingAdGroupInfo::getAdGroupType, (a, b) -> a);

        Map<Long, CampaignType> campaignTypesByAdGroupIds = StreamEx.of(retargetingsInfo.values())
                .mapToEntry(RetargetingAdGroupInfo::getAdGroupFakeId, RetargetingAdGroupInfo::getCampaignType)
                .toMap((a, b) -> a);

        Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageAdGroupPriceRestrictionsMap =
                StreamEx.of(retargetingsInfo.values())
                        .filter(t -> t.getCpmYndxFrontpageAdGroupPriceRestrictions() != null)
                        .toMap(RetargetingAdGroupInfo::getAdGroupFakeId,
                                RetargetingAdGroupInfo::getCpmYndxFrontpageAdGroupPriceRestrictions,
                                (a, b) -> a);

        RetargetingsCommonValidator validator =
                new RetargetingsCommonValidator(ownRetConditionsById, retargetingConditionsNonexistent,
                        new HashMap<>(), clientCurrency, allTargetingCategories, adGroupTypesByIds,
                        campaignTypesByAdGroupIds, cpmYndxFrontpageAdGroupPriceRestrictionsMap,
                        doNotCheckRegionalCpmYndxFrontpagePrice);

        RetargetingConditionsCpmPriceValidationData cpmPriceValidationData = cpmPriceValidationDataFactory
                .createForAddRetargetingsWithNonExistentAdGroups(shard, retargetingsInfoByRetConditionId);

        return new ListValidationBuilder<>(validationResult)
                .check(eachNotNull(), When.isValid())
                .checkBy(validator, When.isValid())
                .checkEachBy((index, targetInterest) -> {
                    ModelItemValidationBuilder<TargetInterest> v =
                            ModelItemValidationBuilder.of(targetInterest);

                    RetargetingAdGroupInfo retargetingInfo = retargetingsInfo.get(index);
                    AdGroupType adGroupType = retargetingInfo.getAdGroupType();
                    CampaignType campaignType = retargetingInfo.getCampaignType();
                    RetargetingCondition retargetingCondition =
                            ownRetConditionsById.get(targetInterest.getRetargetingConditionId());

                    v.item(TargetInterest.AD_GROUP_ID)
                            .check(RetargetingConstraints
                                    .adGroupTypeValid(targetInterest, adGroupType, retargetingCondition, textBannerInterestsRetCondEnabled));
                    v.item(TargetInterest.RETARGETING_CONDITION_ID)
                            .check(notNull(),
                                    When.isValidAnd(When.isTrue(adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE)))
                            .check(RetargetingConstraints.retConditionIsValidForCpmYndxFrontpage(retargetingCondition),
                                    When.isValidAnd(When.isTrue(adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE && campaignType != CampaignType.CPM_PRICE
                                            && !retargetingsAllowedForCpmYndxFrontpage)))
                            .check(RetargetingConstraints.onlyToDemographics(retargetingCondition),
                                    When.isValidAnd(When.isTrue(adGroupType == AdGroupType.CPM_INDOOR)))
                            .checkBy(new RetargetingConditionIdCpmPriceValidator(cpmPriceValidationData,
                                            ownRetConditionsById),
                                    When.isValidAnd(When.isTrue(campaignType == CampaignType.CPM_PRICE)));

                    return v.getResult();
                }, When.isValid())
                .getResult();
    }

    private Map<Long, RetargetingCondition> getOwnRetConditions(Collection<TargetInterest> targetInterests,
                                                                ClientId clientId, int shard) {
        Set<Long> retCondIds = StreamEx.of(targetInterests)
                .filter(Objects::nonNull)
                .map(TargetInterest::getRetargetingConditionId)
                .filter(Objects::nonNull)
                .collect(toSet());
        List<RetargetingCondition> ownRetConditions =
                retargetingConditionRepository.getFromRetargetingConditionsTable(shard, clientId, retCondIds);
        return listToMap(ownRetConditions, RetargetingCondition::getId);
    }


}
