package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.retargetingConditionIsInvalidForRetargeting;

public class RetargetingConditionsIndoorValidator implements DefaultValidator<List<RetargetingCondition>> {

    private final Map<Long, List<Long>> adGroupIdsByRetConditionIds;
    private final Map<Long, AdGroupWithType> adGroupsWithType;

    public RetargetingConditionsIndoorValidator(
            Map<Long, List<Long>> adGroupIdsByRetConditionIds,
            Map<Long, AdGroupWithType> adGroupsWithType) {
        this.adGroupIdsByRetConditionIds = adGroupIdsByRetConditionIds;
        this.adGroupsWithType = adGroupsWithType;
    }

    @Override
    public ValidationResult<List<RetargetingCondition>, Defect> apply(
            List<RetargetingCondition> retargetingConditions)
    {
        return ListValidationBuilder.of(retargetingConditions, Defect.class)
                .checkEach(hasIndoorDefects())
                .getResult();
    }

    private Constraint<RetargetingCondition, Defect> hasIndoorDefects() {
        return retCondition -> {
            List<Long> linkedGroupIds = adGroupIdsByRetConditionIds.get(retCondition.getId());
            boolean hasIndoorGroups = linkedGroupIds != null &&
                    linkedGroupIds.stream().map(adGroupsWithType::get).filter(Objects::nonNull)
                            .anyMatch(adGroupWithType -> adGroupWithType.getType() == AdGroupType.CPM_INDOOR);
            if (!hasIndoorGroups) {
                return null;
            }
            boolean result = RetargetingConstraints.onlyToDemographics(Collections.singletonList(retCondition));
            if (result) {
                return null;
            }
            return retargetingConditionIsInvalidForRetargeting();
        };
    }
}
