package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionMappings;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Predicate.not;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.retargeting.Constants.CRYPTA_PARENT_IDS_FOR_VALIDATION;
import static ru.yandex.direct.core.entity.retargeting.Constants.GOALS_PER_RULE_FOR_INTEREST_TARGETING;
import static ru.yandex.direct.core.entity.retargeting.Constants.INTEREST_LINK_TIME_VALUE;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_DESCRIPTION_LENGTH;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOALS_PER_INTEREST_RULE;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOALS_PER_RULE;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOALS_PER_RULE_FOR_CA;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOAL_TIME;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_INTEREST_RULES_PER_CONDITION;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_NAME_LENGTH;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_RULES_PER_CONDITION;
import static ru.yandex.direct.core.entity.retargeting.Constants.MIN_GOALS_PER_RULE;
import static ru.yandex.direct.core.entity.retargeting.Constants.MIN_GOAL_TIME;
import static ru.yandex.direct.core.entity.retargeting.Constants.MIN_RULES_PER_CONDITION;
import static ru.yandex.direct.core.entity.retargeting.Constants.MIN_RULES_PER_INTEREST_CONDITION;
import static ru.yandex.direct.core.entity.retargeting.Constants.RULES_PER_CONDITION_FOR_INTEREST_TARGETING;
import static ru.yandex.direct.core.entity.retargeting.model.ConditionType.ab_segments;
import static ru.yandex.direct.core.entity.retargeting.model.ConditionType.interests;
import static ru.yandex.direct.core.entity.retargeting.model.ConditionType.shortcuts;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.BEHAVIORS;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.FAMILY;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.GOAL;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.INTERESTS;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.LAL_SEGMENT;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.SOCIAL_DEMO;
import static ru.yandex.direct.core.entity.retargeting.model.RuleType.NOT;
import static ru.yandex.direct.core.entity.retargeting.model.RuleType.OR;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.allCryptaGoalsMustHaveSameType;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.allElementsAreNegative;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.allGoalsMustBeEitherFromMetrikaOrCrypta;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.cryptaGoalsAllowedOnlyForInterestsType;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.cryptaGoalsAllowedOnlyForOrCondition;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.duplicatedGoal;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.duplicatedObjectWithName;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.duplicatedObjectWithRules;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.interestsTypeIsNotSpecified;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.mustHaveSameParentId;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.mustNotContainAllElements;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.mutuallyExclusiveParameters;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.retargetingConditionAlreadyExists;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.retargetingConditionIsInvalidForRetargeting;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.ConditionConsistencyConstraint3.conditionIsConsistent;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.GoalExistenceConstraint2.goalExists;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.GoalTimeConsistencyConstraint2.goalTimeIsConsistent;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.OrphanFamilySegmentConstraint.isNotAnOrphanFamilySegment;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.collectionSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.StringConstraints.admissibleChars;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentStateAlreadyExists;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

public class RetargetingConditionsValidator implements DefaultValidator<List<RetargetingCondition>> {

    public static final Long ORPHAN_SEGMENT_PARENT_ID = 0L;
    private static final Set<ConditionType> TYPES_ALLOWED_DUPLICATE_NAME = Set.of(interests, ab_segments, shortcuts);
    private static final Set<ConditionType> TYPES_ALLOWED_DUPLICATE_RULES = Set.of(interests, shortcuts);
    private static final Set<ConditionType> TYPES_ALLOWED_BLANK_NAME = Set.of(ab_segments);

    private final Set<Long> metrikaGoalIds;
    private final Set<Long> lalSegmentIds;
    private final Map<Long, Long> lalSegmentIdToParentId;
    private final Map<Long, Goal> allCryptaGoals;
    private final Set<String> existingNames;
    private final Set<Long> existingInterestTargetingIds;
    private final Map<Long, TargetingCategory> targetingCategoryByImportId;
    private final Set<Long> orphanFamilySegmentIDs;
    private final Map<Long, Set<Long>> mutuallyExclusiveGoals;
    private final Set<List<Rule>> parsedExistingRules;
    private final boolean skipGoalExistenceCheck;
    private final boolean forInternalAd;
    private final boolean isCustomAudienceEnabled;
    private final boolean isNewCustomAudienceEnabled;

    /**
     * @param metrikaGoalIds               все метричные цели, доступные хотябы одному представителю клиента
     * @param lalSegments                  доступные клиенту lal-сегменты
     * @param allCryptaGoals               доступные клиенту крипта-сегменты
     * @param existingNames                имена существующих у клиента условий ретаргетинга
     * @param existingRules                существующие у клиента правила условий ретаргетинга
     * @param existingInterestTargetingIds существующие у клиента условия нацеливания по интересам (категории интересов)
     *                                     когда retargeting_conditions c properties = interest
     * @param targetingCategories          список категорий интересов
     * @param mutuallyExclusiveGoals       наборы взаимоисключающих целей, которые не должны появляться в рамках одного
     *                                     правила
     * @param skipGoalExistenceCheck       не проверять доступность цели !! внимание !! использовать с осторожностью
     *                                     т.к. может привести к уязвимостям
     * @param forInternalAd                выполнять валидацию, учитывая разрешения внутренней рекламы (internalAd,
     *                                     banana)
     */
    private RetargetingConditionsValidator(
            Set<Long> metrikaGoalIds,
            List<Goal> lalSegments,
            Map<Long, Goal> allCryptaGoals,
            Set<String> existingNames,
            Set<String> existingRules,
            Set<Long> existingInterestTargetingIds,
            List<TargetingCategory> targetingCategories,
            Map<Long, Set<Long>> mutuallyExclusiveGoals,
            boolean skipGoalExistenceCheck,
            boolean forInternalAd,
            boolean isCustomAudienceEnabled,
            boolean isNewCustomAudienceEnabled) {
        this.metrikaGoalIds = metrikaGoalIds;
        this.lalSegmentIds = listToSet(lalSegments, GoalBase::getId);
        this.lalSegmentIdToParentId = listToMap(lalSegments, GoalBase::getId, GoalBase::getParentId);
        this.allCryptaGoals = allCryptaGoals;
        this.existingNames = existingNames;
        this.parsedExistingRules = listToSet(existingRules, RetargetingConditionMappings::rulesFromJson);
        this.existingInterestTargetingIds = existingInterestTargetingIds;
        this.targetingCategoryByImportId = StreamEx.of(targetingCategories)
                .mapToEntry(cat -> cat.getImportId().longValue())
                .invert()
                .toMap();

        this.orphanFamilySegmentIDs = Optional.ofNullable(allCryptaGoals).orElse(emptyMap()).values().stream()
                .filter(g -> g.getType() == FAMILY && ORPHAN_SEGMENT_PARENT_ID.equals(g.getParentId()))
                .map(GoalBase::getId)
                .collect(toSet());
        this.mutuallyExclusiveGoals = mutuallyExclusiveGoals;
        this.skipGoalExistenceCheck = skipGoalExistenceCheck;
        this.forInternalAd = forInternalAd;
        this.isCustomAudienceEnabled = isCustomAudienceEnabled;
        this.isNewCustomAudienceEnabled = isNewCustomAudienceEnabled;
    }

    static RetargetingConditionsValidator retConditionsIsValid(
            Set<Long> clientGoalIds,
            List<Goal> lalSegments,
            Map<Long, Goal> allCryptaGoals,
            Set<String> existingNames,
            Set<String> existingRules,
            Set<Long> existingInterestTargetingIds,
            List<TargetingCategory> targetingCategories,
            Map<Long, Set<Long>> mutuallyExclusiveGoals,
            boolean skipGoalExistenceCheck, boolean forInternalAd,
            boolean isCustomAudienceEnabled,
            boolean isNewCustomAudienceEnabled) {
        return new RetargetingConditionsValidator(clientGoalIds, lalSegments, allCryptaGoals,
                existingNames, existingRules, existingInterestTargetingIds, targetingCategories,
                mutuallyExclusiveGoals, skipGoalExistenceCheck, forInternalAd, isCustomAudienceEnabled,
                isNewCustomAudienceEnabled);
    }

    /*
        Этот метод можно использовать только для простой проверки при оценке охвата
        Для настоящей валидации нужно пользоваться одноимённым методом с большим числом параметров
     */
    public static RetargetingConditionsValidator retConditionsIsValid(
            Set<Long> clientGoalIds,
            List<Goal> lalSegments,
            Map<Long, Goal> allCryptaGoals,
            Map<Long, Set<Long>> mutuallyExclusiveGoals,
            boolean skipGoalExistenceCheck
    ) {
        return retConditionsIsValid(clientGoalIds, lalSegments, allCryptaGoals, emptySet(),
                emptySet(), emptySet(), emptyList(), mutuallyExclusiveGoals, skipGoalExistenceCheck,
                false, false, false);
    }

    private static Constraint<String, Defect> nameDoesNotExist(Set<String> existingNames) {
        return Constraint.fromPredicate((String name) -> !existingNames.contains(name),
                inconsistentStateAlreadyExists());
    }

    private static Constraint<List<Rule>, Defect> parsedRulesDoesNotExist(Set<List<Rule>> parsedExistingRules) {
        return Constraint.fromPredicate(rules -> !parsedExistingRules.contains(rules),
                inconsistentStateAlreadyExists());
    }

    @Override
    public ValidationResult<List<RetargetingCondition>, Defect> apply(
            List<RetargetingCondition> retargetingConditions) {
        Function<RetargetingCondition, String> nameGetter = rc ->
                rc.getType() != null && TYPES_ALLOWED_DUPLICATE_NAME.contains(rc.getType()) ? null : rc.getName();

        Function<RetargetingCondition, String> ruleGetter = rc ->
                rc.getType() != null && TYPES_ALLOWED_DUPLICATE_RULES.contains(rc.getType()) ? null :
                        RetargetingConditionMappings.rulesToJson(rc.getRules());

        return ListValidationBuilder.<RetargetingCondition, Defect>of(retargetingConditions)
                .checkEach(notNull())
                .checkEach(unique(nameGetter), duplicatedObjectWithName())
                .checkEach(unique(ruleGetter), duplicatedObjectWithRules())
                .checkEachBy(this::validateRetCondition,
                        When.notNullAnd(When.valueIs(not(RetargetingCondition::getInterest))))
                .checkEachBy(this::validateRetConditionForInterestTargeting,
                        When.notNullAnd(When.valueIs(RetargetingCondition::getInterest)))
                .checkEachBy(this::validateDuplicatedGoalWithAllAndNotRuleType, When.isTrue(forInternalAd))
                .getResult();
    }

    public ValidationResult<RetargetingCondition, Defect> validateRetConditionForEstimate(
            RetargetingCondition condition) {
        ModelItemValidationBuilder<RetargetingCondition> vb = ModelItemValidationBuilder.of(condition);

        vb.check(conditionIsConsistent());

        vb.item(RetargetingCondition.CLIENT_ID)
                .checkBy(this::validateClientId);

        vb.list(RetargetingCondition.RULES)
                .checkBy(rules -> validateRetConditionRules(rules,
                        condition.getType(), condition.getAutoRetargeting()));

        return vb.getResult();
    }

    private ValidationResult<RetargetingCondition, Defect> validateRetCondition(
            RetargetingCondition condition) {
        ModelItemValidationBuilder<RetargetingCondition> vb = ModelItemValidationBuilder.of(condition);

        vb.check(conditionIsConsistent(), When.isFalse(forInternalAd));

        vb.item(RetargetingCondition.CLIENT_ID)
                .checkBy(this::validateClientId);

        vb.item(RetargetingCondition.TYPE)
                .check(notNull());

        vb.item(RetargetingCondition.NAME)
                .check(notNull())
                .check(notBlank(), When.isTrue(!TYPES_ALLOWED_BLANK_NAME.contains(condition.getType())))
                .check(maxStringLength(MAX_NAME_LENGTH))
                .check(admissibleChars())
                .check(nameDoesNotExist(existingNames),
                        When.isTrue(conditionNotAllowedDuplicateName(condition)));

        vb.item(RetargetingCondition.DESCRIPTION)
                .check(maxStringLength(MAX_DESCRIPTION_LENGTH))
                .check(admissibleChars());

        vb.list(RetargetingCondition.RULES)
                .checkBy(rules -> validateRetConditionRules(rules,
                        condition.getType(), condition.getAutoRetargeting()));

        return vb.getResult();
    }

    public static boolean conditionNotAllowedDuplicateName(RetargetingCondition condition) {
        return !TYPES_ALLOWED_DUPLICATE_NAME.contains(condition.getType());
    }

    private ValidationResult<Rule, Defect> validateRetConditionRule(
            Rule rule, ConditionType type) {
        ModelItemValidationBuilder<Rule> vb = ModelItemValidationBuilder.of(rule);

        vb.item(Rule.TYPE)
                .check(notNull());
        var isForCustomAudience = isCustomAudienceEnabled || isNewCustomAudienceEnabled;

        vb.list(Rule.GOALS)
                .check(notNull())
                .check(collectionSize(MIN_GOALS_PER_RULE, MAX_GOALS_PER_RULE))
                .checkEach(notNull())
                .checkEachBy(this::validateRetConditionGoal, When.notNull())
                .check(cryptaGoalsAreNotAllowedForThisType(),
                        When.isValidAnd(When.isTrue(type != interests)))
                .check(cryptaGoalsAreNotAllowedForThisCondition(),
                        When.isValidAnd(When.isTrue(!OR.equals(rule.getType()) && !forInternalAd)))
                .check(interestsLimit(isForCustomAudience), When.isValidAnd(When.isTrue(type == interests)))
                .check(cryptaGoalsHaveSameType(), When.isValidAnd(When.isFalse(forInternalAd || isForCustomAudience)))
                .check(cryptaGoalsAreNotMixedWithOtherGoals(), When.isValidAnd(When.isFalse(isCustomAudienceEnabled)))
                .check(interestsAreNotAllowed(), When.isValidAnd(When.isTrue(rule.getInterestType() == null)))
                .check(socialDemoGoalsHaveSameParentId(), When.isValid())
                .check(cryptaGoalsDoesNotContainAllPossibleValues(), When.isValid())
                .check(doesNotContainMutuallyExclusiveGoals(), When.isValid())
                .check(unionWithIdGoalIsPresentInRule(), When.isValid())
                .check(isNotNegativeIfBehaviorsArePresent(),
                        When.isValidAnd(When.isTrue(NOT.equals(rule.getType()) && !forInternalAd)));

        return vb.getResult();
    }

    /**
     * Проверяем, что поле union_with_id указано только для целей с правильным типом:
     * пока что только LAL_SEGMENT
     */
    private Constraint<Goal, Defect> goalTypeIsAllowedForUnionWithId() {
        return Constraint.fromPredicate(goal -> goal.getUnionWithId() == null || goal.getType() == LAL_SEGMENT,
                retargetingConditionIsInvalidForRetargeting());
    }

    /**
     * Проверяем, что в поле union_with_id для lal-сегментов лежит его родитель (union_with_id == parent_id)
     */
    private Constraint<Goal, Defect> unionWithIdEqualsToParentIdForLalSegments() {
        return Constraint.fromPredicate(
                goal -> goal.getUnionWithId() == null || goal.getType() != LAL_SEGMENT ||
                        goal.getUnionWithId().equals(lalSegmentIdToParentId.get(goal.getId())),
                retargetingConditionIsInvalidForRetargeting());
    }

    /**
     * Проверяем, что в правиле присутствуют все цели, на которые ссылаются другие цели через поле union_with_id
     */
    private Constraint<List<Goal>, Defect> unionWithIdGoalIsPresentInRule() {
        return Constraint.fromPredicate(
                goals -> {
                    Set<Long> goalIds = listToSet(goals, Goal::getId);
                    return goals.stream()
                            .noneMatch(goal -> goal.getUnionWithId() != null && !goalIds.contains(goal.getUnionWithId()));
                },
                retargetingConditionIsInvalidForRetargeting());
    }

    /**
     * Проверяет что в списке для отрицания нет целей по поведенческим признакам.
     * <p>
     * Выполняется с условием {@code NOT.equals(rule.getType())}
     */
    private Constraint<List<Goal>, Defect> isNotNegativeIfBehaviorsArePresent() {
        return Constraint.fromPredicate(goals -> goals.stream().noneMatch(goal -> goal.getType() == BEHAVIORS),
                cryptaGoalsAllowedOnlyForInterestsType());
    }

    /**
     * Проверяет что в списке ID целей нет взаимоисключающих целей
     */
    private Constraint<List<Goal>, Defect> doesNotContainMutuallyExclusiveGoals() {
        return goals -> {
            for (Goal goal : goals) {
                Set<Long> excludedGoals = mutuallyExclusiveGoals.getOrDefault(goal.getId(), emptySet());
                Optional<Long> invalidValue = goals.stream()
                        .map(Goal::getId)
                        .filter(excludedGoals::contains)
                        .findFirst();
                if (invalidValue.isPresent()) {
                    return mutuallyExclusiveParameters(
                            path(field(RetargetingCondition.ID.name()), field(goal.getId().toString())),
                            path(field(RetargetingCondition.ID.name()), field(invalidValue.get().toString())));
                }
            }
            return null;
        };
    }

    public ValidationResult<List<Rule>, Defect> validateRetConditionRules(List<Rule> rules,
                                                                          ConditionType type,
                                                                          Boolean isAutoRetargeting) {
        ListValidationBuilder<Rule, Defect> vb =
                ListValidationBuilder.<Rule, Defect>of(rules)
                        .check(notNull())
                        .check(collectionSize(
                                type == ConditionType.interests && !forInternalAd
                                        ? MIN_RULES_PER_INTEREST_CONDITION
                                        : MIN_RULES_PER_CONDITION,
                                MAX_RULES_PER_CONDITION))
                        .check(rulesWithInterestsLimit(), When.isTrue(type == interests && !isCustomAudienceEnabled))
                        .check(parsedRulesDoesNotExist(parsedExistingRules),
                                When.isTrue(conditionNotAllowedDuplicateRules(type, isAutoRetargeting))
                        )
                        .checkEach(notNull())
                        .checkEachBy(rule -> validateRetConditionRule(rule, type), When.notNull())
                        .check(positiveRuleExists(), When.isValidAnd(When.isTrue(type == interests && !forInternalAd)));
        return vb.getResult();
    }

    public static boolean conditionNotAllowedDuplicateRules(ConditionType type, Boolean isAutoRetargeting) {
        return (type == null || !TYPES_ALLOWED_DUPLICATE_RULES.contains(type))
                && !Boolean.TRUE.equals(isAutoRetargeting);
    }

    /**
     * Валидация RetargetingCondition для условия нацеливания по интересам (для таргетинга по интересам)
     */
    private ValidationResult<RetargetingCondition, Defect> validateRetConditionForInterestTargeting(
            RetargetingCondition condition) {
        ModelItemValidationBuilder<RetargetingCondition> vb = ModelItemValidationBuilder.of(condition);

        vb.check(conditionIsConsistent(), When.isFalse(forInternalAd));

        vb.item(RetargetingCondition.CLIENT_ID)
                .checkBy(this::validateClientId);

        vb.item(RetargetingCondition.TYPE)
                .check(notNull())
                .check(isEqual(ConditionType.metrika_goals, CommonDefects.inconsistentState()));

        vb.item(RetargetingCondition.DESCRIPTION)
                .check(isNull());

        vb.list(RetargetingCondition.RULES)
                .checkBy(this::validateRetConditionRulesForInterestTargeting);

        return vb.getResult();
    }

    /**
     * Валидация списка Rule для условия нацеливания по интересам (для таргетинга по интересам)
     */
    private ValidationResult<List<Rule>, Defect> validateRetConditionRulesForInterestTargeting(List<Rule> rules) {
        ListValidationBuilder<Rule, Defect> vb =
                ListValidationBuilder.<Rule, Defect>of(rules)
                        .check(notNull())
                        .check(collectionSize(RULES_PER_CONDITION_FOR_INTEREST_TARGETING,
                                RULES_PER_CONDITION_FOR_INTEREST_TARGETING))
                        .checkEach(notNull())
                        .checkEachBy(this::validateRetConditionRuleForInterestTargeting, When.notNull());
        return vb.getResult();
    }

    /**
     * Валидация Rule для условия нацеливания по интересам (для таргетинга по интересам)
     */
    private ValidationResult<Rule, Defect> validateRetConditionRuleForInterestTargeting(Rule rule) {
        ModelItemValidationBuilder<Rule> vb = ModelItemValidationBuilder.of(rule);

        vb.item(Rule.TYPE)
                .check(notNull())
                .check(isEqual(RuleType.ALL, CommonDefects.inconsistentState()));

        vb.list(Rule.GOALS)
                .check(notNull())
                .check(collectionSize(GOALS_PER_RULE_FOR_INTEREST_TARGETING,
                        GOALS_PER_RULE_FOR_INTEREST_TARGETING))
                .checkEach(notNull())
                .checkEachBy(this::validateRetConditionGoalForInterestTargeting, When.notNull());

        return vb.getResult();
    }

    /**
     * Валидация Goal для условия нацеливания по интересам (для таргетинга по интересам)
     */
    private ValidationResult<Goal, Defect> validateRetConditionGoalForInterestTargeting(Goal goal) {
        ModelItemValidationBuilder<Goal> vb = ModelItemValidationBuilder.of(goal);

        vb.item(Goal.TYPE)
                .check(notNull())
                .check(isEqual(GOAL, CommonDefects.invalidValue()));

        vb.item(Goal.ID)
                .check(notNull())
                .check(validId())
                .check(goalDoesNotExistInClientForTargetingInterests(existingInterestTargetingIds), When.isValid())
                .check(goalExistsInTargetingCategories(targetingCategoryByImportId), When.isValid())
                .check(goalAvailableInTargetingCategories(targetingCategoryByImportId), When.isValid());

        vb.item(Goal.TIME)
                .check(notNull())
                .check(isEqual(INTEREST_LINK_TIME_VALUE, CommonDefects.invalidValue()));

        // в интересах не используется
        vb.item(Goal.UNION_WITH_ID)
                .check(isNull());

        return vb.getResult();
    }

    /**
     * Проверка, что у клиента еще нет такой цели-интереса (в Метрике для rmp_interest)
     */
    private Constraint<Long, Defect> goalDoesNotExistInClientForTargetingInterests(
            Set<Long> existingInterestTargetingIds) {
        return Constraint.fromPredicate((Long goalId) -> !existingInterestTargetingIds.contains(goalId),
                retargetingConditionAlreadyExists());
    }

    /**
     * Проверка, что такая цель-интерес есть в категориях таргетинга
     */
    private Constraint<Long, Defect> goalExistsInTargetingCategories(Map<Long, TargetingCategory> importIdToTargetingCategory) {
        return Constraint.fromPredicate(importIdToTargetingCategory::containsKey, objectNotFound());
    }

    /**
     * Проверка, что цель-интерес можно использовать (доступны только те категории, которые являются листьями
     * (не имеют дочерних категорий)), для условия нацеливания по интересам
     */
    private Constraint<Long, Defect> goalAvailableInTargetingCategories(Map<Long, TargetingCategory> targetingCategoryByImportId) {
        return Constraint.fromPredicate((Long goalId) -> targetingCategoryByImportId.containsKey(goalId)
                        && targetingCategoryByImportId.get(goalId).isAvailable(),
                RetargetingDefects.inconsistentStateTargetingCategoryUnavailable());
    }

    private ValidationResult<Long, Defect> validateClientId(
            Long clientId) {
        ItemValidationBuilder<Long, Defect> vb =
                ItemValidationBuilder.<Long, Defect>of(clientId)
                        .check(notNull())
                        .check(validId());

        return vb.getResult();
    }

    private ValidationResult<Goal, Defect> validateRetConditionGoal(Goal goal) {
        ModelItemValidationBuilder<Goal> vb = ModelItemValidationBuilder.of(goal);

        vb.check(goalTimeIsConsistent());
        vb.check(goalTypeIsAllowedForUnionWithId());
        vb.check(unionWithIdEqualsToParentIdForLalSegments(), When.isFalse(skipGoalExistenceCheck));

        Set<Long> clientGoalIds = new HashSet<>(allCryptaGoals.keySet());
        clientGoalIds.addAll(lalSegmentIds);
        clientGoalIds.addAll(metrikaGoalIds);

        vb.item(Goal.ID)
                .check(notNull())
                .check(validId())
                // для CA мы проверяем наличие целей до того, как они будут сконвертированы в retargetingCondition
                .check(goalExists(clientGoalIds), When.isValidAnd(
                        When.isFalse(skipGoalExistenceCheck || isCustomAudienceEnabled)
                )).check(isNotAnOrphanFamilySegment(orphanFamilySegmentIDs), When.isValid());

        vb.item(Goal.TIME).check(inRange(MIN_GOAL_TIME, MAX_GOAL_TIME),
                When.isTrue(goal.getType() == null || goal.getType().isMetrika()));

        vb.item(Goal.UNION_WITH_ID)
                .check(validId(), When.notNull())
                .check(goalExists(clientGoalIds), When.isValidAnd(When.isFalse(skipGoalExistenceCheck)))
                .check(isNotAnOrphanFamilySegment(orphanFamilySegmentIDs), When.isValid());

        return vb.getResult();
    }

    /**
     * Группа должна содержать не более трех условий по интересам
     */
    private Constraint<List<Rule>, Defect> rulesWithInterestsLimit() {

        return Constraint.fromPredicate(rules -> rules.stream()
                        .filter(rule -> rule.getGoals().stream().anyMatch(goal -> goal.getType() == INTERESTS))
                        .count() <= MAX_INTEREST_RULES_PER_CONDITION,
                RetargetingDefects.interestLimitExceeded());
    }

    /**
     * Цели крипты можно использовать только с типом interests (проверка вызывается с условием type != interests)
     */
    private Constraint<List<Goal>, Defect> cryptaGoalsAreNotAllowedForThisType() {
        return Constraint.fromPredicate(g -> g.stream().noneMatch(goal -> goal.getType().isCrypta()),
                cryptaGoalsAllowedOnlyForInterestsType());
    }

    /**
     * Цели крипты можно использовать только с типом interests (проверка вызывается с условием rule.getType() != OR)
     */
    private Constraint<List<Goal>, Defect> cryptaGoalsAreNotAllowedForThisCondition() {
        return Constraint.fromPredicate(g -> g.stream().noneMatch(goal -> goal.getType().isCrypta()),
                cryptaGoalsAllowedOnlyForOrCondition());
    }

    /**
     * Для интересов должна быть задана длительность (проверка вызывается с условием rule.getInterestType() == null)
     */
    private Constraint<List<Goal>, Defect> interestsAreNotAllowed() {
        return Constraint
                .fromPredicate(g -> g.stream().noneMatch(goal -> goal.getType() == INTERESTS),
                        interestsTypeIsNotSpecified());
    }

    /**
     * Условие должно содержать не более 30 интересов (10 для подхода без Custom Audience)
     */
    private Constraint<List<Goal>, Defect> interestsLimit(boolean isForCustomAudience) {
        var maxCount = isForCustomAudience ? MAX_GOALS_PER_RULE_FOR_CA : MAX_GOALS_PER_INTEREST_RULE;
        return Constraint
                .fromPredicate(g -> g.stream()
                                .filter(goal -> goal.getType() == INTERESTS).count() <= maxCount,
                        CollectionDefects.collectionSizeIsValid(0, maxCount));
    }

    /**
     * Для типа interests не должно быть только условий с отрицанием
     */
    private Constraint<List<Rule>, Defect> positiveRuleExists() {

        return Constraint
                .fromPredicate(rules -> rules.isEmpty()
                                || rules.stream().anyMatch(rule -> !NOT.equals(rule.getType())),
                        allElementsAreNegative());
    }

    /**
     * Цели крипты не должны пересекаться с другими целями
     */
    private Constraint<List<Goal>, Defect> cryptaGoalsAreNotMixedWithOtherGoals() {
        return Constraint
                .fromPredicate(g -> g.stream()
                                .map(goal -> goal.getType().isCrypta())
                                .distinct()
                                .count() <= 1,
                        allGoalsMustBeEitherFromMetrikaOrCrypta());
    }

    /**
     * Все цели крипты должны иметь одинаковый тип
     */
    private Constraint<List<Goal>, Defect> cryptaGoalsHaveSameType() {
        return Constraint
                .fromPredicate(g -> g.stream()
                                .map(GoalBase::getType)
                                .filter(GoalType::isCrypta)
                                .distinct()
                                .count() <= 1,
                        allCryptaGoalsMustHaveSameType());
    }

    /**
     * Все цели крипты типа SocialDemo должны иметь один parent_id (пол не должен пересекаться с возрастом)
     */
    private Constraint<List<Goal>, Defect> socialDemoGoalsHaveSameParentId() {
        return Constraint
                .fromPredicate(g -> g.stream()
                                .filter(goal -> goal.getType() == SOCIAL_DEMO || goal.getType() == FAMILY)
                                .map(Goal::getId)
                                .map(allCryptaGoals::get)
                                .filter(Objects::nonNull)
                                .map(Goal::getParentId)
                                .distinct()
                                .count() <= 1,
                        mustHaveSameParentId());
    }

    /**
     * Некоторые коллекции целей крипты не должны содержать все значения с одинаковым parentId.
     * Например, мужской и женский пол.
     * Так как это сужает возможную аудиторию из-за наличия неразмеченных людей.
     */
    private Constraint<List<Goal>, Defect> cryptaGoalsDoesNotContainAllPossibleValues() {
        return Constraint.fromPredicate(goals -> {
            // В первоначальном списке есть только идентификаторы, поэтому сначала получаем данные из allCryptaGoals
            List<Goal> cryptaGoals = goals.stream()
                    .filter(goal -> goal.getType().isCrypta())
                    .map(Goal::getId)
                    .map(allCryptaGoals::get)
                    .filter(Objects::nonNull)
                    .collect(toList());

            if (cryptaGoals.isEmpty()) {
                return true;
            }

            Long parentId = cryptaGoals.get(0).getParentId();

            //Валидироваться должны только несколько групп
            if (!CRYPTA_PARENT_IDS_FOR_VALIDATION.contains(parentId)) {
                return true;
            }

            Set<Goal> allGoalsByParent = allCryptaGoals.values().stream()
                    .filter(goal -> parentId.equals(goal.getParentId()))
                    .collect(toSet());

            return !cryptaGoals.containsAll(allGoalsByParent);

        }, mustNotContainAllElements());
    }

    private ValidationResult<RetargetingCondition, Defect> validateDuplicatedGoalWithAllAndNotRuleType(RetargetingCondition condition) {
        return ItemValidationBuilder.<RetargetingCondition, Defect>of(condition)
                .check(Constraint.fromPredicate(
                        rc -> {
                            if (CollectionUtils.isEmpty(rc.getRules())) {
                                return true;
                            }
                            List<Long> duplicated = StreamEx.of(rc.getRules())
                                    .map(Rule::getGoals)
                                    .map(goals -> mapList(goals, Goal::getId))
                                    .flatMap(Collection::stream)
                                    .toList();

                            return duplicated.size() == new HashSet<>(duplicated).size();
                        }, duplicatedGoal()))
                .getResult();
    }

}
