package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.retargeting.container.AllowedRetargetingComponentsInUserProfile;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;

public class RetargetingConditionsWithAdGroupsValidator implements DefaultValidator<List<RetargetingCondition>> {

    private final Map<Long, List<Long>> adGroupIdsByRetConditionIds;
    private final Map<Long, AdGroupWithType> adGroupsWithTypeByAdGroupId;
    private final Map<Long, CampaignWithType> campaignsWithTypeByCampaignId;
    private final boolean retargetingsAllowedForCpmYndxFrontpage;
    private final AllowedRetargetingComponentsInUserProfile allowedComponents;

    public RetargetingConditionsWithAdGroupsValidator(
            Map<Long, List<Long>> adGroupIdsByRetConditionIds,
            Map<Long, AdGroupWithType> adGroupsWithTypeByAdGroupId,
            Map<Long, CampaignWithType> campaignsWithTypeByCampaignId,
            boolean retargetingsAllowedForCpmYndxFrontpage,
            AllowedRetargetingComponentsInUserProfile allowedComponents) {
        this.adGroupIdsByRetConditionIds = adGroupIdsByRetConditionIds;
        this.adGroupsWithTypeByAdGroupId = adGroupsWithTypeByAdGroupId;
        this.campaignsWithTypeByCampaignId = campaignsWithTypeByCampaignId;
        this.retargetingsAllowedForCpmYndxFrontpage = retargetingsAllowedForCpmYndxFrontpage;
        this.allowedComponents = allowedComponents;
    }

    @Override
    public ValidationResult<List<RetargetingCondition>, Defect> apply(
            List<RetargetingCondition> retargetingConditions) {
        return ListValidationBuilder.of(retargetingConditions, Defect.class)
                .checkEachBy(this::validateCondForCpmYndx, When.valueIs(conditionConnectedToYndxFrontpageAdGroupInYndxFrontpageCampaign()))
                .checkEachBy(this::validateCondForTextGroup, When.valueIs(conditionConnectedToTextAdGroup()))
                .getResult();
    }

    private ValidationResult<RetargetingCondition, Defect> validateCondForCpmYndx(RetargetingCondition retCondition) {
        return ModelItemValidationBuilder.of(retCondition)
                .check(RetargetingConstraints.retConditionIsValidForCpmYndxFrontpage(retCondition),
                        When.isFalse(retargetingsAllowedForCpmYndxFrontpage))
                .getResult();
    }

    private ValidationResult<RetargetingCondition, Defect> validateCondForTextGroup(RetargetingCondition retCondition) {
        return ModelItemValidationBuilder.of(retCondition)
                .check(RetargetingConstraints.conditionIsAllowedForTextGroup(retCondition, allowedComponents))
                .getResult();
    }

    private Predicate<RetargetingCondition> conditionConnectedToYndxFrontpageAdGroupInYndxFrontpageCampaign() {
        return conditionConnectedToAdGroupAndCampaignOfTypes(AdGroupType.CPM_YNDX_FRONTPAGE, CampaignType.CPM_YNDX_FRONTPAGE);
    }

    private Predicate<RetargetingCondition> conditionConnectedToTextAdGroup() {
        return conditionConnectedToAdGroupAndCampaignOfTypes(AdGroupType.BASE, CampaignType.TEXT);
    }

    private Predicate<RetargetingCondition> conditionConnectedToAdGroupAndCampaignOfTypes(AdGroupType adGroupType, CampaignType campaignType) {
        return retargetingCondition -> Optional.ofNullable(adGroupIdsByRetConditionIds.get(retargetingCondition.getId()))
                .orElse(emptyList())
                .stream()
                .map(adGroupsWithTypeByAdGroupId::get)
                .filter(adGroup -> adGroup.getType() == adGroupType)
                .map(AdGroupWithType::getCampaignId)
                .map(campaignsWithTypeByCampaignId::get)
                .anyMatch(campaign -> campaign.getType() == campaignType);
    }
}
