package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithPixels;
import ru.yandex.direct.core.entity.banner.service.validation.BannerValidationInfo;
import ru.yandex.direct.core.entity.banner.type.pixels.BannerWithPixelsInterconnectionsValidator;
import ru.yandex.direct.core.entity.banner.type.pixels.BannerWithPixelsValidatorProvider;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Валидация свойств баннеров, неявно зависящих от условий ретаргетинга
 */
@Component
@ParametersAreNonnullByDefault
public class RetargetingConditionsWithAdsValidator {

    private final BannerWithPixelsValidatorProvider bannerWithPixelsValidatorProvider;
    private final FeatureService featureService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignTypedRepository campaignTypedRepository;

    @Autowired
    public RetargetingConditionsWithAdsValidator(BannerWithPixelsValidatorProvider bannerWithPixelsValidatorProvider,
                                                 FeatureService featureService,
                                                 AdGroupRepository adGroupRepository,
                                                 CampaignTypedRepository campaignTypedRepository) {

        this.bannerWithPixelsValidatorProvider = bannerWithPixelsValidatorProvider;
        this.featureService = featureService;
        this.adGroupRepository = adGroupRepository;
        this.campaignTypedRepository = campaignTypedRepository;
    }

    /**
     * @param shard                                 номер шарда
     * @param clientId                              идентификатор клиента
     * @param retargetingConditions                 список условий ретаргетинга, которые надо провалидировать
     * @param retConditionIdsToBannersWithPixelsMap мапа идентфикаторов условий ретаргетинга в список баннеров с
     *                                              пикселями, валидация пикселей которых зависит от данных условий
     * @param skipPixelValidation                   нужно ли валидировать пиксели в рамках общей валидации (иногда они
     *                                              валидируются в другом месте)
     */
    public <T extends RetargetingCondition> ValidationResult<List<T>, Defect> validateInterconnectionsWithAds(
            int shard,
            ClientId clientId,
            List<T> retargetingConditions,
            Map<Long, List<BannerWithPixels>> retConditionIdsToBannersWithPixelsMap,
            boolean skipPixelValidation) {
        BannerWithPixelsInterconnectionsValidator bannerWithPixelsInterconnectionsValidator =
                new BannerWithPixelsInterconnectionsValidator(bannerWithPixelsValidatorProvider,
                        featureService,
                        adGroupRepository,
                        campaignTypedRepository);

        Map<Integer, List<BannerWithPixels>> retConditionIndexToBannersMap = EntryStream.of(retargetingConditions)
                .mapValues(retCondition -> retConditionIdsToBannersWithPixelsMap.get(retCondition.getId()))
                .filterValues(Objects::nonNull)
                .toMap();
        bannerWithPixelsInterconnectionsValidator.processValidation(shard, clientId,
                retConditionIndexToBannersMap, retargetingConditions,
                retConditions -> new BannerValidationInfo(
                        Collections.emptyList(), null, null,
                        new ArrayList<>(retConditions), null), skipPixelValidation);
        return ListValidationBuilder.of(retargetingConditions, Defect.class)
                .checkEachBy((ind, retCondition) -> bannerWithPixelsInterconnectionsValidator
                        .getValidationResultForEntity(retCondition, ind))
                .getResult();
    }
}
