package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.service.CommonSetBidsValidationService;
import ru.yandex.direct.core.entity.bids.validation.SetBidConstraints;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.service.RequestSetBidType;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.badStatusCampaignArchivedOnUpdateBids;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.notFoundShowConditionByParameters;
import static ru.yandex.direct.core.entity.bids.validation.SetBidConstraints.getBidParams;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.autobudgetPriorityNotMatchStrategy;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.autoBudgetPriorityIsNotNullForAutoStrategy;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.isStrategyRequiredUpdatePriceContext;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Service
@ParametersAreNonnullByDefault
public class RetargetingSetBidsValidationService {
    private final CommonSetBidsValidationService commonSetBidsValidationService;
    private final RbacService rbacService;
    private final CampaignRepository campaignRepository;

    @Autowired
    public RetargetingSetBidsValidationService(CommonSetBidsValidationService commonSetBidsValidationService,
                                               RbacService rbacService, CampaignRepository campaignRepository) {
        this.commonSetBidsValidationService = commonSetBidsValidationService;
        this.rbacService = rbacService;
        this.campaignRepository = campaignRepository;
    }

    public ValidationResult<List<SetBidItem>, Defect> compatiblePreValidation(List<SetBidItem> setBids) {
        return ListValidationBuilder.of(setBids, Defect.class)
                .checkBy(commonSetBidsValidationService::preValidate)
                .checkEachBy(SetBidConstraints.positiveIdsValidator())
                .getResult();
    }

    public ValidationResult<List<SetBidItem>, Defect> compatibleValidation(List<SetBidItem> setBids,
                                                                           List<Retargeting> retargetings, Long uid, int shard) {
        Map<Long, Long> campaignIdByRetargetingId = retargetings.stream()
                .collect(toMap(Retargeting::getId, Retargeting::getCampaignId, (id1, id2) -> id1));
        Map<Long, Long> campaignIdByAdGroupId = retargetings.stream()
                .collect(toMap(Retargeting::getAdGroupId, Retargeting::getCampaignId, (id1, id2) -> id1));
        Set<Long> visibleCampaignIds = rbacService.getVisibleCampaigns(uid, campaignIdByRetargetingId.values());
        Map<Long, Campaign> visibleCampaignsById =
                listToMap(campaignRepository.getCampaigns(shard, visibleCampaignIds), Campaign::getId);

        return ListValidationBuilder.of(setBids, Defect.class)
                .checkEachBy(sb -> validateItem(sb,
                        campaignIdByAdGroupId,
                        campaignIdByRetargetingId,
                        visibleCampaignsById))
                .getResult();
    }

    private ValidationResult<SetBidItem, Defect> validateItem(SetBidItem setBidItem,
                                                              Map<Long, Long> campaignIdByAdgroupId,
                                                              Map<Long, Long> campaignIdByRetargetingId,
                                                              Map<Long, Campaign> visibleCampaignsByIds) {
        Long campaignId = getCampaignId(setBidItem, campaignIdByAdgroupId, campaignIdByRetargetingId);
        Campaign campaign = visibleCampaignsByIds.getOrDefault(campaignId, new Campaign());

        ItemValidationBuilder<SetBidItem, Defect> vb = ModelItemValidationBuilder.of(setBidItem)
                .check(fromPredicate(b -> campaignId != null && visibleCampaignsByIds.containsKey(campaignId),
                        notFoundShowConditionByParameters(getBidParams(setBidItem))))
                .check(fromPredicate(mc -> !campaign.getStatusArchived(),
                        badStatusCampaignArchivedOnUpdateBids(Retargeting.CAMPAIGN_ID, campaign.getId())),
                        When.isValid());

        DbStrategy strategy = campaign.getStrategy();

        vb.item(setBidItem.getPriceContext(), Retargeting.PRICE_CONTEXT.name())
                .check(isStrategyRequiredUpdatePriceContext(strategy), When.isValidAnd(When.isTrue(strategy != null)));

        vb.item(setBidItem.getAutobudgetPriority(), Retargeting.AUTOBUDGET_PRIORITY.name())
                .check(autoBudgetPriorityIsNotNullForAutoStrategy(campaign.getStrategy()),
                        autobudgetPriorityNotMatchStrategy(), When.isValidAnd(When.isTrue(strategy != null)));

        return vb.getResult();
    }

    private Long getCampaignId(SetBidItem setBidItem,
                               Map<Long, Long> campaignIdByAdgroupId,
                               Map<Long, Long> campaignIdByRetargetingId) {
        RequestSetBidType requestType = RequestSetBidType.getByBidSelectionCriteria(setBidItem);

        if (requestType == RequestSetBidType.CAMPAIGN_ID) {
            return setBidItem.getCampaignId();
        } else if (requestType == RequestSetBidType.ADGROUP_ID) {
            return campaignIdByAdgroupId.get(setBidItem.getAdGroupId());
        } else if (requestType == RequestSetBidType.ID) {
            return campaignIdByRetargetingId.get(setBidItem.getId());
        } else {
            return null;
        }
    }

}
