package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.validation.AutobudgetValidator;
import ru.yandex.direct.core.entity.bids.validation.PriceValidator;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListConstraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.adgroup.service.complex.text.RestrictedTextAdGroupUpdateOperation.isFakeCpmRetCondIdForTextAdGroup;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_RETARGETINGS_IN_ADGROUP;
import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_RETARGETINGS_IN_CPM_ADGROUP;
import static ru.yandex.direct.core.entity.retargeting.model.ConditionType.interests;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.duplicatedRetargetingConditionIdForAdgroupId;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.maxCollectionSizeAdGroup;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.maxCollectionSizeCpmAdGroup;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.maxCollectionSizeUserProfileInAdGroup;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.mutuallyExclusiveParameters;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.suspendIsAllowed;
import static ru.yandex.direct.utils.CommonUtils.getDuplicatedItemsFlags;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Общая валидация, как для ретаргетингов с существующими группами, так и в рамках комплексного добавления.
 * При комплексном добавлении в ретаргетингах должны быть проставлены фейковые id групп, чтобы линковать
 * ретаргетинги между собой для проверок на дубликаты
 */
public class RetargetingsCommonValidator implements DefaultValidator<List<TargetInterest>> {
    // todo maxlog: необходимо реализовать трансляцию путей в тексте сообщений об ошибках. Сейчас приходится в core
    //  знать названия API
    private static final PathNode.Field API_FIELD_RETARGETING_LIST_ID = field("RetargetingListId");
    private static final PathNode.Field API_FIELD_INTEREST_ID = field("InterestId");
    private static final Set<AdGroupType> CPM_ADGROUPS = ImmutableSet.of(AdGroupType.CPM_BANNER, AdGroupType.CPM_VIDEO,
            AdGroupType.CPM_GEOPRODUCT, AdGroupType.CPM_YNDX_FRONTPAGE, AdGroupType.CPM_GEO_PIN);

    private final Map<Long, RetargetingCondition> ownRetConditionsById;
    private final boolean retargetingConditionsHaveFakeId;
    private final Map<Long, TargetInterest> existingTargetInterests;
    private final Currency workCurrency;
    private final Collection<TargetingCategory> targetingCategories;
    private final Map<Long, AdGroupType> adGroupTypesByIds;
    private final Map<Long, CampaignType> campaignTypesByAdGroupIds;
    private final Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageAdGroupRestrictionsMap;
    private final boolean doNotCheckRegionalCpmYndxFrontpagePrice;

    public RetargetingsCommonValidator(
            Map<Long, RetargetingCondition> ownRetConditionsById,
            boolean retargetingConditionsHaveFakeId,
            Map<Long, TargetInterest> existingTargetInterests, Currency workCurrency,
            Collection<TargetingCategory> targetingCategories,
            Map<Long, AdGroupType> adGroupTypesByIds,
            Map<Long, CampaignType> campaignTypesByAdGroupIds,
            Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageAdGroupRestrictionsMap,
            boolean doNotCheckRegionalCpmYndxFrontpagePrice) {
        this.ownRetConditionsById = ownRetConditionsById;
        this.retargetingConditionsHaveFakeId = retargetingConditionsHaveFakeId;
        this.existingTargetInterests = existingTargetInterests;
        this.workCurrency = workCurrency;
        this.targetingCategories = targetingCategories;
        this.adGroupTypesByIds = adGroupTypesByIds;
        this.campaignTypesByAdGroupIds = campaignTypesByAdGroupIds;
        this.cpmYndxFrontpageAdGroupRestrictionsMap = cpmYndxFrontpageAdGroupRestrictionsMap;
        this.doNotCheckRegionalCpmYndxFrontpagePrice = doNotCheckRegionalCpmYndxFrontpagePrice;
    }

    @Override
    public ValidationResult<List<TargetInterest>, Defect> apply(List<TargetInterest> targetInterests) {
        return ListValidationBuilder.of(targetInterests, Defect.class)
                .checkEach(notNull())
                .checkEach(noMutuallyExclusiveParameters(), When.isValid())
                .checkEachBy(this::validateRetargeting, When.isValid())
                .checkEach(unique(getTargetInterestComparator()),
                        duplicatedRetargetingConditionIdForAdgroupId(), When.isValid())
                .checkEach(checkDuplicatesInTargetInterests(), When.isValid())
                .checkEach(checkRetargetingLimitInAdGroup(), When.isValid())
                .checkEach(checkRetargetingInterestsLimitInAdGroup(), When.isValid())
                .getResult();
    }

    /**
     * Проверка, что в запросе указано только одно из значений: {@code InterestId} или {@code RetargetingConditionId}
     */
    private Constraint<TargetInterest, Defect> noMutuallyExclusiveParameters() {
        return fromPredicate(r -> r.getInterestId() == null || r.getRetargetingConditionId() == null,
                mutuallyExclusiveParameters(path(field(TargetInterest.INTEREST_ID.name())),
                        path(field(TargetInterest.RETARGETING_CONDITION_ID.name()))));
    }

    private ValidationResult<TargetInterest, Defect> validateRetargeting(TargetInterest targetInterest) {
        Long adGroupId = targetInterest.getAdGroupId();
        CampaignType campaignType = campaignTypesByAdGroupIds.get(adGroupId);

        ModelItemValidationBuilder<TargetInterest> v = ModelItemValidationBuilder.of(targetInterest);

        v.check(Constraint.fromPredicate(r -> r.getRetargetingConditionId() != null || r.getInterestId() != null,
                RetargetingDefects.requiredAtLeastOneOfFields(path(API_FIELD_INTEREST_ID),
                        path(API_FIELD_RETARGETING_LIST_ID))));

        // костылик для RestrictedTextAdGroupUpdateOperation
        // todo: write a doc
        Predicate<Long> notFakeRetCondId = id -> !isFakeCpmRetCondIdForTextAdGroup(id, targetInterest);

        v.item(TargetInterest.RETARGETING_CONDITION_ID)
                .check(validId(), When.valueIs((id -> id != null
                        && !retargetingConditionsHaveFakeId
                        && !isFakeCpmRetCondIdForTextAdGroup(id, targetInterest))))
                .check(retargetingConditionFound(), When.isValidAnd(When.valueIs(notFakeRetCondId)))
                .check(retargetingConditionRulesExists(), When.isValidAnd(When.valueIs(notFakeRetCondId)))
                .check(retargetingConditionIsNotNegative(), When.isValidAnd(When.valueIs(notFakeRetCondId)));

        v.item(TargetInterest.INTEREST_ID)
                .check(validId(), When.notNull())
                .check(checkTargetingCategoryExists(), When.isValid())
                .check(checkTargetingCategoryAvailable(), When.isValid());

        if (campaignType == CampaignType.CPM_PRICE) {
            v.item(TargetInterest.PRICE_CONTEXT)
                    .check(isNull());
        } else {
            // Если нам не нужно ограничивать цену региональными ограничениями, то ограничиваем их ценами из валюты.
            // Такая необходимость возникает, например, при копировании медийных кампаний с автобюджетными стратегиями,
            // у которых ставки, выставленные автобюджетом, могут быть существенно ниже региональных ограничений.
            // При обычном создании кампаний такая проблема не возникает.
            // https://st.yandex-team.ru/DIRECT-166071
            v.item(TargetInterest.PRICE_CONTEXT)
                    .checkBy(doNotCheckRegionalCpmYndxFrontpagePrice ?
                            new PriceValidator(
                                    this.workCurrency,
                                    adGroupTypesByIds.get(targetInterest.getAdGroupId()))
                            :
                            new PriceValidator(
                                    this.workCurrency,
                                    adGroupTypesByIds.get(targetInterest.getAdGroupId()),
                                    cpmYndxFrontpageAdGroupRestrictionsMap.get(targetInterest.getAdGroupId()))
                    );
        }

        v.item(TargetInterest.AUTOBUDGET_PRIORITY)
                .checkBy(new AutobudgetValidator(), When.notNull());

        v.item(TargetInterest.AD_GROUP_ID)
                .check(notNull());

        v.item(TargetInterest.IS_SUSPENDED)
                .check(suspendIsAllowed(campaignType), When.isTrue(campaignType != null));

        return v.getResult();
    }

    private Constraint<Long, Defect> retargetingConditionFound() {
        return fromPredicate(ownRetConditionsById::containsKey,
                RetargetingDefects.retargetingConditionNotFoundDetailed());
    }

    private Constraint<Long, Defect> retargetingConditionRulesExists() {
        return fromPredicate(retId -> ownRetConditionsById.get(retId).getRules() != null,
                RetargetingDefects.retargetingConditionIsInvalidForRetargeting());
    }

    private Constraint<Long, Defect> retargetingConditionIsNotNegative() {
        return fromPredicate(retId -> !ownRetConditionsById.get(retId).getNegative(),
                RetargetingDefects.retargetingConditionIsInvalidForRetargeting());
    }

    /**
     * Проверяем, что переданная категория существует в системе
     */
    private Constraint<Long, Defect> checkTargetingCategoryExists() {
        Set<Long> existingCategories = listToSet(targetingCategories, TargetingCategory::getTargetingCategoryId);
        return Constraint.fromPredicate(existingCategories::contains, objectNotFound());
    }

    /**
     * Проверяем, что переданную категорию можно использовать
     */
    private Constraint<Long, Defect> checkTargetingCategoryAvailable() {
        Set<Long> existingCategories = targetingCategories.stream()
                .filter(TargetingCategory::isAvailable)
                .map(TargetingCategory::getTargetingCategoryId)
                .collect(toSet());
        return Constraint.fromPredicate(existingCategories::contains,
                RetargetingDefects.inconsistentStateTargetingCategoryUnavailable());
    }

    private Comparator<TargetInterest> getTargetInterestComparator() {
        return Comparator.comparing(TargetInterest::getAdGroupId)
                .thenComparing(r -> nvl(r.getInterestId(), nvl(r.getRetargetingConditionId(), 0L)));
    }

    /**
     * Проверяем, что среди добавляемых элементов нет дубликатов
     */
    private ListConstraint<TargetInterest, Defect> checkDuplicatesInTargetInterests() {
        return retargetings -> {
            List<TargetInterest> allTargetInterests = new ArrayList<>(retargetings);
            allTargetInterests.addAll(existingTargetInterests.values());

            BitSet retargetingDuplicates = getRetargetingDuplicates(allTargetInterests);

            // из bitSet'а с дубликатами берём лишь те, что относятся к элементам из запроса
            Map<Integer, Defect> result = new HashMap<>();

            retargetingDuplicates.stream()
                    .filter(idx -> idx < retargetings.size())
                    .forEachOrdered(
                            idx -> result.put(idx, RetargetingDefects.retargetingConditionAlreadyExists()));
            return result;
        };
    }

    /**
     * Находит дубликаты в списке {@code targetInterests} по паре ({@code adGroupId}, {@code retargetingConditionID})
     *
     * @param targetInterests список {@link TargetInterest}'ов
     * @return {@link BitSet}, где биты установлены на позициях элементов с дубликатами
     */
    private BitSet getRetargetingDuplicates(List<TargetInterest> targetInterests) {
        return getDuplicatedItemsFlags(targetInterests, getTargetInterestComparator());
    }

    /**
     * Проверка количества bids_retargeting для одной adGroup
     */
    private ListConstraint<TargetInterest, Defect> checkRetargetingLimitInAdGroup() {
        return retargetings -> {
            Map<Long, Long> adGroupCardinalities =
                    StreamEx.of(retargetings)
                            .append(existingTargetInterests.values())
                            .groupingBy(TargetInterest::getAdGroupId, Collectors.counting());

            Map<Integer, Defect> res = new HashMap<>();

            for (int i = 0; i < retargetings.size(); i++) {
                boolean isCpmAdGroup = CPM_ADGROUPS.contains(adGroupTypesByIds.get(retargetings.get(i).getAdGroupId()));
                int limit = isCpmAdGroup ? MAX_RETARGETINGS_IN_CPM_ADGROUP : MAX_RETARGETINGS_IN_ADGROUP;
                if (adGroupCardinalities.get(retargetings.get(i).getAdGroupId()) > limit) {
                    res.put(i, isCpmAdGroup ? maxCollectionSizeCpmAdGroup(limit) : maxCollectionSizeAdGroup(limit));
                }
            }

            return res;
        };
    }

    /**
     * Проверка количества ретаргетингов на профиль пользователя для одной adGroup. Может быть только одна запись
     */
    private ListConstraint<TargetInterest, Defect> checkRetargetingInterestsLimitInAdGroup() {
        return retargetings -> {
            Set<Long> cryptaRetargetingConditionIds = filterAndMapToSet(ownRetConditionsById.values(),
                    c -> c.getType() == interests, RetargetingConditionBase::getId);

            Map<Long, Long> adGroupCardinalities =
                    StreamEx.of(retargetings)
                            .append(existingTargetInterests.values())
                            .filter(t -> cryptaRetargetingConditionIds.contains(t.getRetargetingConditionId()))
                            .groupingBy(TargetInterest::getAdGroupId, Collectors.counting());

            Map<Integer, Defect> res = new HashMap<>();

            for (int i = 0; i < retargetings.size(); i++) {
                if (nvl(adGroupCardinalities.get(retargetings.get(i).getAdGroupId()), 0L) > 1) {
                    res.put(i, maxCollectionSizeUserProfileInAdGroup());
                }
            }

            return res;
        };
    }
}
