package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.retargeting.container.AllowedRetargetingComponentsInUserProfile;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;
import ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionIdCpmPriceValidator;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionsCpmPriceValidationData;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.validation.defects.RightsDefects.noRightsCantWrite;
import static ru.yandex.direct.utils.CommonUtils.getDuplicatedItemsFlags;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

public class RetargetingsWithAdGroupsValidator implements DefaultValidator<List<TargetInterest>> {

    private final Map<Long, AdGroupSimple> adGroupsById;
    private final Map<Long, RetargetingCondition> ownRetConditionsById;
    private final Map<Long, TargetInterest> existingTargetInterests;
    private final Map<Long, CampaignSimple> campaignsById;
    private final Currency workCurrency;
    private final Set<Long> writableCampaigns;
    private final Set<Long> visibleCampaigns;
    private final Collection<TargetingCategory> targetingCategories;
    private final Map<Long, CriterionType> cpmBannerAdGroupsWithCriterionType;
    private final Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageAdGroupRestrictionsMap;
    private final Boolean retargetingsAllowedForCpmYndxFrontpage;
    private final boolean textBannerInterestsRetCondEnabled;
    private final AllowedRetargetingComponentsInUserProfile allowedComponents;
    private final RetargetingConditionsCpmPriceValidationData cpmPriceValidationData;
    private final boolean doNotCheckRegionalCpmYndxFrontpagePrice;

    public RetargetingsWithAdGroupsValidator(
            Map<Long, AdGroupSimple> adGroupsById,
            Map<Long, RetargetingCondition> ownRetConditionsById,
            Map<Long, TargetInterest> existingTargetInterests,
            Map<Long, CampaignSimple> campaignsById,
            Currency workCurrency,
            Set<Long> writableCampaigns,
            Set<Long> visibleCampaigns,
            Collection<TargetingCategory> targetingCategories,
            Map<Long, CriterionType> cpmBannerAdGroupsWithCriterionType,
            Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageAdGroupRestrictionsMap,
            Boolean retargetingsAllowedForCpmYndxFrontpage,
            boolean textBannerInterestsRetCondEnabled,
            AllowedRetargetingComponentsInUserProfile allowedComponents,
            RetargetingConditionsCpmPriceValidationData cpmPriceValidationData,
            boolean doNotCheckRegionalCpmYndxFrontpagePrice) {
        this.adGroupsById = adGroupsById;
        this.ownRetConditionsById = ownRetConditionsById;
        this.existingTargetInterests = existingTargetInterests;
        this.campaignsById = campaignsById;
        this.workCurrency = workCurrency;
        this.writableCampaigns = writableCampaigns;
        this.visibleCampaigns = visibleCampaigns;
        this.targetingCategories = targetingCategories;
        this.cpmBannerAdGroupsWithCriterionType = cpmBannerAdGroupsWithCriterionType;
        this.retargetingsAllowedForCpmYndxFrontpage = retargetingsAllowedForCpmYndxFrontpage;
        this.cpmYndxFrontpageAdGroupRestrictionsMap = cpmYndxFrontpageAdGroupRestrictionsMap;
        this.textBannerInterestsRetCondEnabled = textBannerInterestsRetCondEnabled;
        this.allowedComponents = allowedComponents;
        this.cpmPriceValidationData = cpmPriceValidationData;
        this.doNotCheckRegionalCpmYndxFrontpagePrice = doNotCheckRegionalCpmYndxFrontpagePrice;
    }

    @Override
    public ValidationResult<List<TargetInterest>, Defect> apply(List<TargetInterest> targetInterests) {

        // генерирует исключение, если в существующих данных есть дубликаты
        checkDuplicatesInExistingData(existingTargetInterests);

        Map<Long, AdGroupType> adGroupTypesByIds =
                listToMap(adGroupsById.values(), AdGroupSimple::getId, AdGroupSimple::getType);
        Map<Long, CampaignType> campaignTypesByAdGroupIds =
                listToMap(adGroupsById.values(), AdGroupSimple::getId, adGroup -> campaignsById.get(adGroup.getCampaignId()).getType());

        RetargetingsCommonValidator retargetingsCommonValidator =
                new RetargetingsCommonValidator(ownRetConditionsById, false,
                        existingTargetInterests, workCurrency, targetingCategories, adGroupTypesByIds,
                        campaignTypesByAdGroupIds, cpmYndxFrontpageAdGroupRestrictionsMap,
                        doNotCheckRegionalCpmYndxFrontpagePrice);

        return ListValidationBuilder.of(targetInterests, Defect.class)
                .checkBy(retargetingsCommonValidator)
                .checkEachBy(this::validateAdGroup, When.isValid())
                .getResult();
    }

    /**
     * @param existingTargetInterests существующие в системе TargetInterest'ы
     * @throws IllegalStateException если для существующих TargetInterest'ов уже существуют дубликаты по retCondId + adGroupId
     */
    private void checkDuplicatesInExistingData(Map<Long, TargetInterest> existingTargetInterests) {
        ArrayList<TargetInterest> targetInterests = new ArrayList<>(existingTargetInterests.values());
        BitSet duplicatedItemsFlags = getRetargetingDuplicates(targetInterests);
        if (!duplicatedItemsFlags.isEmpty()) {
            TargetInterest firstDuplicatedItem = targetInterests.get(duplicatedItemsFlags.nextSetBit(0));
            throw new IllegalStateException(String.format("ret_cond_id %s in adgroup %s is already duplicated",
                    firstDuplicatedItem.getRetargetingConditionId(), firstDuplicatedItem.getAdGroupId()));
        }
    }

    /**
     * Находит дубликаты в списке {@code targetInterests} по паре ({@code adGroupId}, {@code retargetingConditionID})
     *
     * @param targetInterests список {@link TargetInterest}'ов
     * @return {@link BitSet}, где биты установлены на позициях элементов с дубликатами
     */
    private BitSet getRetargetingDuplicates(List<TargetInterest> targetInterests) {
        return getDuplicatedItemsFlags(targetInterests, getTargetInterestComparator());
    }

    private Comparator<TargetInterest> getTargetInterestComparator() {
        return Comparator.comparing(TargetInterest::getAdGroupId)
                .thenComparing(r -> nvl(r.getInterestId(), nvl(r.getRetargetingConditionId(), 0L)));
    }

    private ValidationResult<TargetInterest, Defect> validateAdGroup(TargetInterest targetInterest) {
        ModelItemValidationBuilder<TargetInterest> v = ModelItemValidationBuilder.of(targetInterest);

        AdGroupSimple adGroupSimple = adGroupsById.get(targetInterest.getAdGroupId());
        AdGroupType adGroupType = ifNotNull(adGroupSimple, AdGroupSimple::getType);
        CampaignType campaignType = ifNotNull(adGroupSimple, adGroup -> campaignsById.get(adGroup.getCampaignId()).getType());
        RetargetingCondition retargetingCondition = ownRetConditionsById.get(targetInterest.getRetargetingConditionId());

        v.item(TargetInterest.AD_GROUP_ID)
                .check(notNull())
                .check(validId(), When.isValid())
                .check(adGroupExists(targetInterest), When.isValid())
                .check(adGroupVisible(targetInterest), When.isValid())
                .check(writeAccessToCampaign(), When.isValid())
                .check(RetargetingConstraints.adGroupTypeValid(targetInterest, adGroupType, retargetingCondition,
                        textBannerInterestsRetCondEnabled), When.isValid())
                .check(campaignNotArchived(targetInterest), When.isValid())
                .check(adGroupEligible(), When.isTrue(adGroupType == AdGroupType.CPM_BANNER));

        v.item(TargetInterest.RETARGETING_CONDITION_ID)
                .check(notNull(), When.isValidAnd(When.isTrue(adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE)))
                .check(RetargetingConstraints.retConditionIsValidForCpmYndxFrontpage(retargetingCondition),
                        When.isValidAnd(When.isTrue(adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE && campaignType != CampaignType.CPM_PRICE
                                && !retargetingsAllowedForCpmYndxFrontpage)))
                .check(RetargetingConstraints.onlyToDemographics(retargetingCondition),
                        When.isValidAnd(When.isTrue(adGroupType == AdGroupType.CPM_INDOOR)))
                .check(RetargetingConstraints.conditionIsAllowedForTextGroup(retargetingCondition, allowedComponents),
                        When.isValidAnd(When.isTrue(adGroupType == AdGroupType.BASE)))
                .checkBy(new RetargetingConditionIdCpmPriceValidator(cpmPriceValidationData, ownRetConditionsById),
                        When.isValidAnd(When.isTrue(campaignType == CampaignType.CPM_PRICE)));

        return v.getResult();
    }

    private Constraint<Long, Defect> campaignNotArchived(TargetInterest retargeting) {
        return fromPredicate(adGroupId -> !Optional.ofNullable(adGroupId)
                        .map(adGroupsById::get)
                        .map(AdGroupSimple::getCampaignId)
                        .map(campaignsById::get)
                        .map(CampaignSimple::getStatusArchived)
                        .orElse(false),
                RetargetingDefects
                        .badStatusCampaignArchivedOnAdd(getCampaignIdByAdGroupId(retargeting.getAdGroupId())));
    }

    private Constraint<Long, Defect> writeAccessToCampaign() {
        return fromPredicate(adGroupId -> writableCampaigns.contains(getCampaignIdByAdGroupId(adGroupId)),
                noRightsCantWrite());
    }

    private Constraint<Long, Defect> adGroupVisible(TargetInterest retargeting) {
        return fromPredicate(adGroupId -> visibleCampaigns.contains(getCampaignIdByAdGroupId(adGroupId)),
                RetargetingDefects.adGroupNotFound(retargeting.getAdGroupId()));
    }

    private Constraint<Long, Defect> adGroupExists(TargetInterest retargeting) {
        return fromPredicate(adGroupsById::containsKey,
                RetargetingDefects.adGroupNotFound(retargeting.getAdGroupId()));
    }

    private Constraint<Long, Defect> adGroupEligible() {
        return fromPredicate(adGroup -> cpmBannerAdGroupsWithCriterionType.get(adGroup) == CriterionType.USER_PROFILE,
                RetargetingDefects.notEligibleAdGroup());
    }

    private Long getCampaignIdByAdGroupId(Long adGroupId) {
        return Optional.ofNullable(adGroupId)
                .map(adGroupsById::get)
                .map(AdGroupSimple::getCampaignId)
                .orElse(null);
    }
}
