package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithPixels;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.banner.service.validation.BannerValidationInfo;
import ru.yandex.direct.core.entity.banner.type.pixels.BannerWithPixelsInterconnectionsValidator;
import ru.yandex.direct.core.entity.banner.type.pixels.BannerWithPixelsValidatorProvider;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Валидация свойств баннеров, неявно зависящих от ретаргетингов
 */
@Component
@ParametersAreNonnullByDefault
public class RetargetingsWithAdsValidator {

    private final BannerService bannerService;
    private final BannerWithPixelsValidatorProvider bannerWithPixelsValidatorProvider;
    private final FeatureService featureService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignTypedRepository campaignTypedRepository;

    @Autowired
    public RetargetingsWithAdsValidator(BannerService bannerService,
                                        BannerWithPixelsValidatorProvider bannerWithPixelsValidatorProvider,
                                        FeatureService featureService,
                                        AdGroupRepository adGroupRepository,
                                        CampaignTypedRepository campaignTypedRepository) {
        this.bannerService = bannerService;
        this.bannerWithPixelsValidatorProvider = bannerWithPixelsValidatorProvider;
        this.featureService = featureService;
        this.adGroupRepository = adGroupRepository;
        this.campaignTypedRepository = campaignTypedRepository;
    }

    public <T extends Retargeting> ValidationResult<List<T>, Defect> validateInterconnectionsWithAds(
            int shard,
            ClientId clientId,
            List<T> retargetingList,
            Collection<Long> adGroupIds) {
        return validateInterconnectionsWithAds(shard, clientId, retargetingList, adGroupIds, false);
    }

    /**
     * @param shard               номер шарда
     * @param clientId            идентификатор клиента
     * @param retargetingList     список ретаргетингов, которые надо провалидировать
     * @param adGroupIds          список групп объявлений, которые относятся ретаргетингам и содержат баннеры,
     *                            подлежащие валидации
     * @param skipPixelValidation нужно ли валидировать пиксели в рамках общей валидации (иногда они
     *                            валидируются в другом месте)
     */
    public <T extends Retargeting> ValidationResult<List<T>, Defect> validateInterconnectionsWithAds(
            int shard,
            ClientId clientId,
            List<T> retargetingList,
            Collection<Long> adGroupIds,
            boolean skipPixelValidation) {
        Map<Long, List<BannerWithPixels>> bannersByAdgroup =
                bannerService.getBannersWithPixelsByAdGroups(shard, clientId, adGroupIds);

        //мапа индексов ретаргетингов в список баннеров с пикселями, валидация пикселей которых зависит от данных
        // ретаргетингов
        Map<Integer, List<BannerWithPixels>> retargetingIndexToBannersWithPixelsMap = EntryStream.of(retargetingList)
                .mapValues(Retargeting::getAdGroupId)
                .filterValues(Objects::nonNull)
                .mapValues(bannersByAdgroup::get)
                .filterValues(Objects::nonNull)
                .toMap();

        BannerWithPixelsInterconnectionsValidator bannerWithPixelsInterconnectionsValidator =
                new BannerWithPixelsInterconnectionsValidator(
                        bannerWithPixelsValidatorProvider,
                        featureService,
                        adGroupRepository,
                        campaignTypedRepository);

        Function<List<T>, BannerValidationInfo> bannerValidationInfoByRetargetings =
                retargetings -> new BannerValidationInfo(Collections.emptyList(),
                        mapList(retargetings, Retargeting::getRetargetingConditionId));
        bannerWithPixelsInterconnectionsValidator.processValidation(shard, clientId,
                retargetingIndexToBannersWithPixelsMap, retargetingList,
                bannerValidationInfoByRetargetings, skipPixelValidation);
        return ListValidationBuilder.of(retargetingList, Defect.class)
                .checkEachBy((ind, retargeting) -> bannerWithPixelsInterconnectionsValidator
                        .getValidationResultForEntity(retargeting, ind))
                .getResult();
    }
}
