package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.retargeting.container.SwitchRetargeting;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.retargeting.container.SwitchRetargeting.IS_SUSPENDED;
import static ru.yandex.direct.core.entity.retargeting.container.SwitchRetargeting.RET_COND_ID;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.suspendIsAllowed;
import static ru.yandex.direct.utils.CollectionUtils.flatToList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
@ParametersAreNonnullByDefault
public class SwitchRetargetingOnBannersValidationService2 {

    private final RetargetingConditionRepository retConditionRepository;

    private final CampaignRepository campaignRepository;

    private final ShardHelper shardHelper;

    @Autowired
    public SwitchRetargetingOnBannersValidationService2(
            RetargetingConditionRepository retConditionRepository,
            CampaignRepository campaignRepository,
            ShardHelper shardHelper) {
        this.retConditionRepository = retConditionRepository;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
    }


    public ValidationResult<List<SwitchRetargeting>, Defect> validate(
            List<SwitchRetargeting> switchRetargetings,
            ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Set<Long> existingIds = new HashSet<>(retConditionRepository.getExistingIds(shard, clientId,
                mapList(switchRetargetings, SwitchRetargeting::getRetCondId)
        ));
        Map<Long, List<Long>> campaignsIdByRetConditionId = retConditionRepository.getCampaignIds(shard, existingIds);
        List<Long> allCampaignsId = flatToList(campaignsIdByRetConditionId.values());
        Map<Long, CampaignType> campaignsType = campaignRepository.getCampaignsTypeMap(shard, allCampaignsId);

        ListValidationBuilder<SwitchRetargeting, Defect> listValidationBuilder =
                ListValidationBuilder.<SwitchRetargeting, Defect>of(switchRetargetings)
                        .check(notNull())
                        .check(minListSize(1), CollectionDefects.minCollectionSize(1));

        if (!listValidationBuilder.getResult().hasErrors()) {
            listValidationBuilder.checkEachBy(element -> validateSwitchRetargeting(element, existingIds,
                    campaignsIdByRetConditionId, campaignsType), When.isValid());
        }
        listValidationBuilder.checkEach(unique(Comparator.comparing(SwitchRetargeting::getRetCondId)), When.isValid());

        return listValidationBuilder.getResult();
    }

    private ValidationResult<SwitchRetargeting, Defect> validateSwitchRetargeting(
            SwitchRetargeting switchRetargeting,
            Set<Long> existingIds,
            Map<Long, List<Long>> campaignsIdByRetConditionId,
            Map<Long, CampaignType> campaignsType) {
        var retConditionCampaignsType = mapList(
                campaignsIdByRetConditionId.getOrDefault(switchRetargeting.getRetCondId(), emptyList()),
                campaignsType::get
        );

        ModelItemValidationBuilder<SwitchRetargeting> vb = ModelItemValidationBuilder.of(switchRetargeting);
        vb.item(RET_COND_ID)
                .check(notNull())
                .check(validId());

        if (!vb.getResult().hasAnyErrors()) {
            vb.item(RET_COND_ID)
                    .check(inSet(existingIds), CommonDefects.objectNotFound(), When.isValid());
            vb.item(IS_SUSPENDED)
                    .check(notNull())
                    .check(suspendIsAllowed(retConditionCampaignsType), When.isValid());
        }

        return vb.getResult();
    }

}
