package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithPixels;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.repository.LalSegmentRepository;
import ru.yandex.direct.core.entity.metrika.service.MobileGoalsService;
import ru.yandex.direct.core.entity.retargeting.container.AllowedRetargetingComponentsInUserProfile;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingConditionValidationData;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesCache;
import ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService;
import ru.yandex.direct.core.entity.retargeting.service.helper.RetargetingConditionBannerWithPixelsValidationHelper;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionCpmPriceValidator;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionsCpmPriceValidationData;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionsCpmPriceValidationDataFactory;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListConstraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.collections4.MapUtils.isEmpty;
import static ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition.RULES;
import static ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition.calcNegativeByRules;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingConditionsValidator.retConditionsIsValid;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
public class UpdateRetargetingConditionValidationService2 {

    private final RetargetingConditionRepository retConditionRepository;
    private final RetargetingConditionBannerWithPixelsValidationHelper bannerWithPixelsHelper;
    private final RetargetingConditionCryptaSegmentsProvider retargetingConditionCryptaSegmentsProvider;
    private final LalSegmentRepository lalSegmentRepository;
    private final GoalUtilsService goalUtilsService;
    private final RetargetingConditionsWithAdsValidator retargetingConditionsWithAdsValidator;
    private final FeatureService featureService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final RetargetingConditionsCpmPriceValidationDataFactory cpmPriceValidationDataFactory;
    private final RbacService rbacService;
    private final TargetingCategoriesCache targetingCategoriesCache;
    private final MobileGoalsService mobileGoalsService;
    private final NetAcl netAcl;

    @Autowired
    public UpdateRetargetingConditionValidationService2(
            RetargetingConditionRepository retConditionRepository,
            RetargetingConditionBannerWithPixelsValidationHelper bannerWithPixelsHelper,
            RetargetingConditionCryptaSegmentsProvider retargetingConditionCryptaSegmentsProvider,
            LalSegmentRepository lalSegmentRepository,
            GoalUtilsService goalUtilsService,
            RetargetingConditionsWithAdsValidator retargetingConditionsWithAdsValidator,
            FeatureService featureService,
            AdGroupRepository adGroupRepository,
            CampaignRepository campaignRepository,
            TargetingCategoriesCache targetingCategoriesCache,
            RetargetingConditionsCpmPriceValidationDataFactory cpmPriceValidationDataFactory,
            RbacService rbacService,
            MobileGoalsService mobileGoalsService,
            NetAcl netAcl) {
        this.retConditionRepository = retConditionRepository;
        this.bannerWithPixelsHelper = bannerWithPixelsHelper;
        this.retargetingConditionCryptaSegmentsProvider = retargetingConditionCryptaSegmentsProvider;
        this.lalSegmentRepository = lalSegmentRepository;
        this.goalUtilsService = goalUtilsService;
        this.retargetingConditionsWithAdsValidator = retargetingConditionsWithAdsValidator;
        this.featureService = featureService;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.cpmPriceValidationDataFactory = cpmPriceValidationDataFactory;
        this.rbacService = rbacService;
        this.targetingCategoriesCache = targetingCategoriesCache;
        this.mobileGoalsService = mobileGoalsService;
        this.netAcl = netAcl;
    }

    /**
     * Переносит ошибки и предупреждения с одного результата валидации на другой, включая дочерние
     *
     * @param validationFrom результат, с которого берутся ошибки и предупреждения
     * @param validationTo   результат, на который переносятся ошибки и предупреждения
     */
    private static void transferIssuesFromValidationToValidation(ValidationResult<?, Defect> validationFrom,
                                                                 ValidationResult<?, Defect> validationTo) {
        validationTo.getErrors().addAll(validationFrom.getErrors());
        validationTo.getWarnings().addAll(validationFrom.getWarnings());
        validationFrom.getSubResults().forEach((pathNode, subResultFrom) -> {
            ValidationResult<?, Defect> subResultTo =
                    validationTo.getOrCreateSubValidationResult(pathNode, subResultFrom.getValue());
            transferIssuesFromValidationToValidation(subResultFrom, subResultTo);
        });
    }

    public ValidationResult<List<ModelChanges<RetargetingCondition>>, Defect> preValidate(
            List<ModelChanges<RetargetingCondition>> modelChanges,
            ClientId clientId,
            int shard) {
        List<Long> rcIds = mapList(modelChanges, ModelChanges::getId);
        List<Long> existingIds = retConditionRepository.getExistingIds(shard, clientId, rcIds);
        return ListValidationBuilder.<ModelChanges<RetargetingCondition>, Defect>of(modelChanges)
                .checkBy(elements -> preValidateElements(elements, new HashSet<>(existingIds)), When.isValid())
                .getResult();
    }

    private ValidationResult<List<ModelChanges<RetargetingCondition>>, Defect> preValidateElements(
            List<ModelChanges<RetargetingCondition>> modelChangesList, Set<Long> existingIds) {

        return ListValidationBuilder.<ModelChanges<RetargetingCondition>, Defect>of(modelChangesList)
                .checkEachBy(mc -> validateModelChanges(mc, existingIds))
                .checkEach(unique(ModelChanges::getId), RetargetingDefects.duplicatedRetargetingConditionId())
                .getResult();
    }

    private ValidationResult<ModelChanges<RetargetingCondition>, Defect> validateModelChanges(
            ModelChanges<RetargetingCondition> modelChanges,
            Set<Long> existingIds) {
        ItemValidationBuilder<ModelChanges<RetargetingCondition>, Defect> vb =
                ItemValidationBuilder.of(modelChanges);
        vb.item(modelChanges.getId(), "id")
                .check(notNull())
                .check(validId())
                .check(inSet(existingIds), CommonDefects.objectNotFound(), When.isValid());
        return vb.getResult();
    }

    /**
     * Размер списка передаваемых на валидацию условий ретаргетинга должен совпадать
     * с размером списка положительных результатов предварительной валидации.
     *
     * @param preValidation Объект предварительной валидации списка изменений модели {@link ModelChanges}
     * @param retConditions Список условий ретаргетинга, изменения которых прошли предварительную валидацию,
     *                      с уже примененными изменениями
     * @param changedValues измененные значения
     * @param clientId      clientId
     * @param shard         шард
     * @return итоговый результат валидации условий ретаргетинга, порядок элементов которого
     * соответствует порядку изначального списка изменений модели {@link ModelChanges}, полученного от клиента
     */
    public ValidationResult<List<RetargetingCondition>, Defect> validate(
            ValidationResult<List<ModelChanges<RetargetingCondition>>, Defect> preValidation,
            List<RetargetingCondition> retConditions,
            Collection<AppliedChanges<RetargetingCondition>> changedValues,
            ClientId clientId,
            int shard) {
        ValidationResult<List<RetargetingCondition>, Defect> validation =
                convertPreValidationToValidation(preValidation, retConditions);
        return validateUpdateElements(validation, retConditions, changedValues, false, false, clientId, shard);
    }

    public ValidationResult<List<RetargetingCondition>, Defect> validateUpdateElements(
            ValidationResult<List<RetargetingCondition>, Defect> validation,
            List<RetargetingCondition> conditions,
            Collection<AppliedChanges<RetargetingCondition>> changedValues,
            boolean skipPixelValidation,
            boolean skipInterconnectionsWithAdsValidation,
            ClientId clientId,
            int shard) {
        List<RetargetingConditionValidationData> existingDataExceptUpdated =
                getExistingDataExceptUpdated(conditions, clientId, shard);

        Set<String> clientFeatures = featureService.getEnabledForClientId(clientId);
        boolean skipGoalExistenceCheck = clientFeatures.contains(
                FeatureName.SKIP_GOAL_EXISTENCE_FOR_AGENCY.getName());

        if (!skipGoalExistenceCheck) {
            boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                    .map(netAcl::isInternalIp)
                    .orElse(false);
            skipGoalExistenceCheck = requestFromInternalNetwork
                    && !featureService.isEnabledForClientId(clientId, FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED);
        }

        Set<Long> metrikaGoalIds = new HashSet<>();
        if (!skipGoalExistenceCheck) {
            var lalGoalIds = StreamEx.of(conditions)
                    .map(RetargetingCondition::collectGoalsSafe)
                    .flatMap(Collection::stream)
                    .filter(t -> t.getType() == GoalType.LAL_SEGMENT)
                    .map(GoalBase::getId)
                    .nonNull()
                    .toSet();
            List<Goal> lalSegments = lalSegmentRepository.getLalSegmentsByIds(lalGoalIds);
            var goalIds = StreamEx.of(conditions)
                    .map(RetargetingCondition::collectGoalsSafe)
                    .flatMap(Collection::stream)
                    .map(GoalBase::getId)
                    .append(mapList(lalSegments, GoalBase::getParentId))
                    .nonNull()
                    .toSet();
            metrikaGoalIds.addAll(goalUtilsService.getAvailableMetrikaGoalIds(clientId, goalIds));
            List<Long> mobileGoalIds = mapList(mobileGoalsService.getAllAvailableInAppMobileGoals(clientId),
                    GoalBase::getId);
            metrikaGoalIds.addAll(mobileGoalIds);
        }

        boolean isNewCustomAudienceEnabled = clientFeatures.contains(FeatureName.NEW_CUSTOM_AUDIENCE_ENABLED.getName());
        boolean isInternalAd = rbacService.isInternalAdProduct(clientId);
        Map<Long, Goal> allCryptaGoals = retargetingConditionCryptaSegmentsProvider.getAllowedCryptaSegments(
                isInternalAd, isNewCustomAudienceEnabled, conditions
        );

        List<Goal> lalSegments = lalSegmentRepository.getLalSegmentsByParentIds(List.copyOf(metrikaGoalIds));
        Map<Long, Set<Long>> mutuallyExclusiveGoals = goalUtilsService.getMutuallyExclusiveGoalsMap();
        Set<String> existingNames = listToSet(existingDataExceptUpdated, RetargetingConditionValidationData::getName);
        Set<String> existingRules =
                listToSet(existingDataExceptUpdated, RetargetingConditionValidationData::getRulesJson);

        Map<Long, List<Long>> adGroupIdsByRetConditionIds = retConditionRepository.getAdGroupIds(shard,
                mapList(conditions, RetargetingCondition::getId));

        List<Long> adGroupIds = StreamEx.of(adGroupIdsByRetConditionIds.values()).toFlatList(t -> t);
        Map<Long, AdGroupWithType> adGroupsWithType = adGroupRepository.getAdGroupsWithType(shard, clientId,
                adGroupIds);

        List<Long> campaignIds = mapList(adGroupsWithType.values(), AdGroupWithType::getCampaignId);
        Map<Long, CampaignWithType> campaignsWithType = campaignRepository.getCampaignsWithTypeByCampaignIds(shard,
                clientId, campaignIds);

        boolean retargetingsAllowedForCpmYndxFrontpage = clientFeatures.contains(
                FeatureName.CPM_YNDX_FRONTPAGE_PROFILE.getName());

        AllowedRetargetingComponentsInUserProfile allowedComponents = new AllowedRetargetingComponentsInUserProfile()
                .withSocialDemoInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_SOCIAL_DEMO_IN_USER_PROFILE.getName()))
                .withFamilyAndBehaviorsInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_FAMILY_AND_BEHAVIORS_IN_USER_PROFILE.getName()))
                .withAllInterestsInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_ALL_INTERESTS_IN_USER_PROFILE.getName()))
                .withMetrikaInTextEnabled(clientFeatures.contains(
                        FeatureName.TGO_METRIKA_AND_AUDIENCE_IN_USER_PROFILE.getName()))
                .withCustomAudienceEnabled(clientFeatures.contains(
                        FeatureName.CUSTOM_AUDIENCE_ENABLED.getName()
                ))
                .withNewCustomAudienceEnabled(isNewCustomAudienceEnabled);

        RetargetingConditionsWithAdGroupsValidator retargetingConditionsWithAdGroupsValidator =
                new RetargetingConditionsWithAdGroupsValidator(adGroupIdsByRetConditionIds, adGroupsWithType,
                        campaignsWithType,
                        retargetingsAllowedForCpmYndxFrontpage,
                        allowedComponents);

        RetargetingConditionsIndoorValidator retargetingConditionsIndoorValidator =
                new RetargetingConditionsIndoorValidator(adGroupIdsByRetConditionIds, adGroupsWithType);

        RetargetingConditionsCpmPriceValidationData cpmPriceValidationData =
                cpmPriceValidationDataFactory.createForUpdateRetargetingConditions(shard, conditions);

        Map<Long, List<BannerWithPixels>> retargetingConditionIdsToBannersWithPixelsMap = bannerWithPixelsHelper
                .getBannersWithPixelsByRetargetingConditionIds(shard, clientId, adGroupIdsByRetConditionIds,
                        adGroupIds);

        List<InterestLink> existingTargetingByInterests = retConditionRepository.getExistingInterest(shard, clientId);
        Set<Long> existingInterestTargetingIds = listToSet(existingTargetingByInterests, InterestLink::getGoalId);

        return new ListValidationBuilder<>(validation)
                .checkBy(retConditionsIsValid(metrikaGoalIds, lalSegments, allCryptaGoals, existingNames,
                        existingRules,
                        existingInterestTargetingIds, targetingCategoriesCache.getTargetingCategories(),
                        mutuallyExclusiveGoals, skipGoalExistenceCheck, isInternalAd,
                        clientFeatures.contains(FeatureName.CUSTOM_AUDIENCE_ENABLED.getName()),
                        isNewCustomAudienceEnabled),
                        When.isValid())
                .checkEach(retConditionsScopeIsNotChanged(changedValues), When.isValid())
                //проверка на то, что не поломали ничего в баннерах групп объявлений, связанных с данными ретаргетингами
                .checkBy(itemList -> retargetingConditionsWithAdsValidator
                                .validateInterconnectionsWithAds(shard, clientId, itemList,
                                        retargetingConditionIdsToBannersWithPixelsMap, skipPixelValidation),
                        When.isValidAnd(When.isFalse(skipInterconnectionsWithAdsValidation)))
                .checkBy(retargetingConditionsWithAdGroupsValidator, When.isValid())
                .checkBy(retargetingConditionsIndoorValidator, When.isValid())
                .checkEachBy(new RetargetingConditionCpmPriceValidator(cpmPriceValidationData), When.isValid())
                .getResult();
    }

    public ValidationResult<List<RetargetingCondition>, Defect> convertPreValidationToValidation(
            ValidationResult<List<ModelChanges<RetargetingCondition>>, Defect> preValidation,
            List<RetargetingCondition> retConditions) {
        Map<Long, RetargetingCondition> retCondById = listToMap(retConditions, RetargetingCondition::getId);

        Function<ModelChanges<RetargetingCondition>, RetargetingCondition> convertChanges =
                mc -> {
                    RetargetingCondition retargetingCondition = new RetargetingCondition();
                    retargetingCondition.withId(mc.getId());
                    return retCondById.getOrDefault(mc.getId(), retargetingCondition);
                };
        List<RetargetingCondition> values = mapList(preValidation.getValue(), convertChanges);

        ValidationResult<List<RetargetingCondition>, Defect> validation = new ValidationResult<>(values);
        validation.getErrors().addAll(preValidation.getErrors());
        validation.getWarnings().addAll(preValidation.getWarnings());

        preValidation.getSubResults().forEach((pathNode, subResultFrom) -> {
            int idx = ((PathNode.Index) pathNode).getIndex();
            RetargetingCondition newValue = values.get(idx);
            ValidationResult<?, Defect> subResultTo =
                    validation.getOrCreateSubValidationResult(pathNode, newValue);
            transferIssuesFromValidationToValidation(subResultFrom, subResultTo);
        });

        return validation;
    }

    /**
     * @param retConditions список обновляемых условий ретаргетинга
     * @param clientId      clientId
     * @param shard         шард
     * @return данные тех условий ретаргетинга, которые отсутствуют в списке обновляемых
     */
    private List<RetargetingConditionValidationData> getExistingDataExceptUpdated(
            List<RetargetingCondition> retConditions,
            ClientId clientId, int shard) {
        Set<Long> ids = StreamEx.of(retConditions)
                .map(RetargetingCondition::getId)
                .filter(Objects::nonNull)
                .toSet();
        return retConditionRepository.getValidationData(shard, clientId).stream()
                .filter(d -> !ids.contains(d.getId()))
                .collect(toList());
    }

    private static ListConstraint<RetargetingCondition, Defect> retConditionsScopeIsNotChanged(
            Collection<AppliedChanges<RetargetingCondition>> changedValues) {

        Defect scopeChangedDefect = RetargetingDefects.cannotChangeRetargetingScope();

        return rcList -> {
            Map<Integer, Defect> defectMap = new HashMap<>();

            Map<Long, AppliedChanges<RetargetingCondition>> rulesChangesById =
                    changedValues.stream()
                            .filter(a -> a.changed(RULES))
                            .collect(toMap(a -> a.getModel().getId(), a -> a));

            if (isEmpty(rulesChangesById)) {
                // нечего проверять -- свойство RULES не изменялось ни в одном условии ретаргетинга
                return defectMap;
            }

            for (int i = 0; i < rcList.size(); i++) {
                RetargetingCondition rc = rcList.get(i);
                if (rc == null || rc.getId() == null || rc.getRules() == null) {
                    continue;
                }


                AppliedChanges<RetargetingCondition> rcChanges = rulesChangesById.get(rc.getId());
                if (rcChanges == null) {
                    continue;
                }

                List<Rule> oldRules = rcChanges.getOldValue(RULES);
                List<Rule> rules = rc.getRules();
                if (calcNegativeByRules(oldRules) != calcNegativeByRules(rules)) {
                    defectMap.put(i, scopeChangedDefect);
                }
            }

            return defectMap;
        };
    }
}
