package ru.yandex.direct.core.entity.retargeting.service.validation2;

import java.math.BigDecimal;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.validation.AutobudgetValidator2;
import ru.yandex.direct.core.entity.bids.validation.PriceValidator;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.retargeting.model.InterestLink;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionIdCpmPriceValidator;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionsCpmPriceValidationData;
import ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice.RetargetingConditionsCpmPriceValidationDataFactory;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.autoBudgetPriorityIsNotNullForAutoStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.contextPriceNotNullForManualStrategy;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.autobudgetPriorityNotMatchStrategy;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.duplicatedRetargetingId;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.warningAlreadySuspendedRetargeting;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.warningNotSuspendedRetargeting;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.audienceTargetAllowedOnlyInMobileContentTypeCompany;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.audienceTargetInterestExistsAndVisible;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.campaignIsNotArchived;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.onlyToDemographics;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.operatorHasRightsToChange;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.retargetingCampaignIsNotArchivedOnResume;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.retargetingCampaignIsNotArchivedOnSuspend;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.retargetingExistsAndVisible;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.constraint.RetargetingConstraints.suspendIsAllowed;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
public class UpdateRetargetingValidationService {

    private final ClientService clientService;
    private final RetargetingRepository retargetingRepository;
    private final CampaignRepository campaignRepository;
    private final RetargetingConditionRepository retargetingConditionRepository;
    private final RbacService rbacService;
    private final RetargetingsWithAdsValidator retargetingsWithAdsValidator;
    private final RetargetingConditionsCpmPriceValidationDataFactory cpmPriceValidationDataFactory;

    @Autowired
    public UpdateRetargetingValidationService(
            ClientService clientService,
            RetargetingRepository retargetingRepository,
            CampaignRepository campaignRepository,
            RetargetingConditionRepository retargetingConditionRepository,
            RbacService rbacService,
            RetargetingsWithAdsValidator retargetingsWithAdsValidator,
            RetargetingConditionsCpmPriceValidationDataFactory cpmPriceValidationDataFactory) {
        this.clientService = clientService;
        this.retargetingRepository = retargetingRepository;
        this.campaignRepository = campaignRepository;
        this.retargetingConditionRepository = retargetingConditionRepository;
        this.rbacService = rbacService;
        this.retargetingsWithAdsValidator = retargetingsWithAdsValidator;
        this.cpmPriceValidationDataFactory = cpmPriceValidationDataFactory;
    }

    public ValidationResult<List<ModelChanges<Retargeting>>, Defect> preValidate(
            List<ModelChanges<Retargeting>> modelChangesList, ClientId clientId, long operatorUid, int shard) {
        List<Long> retargetingIds = mapList(modelChangesList, ModelChanges::getId);
        List<Retargeting> existingRetargetings =
                retargetingRepository.getRetargetingsByIds(shard, retargetingIds, maxLimited());

        Map<Long, Long> campaignIdByRetargetingIdMap = existingRetargetings.stream()
                .collect(toMap(Retargeting::getId, Retargeting::getCampaignId));
        Set<Long> campaignIds = new HashSet<>(campaignIdByRetargetingIdMap.values());

        Set<Long> visibleCampaigns = rbacService.getVisibleCampaigns(operatorUid, campaignIds);
        Set<Long> writableCampaigns = rbacService.getWritableCampaigns(operatorUid, campaignIds);
        Map<Long, Campaign> campaignsById =
                listToMap(campaignRepository.getCampaigns(shard, campaignIds), Campaign::getId);

        Set<Long> suspendedIds = existingRetargetings.stream()
                .filter(Retargeting::getIsSuspended)
                .map(Retargeting::getId)
                .collect(toSet());

        List<InterestLink> existingTargetsWithInterest = retargetingConditionRepository
                .getExistingInterest(shard, clientId);
        Set<Long> existingRetargetingConditionIdsWithInterest = StreamEx.of(existingTargetsWithInterest)
                .map(InterestLink::getRetargetingConditionId)
                .toSet();

        Map<Long, Boolean> isTargetWithInterestByRetargetingId = StreamEx.of(existingRetargetings)
                .mapToEntry(r -> existingRetargetingConditionIdsWithInterest.contains(r.getRetargetingConditionId()))
                .mapKeys(Retargeting::getId)
                .toMap();

        ListValidationBuilder<ModelChanges<Retargeting>, Defect> lvb =
                ListValidationBuilder.of(modelChangesList);

        lvb.check(notNull())
                .checkEach(unique(ModelChanges::getId), duplicatedRetargetingId())
                .checkEachBy(mc ->
                        preValidateItem(mc,
                                campaignIdByRetargetingIdMap,
                                visibleCampaigns,
                                writableCampaigns,
                                campaignsById,
                                isAudienceTargetWithInterest(mc, isTargetWithInterestByRetargetingId,
                                        existingRetargetingConditionIdsWithInterest),
                                suspendedIds));

        return lvb.getResult();
    }

    /**
     * Проверяем условие ли это нацеливания по интересам.
     *
     * @return true - если такой RetargetingId уже есть в БД и он относится к интересу
     * или если передаем RetargetingConditionId интереса (у которого property='interest').
     */
    private boolean isAudienceTargetWithInterest(ModelChanges<Retargeting> modelChanges,
                                                 Map<Long, Boolean> isTargetWithInterestByRetargetingId,
                                                 Set<Long> existingRetargetingConditionIdsWithInterest) {
        return isTargetWithInterestByRetargetingId.getOrDefault(modelChanges.getId(), false)
                || existingRetargetingConditionIdsWithInterest
                .contains(modelChanges.getPropIfChanged(Retargeting.RETARGETING_CONDITION_ID));
    }

    private ValidationResult<ModelChanges<Retargeting>, Defect> preValidateItem(
            ModelChanges<Retargeting> modelChanges,
            Map<Long, Long> campaignIdByRetargetingIdMap,
            Set<Long> visibleCampaigns,
            Set<Long> writableCampaigns,
            Map<Long, Campaign> campaignsById,
            boolean isAudienceTargetWithInterest,
            Set<Long> suspendedIds) {
        ItemValidationBuilder<ModelChanges<Retargeting>, Defect> vb =
                ItemValidationBuilder.of(modelChanges);

        Long retargetingId = modelChanges.getId();

        Boolean suspendChange = modelChanges.getPropIfChanged(Retargeting.IS_SUSPENDED);

        vb.item(retargetingId, Retargeting.ID.name())
                .check(notNull())
                .check(validId())
                .check(retargetingExistsAndVisible(campaignIdByRetargetingIdMap, visibleCampaigns),
                        When.isValidAnd(When.isFalse(isAudienceTargetWithInterest)))
                .check(audienceTargetInterestExistsAndVisible(campaignIdByRetargetingIdMap, visibleCampaigns),
                        When.isValidAnd(When.isTrue(isAudienceTargetWithInterest)))
                .check(operatorHasRightsToChange(campaignIdByRetargetingIdMap, writableCampaigns), When.isValid())
                .checkBy(archivedValidator(suspendChange, campaignIdByRetargetingIdMap, campaignsById), When.isValid())
                .checkBy(setSameValueSuspendValidator(suspendChange, suspendedIds), When.isValid());

        return vb.getResult();
    }

    /**
     * Проверка ретаргетинов на измненение флага c тем же значением
     */
    private Validator<Long, Defect> setSameValueSuspendValidator(Boolean suspendChange,
                                                                 Set<Long> suspendedIds) {
        return id -> {
            ItemValidationBuilder<Long, Defect> vb = ItemValidationBuilder.of(id);
            if (suspendChange != null && suspendChange) {
                vb.weakCheck(notInSet(suspendedIds), warningAlreadySuspendedRetargeting());
            } else if (suspendChange != null) {
                vb.weakCheck(inSet(suspendedIds), warningNotSuspendedRetargeting());
            }

            return vb.getResult();
        };
    }

    private Validator<Long, Defect> archivedValidator(Boolean suspendChange,
                                                      Map<Long, Long> campaignIdByRetargetingIdMap, Map<Long,
            Campaign> campaignsById) {
        Map<Long, Long> retargetingIdToArchivedCampaignId = EntryStream.of(campaignIdByRetargetingIdMap)
                .mapValues(campaignsById::get)
                .nonNullValues()
                .filterValues(Campaign::getStatusArchived)
                .mapValues(Campaign::getId)
                .toMap();

        return id -> validateIsArchived(id, suspendChange, retargetingIdToArchivedCampaignId);
    }

    private ValidationResult<Long, Defect> validateIsArchived(Long id, Boolean suspendChange,
                                                              Map<Long, Long> retargetingIdToArchivedCampaignId) {
        ItemValidationBuilder<Long, Defect> vb = ItemValidationBuilder.of(id);

        if (suspendChange == null) {
            vb.check(campaignIsNotArchived(retargetingIdToArchivedCampaignId), When.isValid());
        } else if (suspendChange) {
            vb.check(retargetingCampaignIsNotArchivedOnSuspend(retargetingIdToArchivedCampaignId), When.isValid());
        } else {
            vb.check(retargetingCampaignIsNotArchivedOnResume(retargetingIdToArchivedCampaignId), When.isValid());
        }

        return vb.getResult();
    }

    public ValidationResult<List<Retargeting>, Defect> validate(
            ValidationResult<List<Retargeting>, Defect> preValidationResult,
            Map<Long, AdGroupSimple> adGroupsById,
            Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageAdGroupRestrictionsMap,
            ClientId clientId, int shard, boolean skipInterconnectionsWithAdsValidation,
            Map<Long, BigDecimal> pricePackagePriceByAdGroupId) {
        ListValidationBuilder<Retargeting, Defect> lvb =
                new ListValidationBuilder<>(preValidationResult);

        List<Retargeting> preValidatedItems = getValidItems(preValidationResult);

        Set<Long> campaignIds = preValidatedItems.stream()
                .map(Retargeting::getCampaignId)
                .filter(Objects::nonNull)
                .collect(toSet());
        Map<Long, Campaign> campaignsById =
                listToMap(campaignRepository.getCampaigns(shard, campaignIds), Campaign::getId);

        List<Long> retargetingConditionIds = mapList(preValidatedItems, Retargeting::getRetargetingConditionId);
        List<InterestLink> interests =
                retargetingConditionRepository.getInterestByIds(shard, clientId, retargetingConditionIds);
        List<RetargetingCondition> ownRetConditions = retargetingConditionRepository
                .getFromRetargetingConditionsTable(shard, clientId, retargetingConditionIds);

        Set<Long> interestRetargetingConditionIds = listToSet(interests, InterestLink::getRetargetingConditionId);
        Map<Long, RetargetingCondition> retargetingConditionMap =
                listToMap(ownRetConditions, RetargetingCondition::getId);

        Currency currency = clientService.getWorkCurrency(clientId);

        RetargetingConditionsCpmPriceValidationData cpmPriceValidationData = cpmPriceValidationDataFactory
                .createForUpdateRetargetings(shard, preValidatedItems);

        lvb.checkEachBy(retargeting -> itemValidator(retargeting,
                currency, campaignsById, adGroupsById, interestRetargetingConditionIds,
                retargetingConditionMap, cpmYndxFrontpageAdGroupRestrictionsMap, pricePackagePriceByAdGroupId,
                cpmPriceValidationData),
                When.isValid())
                .checkEach(unique(Retargeting::getId), duplicatedObject(), When.isValid());

        //фильтруем группы, чтобы отправлять искать cpm-баннеры только для cpm-групп
        List<Long> cpmBannerAdGroupIds = EntryStream.of(adGroupsById)
                .filterValues(t -> t.getType() == AdGroupType.CPM_BANNER || t.getType() == AdGroupType.CPM_VIDEO
                        || t.getType() == AdGroupType.CPM_GEOPRODUCT || t.getType() == AdGroupType.CPM_YNDX_FRONTPAGE
                        || t.getType() == AdGroupType.CPM_GEO_PIN)
                .values()
                .map(AdGroupSimple::getId)
                .toList();

        //проверка на то, что не поломали ничего в баннерах адгрупп, связанных с данными ретаргетингами
        lvb.checkBy(itemList -> retargetingsWithAdsValidator
                        .validateInterconnectionsWithAds(shard, clientId, itemList, cpmBannerAdGroupIds),
                When.isValidAnd(When.isFalse(skipInterconnectionsWithAdsValidation)));

        return lvb.getResult();
    }


    private ValidationResult<Retargeting, Defect> itemValidator(
            Retargeting retargeting,
            Currency currency,
            Map<Long, Campaign> campaingsById,
            Map<Long, AdGroupSimple> adGroupsById,
            Set<Long> interestRetargetingConditionIds,
            Map<Long, RetargetingCondition> retargetingConditionMap,
            Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions>
                    cpmYndxFrontpageAdGroupInfoMap,
            Map<Long, BigDecimal> pricePackagePriceByAdGroupId,
            RetargetingConditionsCpmPriceValidationData cpmPriceValidationData) {
        ModelItemValidationBuilder<Retargeting> vb = ModelItemValidationBuilder.of(retargeting);

        AdGroupSimple adGroup = adGroupsById.get(retargeting.getAdGroupId());
        checkState(adGroup != null);

        Campaign campaign = campaingsById.get(retargeting.getCampaignId());
        checkState(campaign != null);

        DbStrategy strategy = campaign.getStrategy();
        checkState(strategy != null);

        if (campaign.getType() == CampaignType.CPM_PRICE) {
            BigDecimal pricePackagePrice = pricePackagePriceByAdGroupId.get(adGroup.getId());
            vb.item(Retargeting.PRICE_CONTEXT)
                    .check(notNull())
                    .check(fromPredicate(pc -> pricePackagePrice.compareTo(pc) == 0, invalidValue()), When.isValid());
        } else {
            vb.item(Retargeting.PRICE_CONTEXT)
                    .checkBy(new PriceValidator(currency, adGroup.getType(),
                            cpmYndxFrontpageAdGroupInfoMap.get(adGroup.getId())), When.notNull())
                    .check(contextPriceNotNullForManualStrategy(strategy));
        }

        vb.item(Retargeting.AUTOBUDGET_PRIORITY)
                .checkBy(new AutobudgetValidator2(), When.notNull())
                .check(autoBudgetPriorityIsNotNullForAutoStrategy(strategy), autobudgetPriorityNotMatchStrategy());

        vb.item(Retargeting.IS_SUSPENDED)
                .check(suspendIsAllowed(campaign.getType()), When.isTrue(campaign.getType() != null));

        RetargetingCondition retargetingCondition =
                retargetingConditionMap.get(retargeting.getRetargetingConditionId());
        vb.item(Retargeting.RETARGETING_CONDITION_ID)
                .check(audienceTargetAllowedOnlyInMobileContentTypeCompany(
                        interestRetargetingConditionIds, campaign.getType()),
                        When.isValid())
                .check(onlyToDemographics(retargetingCondition),
                        When.isValidAnd(When.isTrue(adGroup.getType() == AdGroupType.CPM_INDOOR)))
                .checkBy(new RetargetingConditionIdCpmPriceValidator(cpmPriceValidationData, retargetingConditionMap),
                        When.isValidAnd(When.isTrue(campaign.getType() == CampaignType.CPM_PRICE)));

        return vb.getResult();
    }
}
