package ru.yandex.direct.core.entity.retargeting.service.validation2.constraint;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.invalidGoalsForType;

public class ConditionConsistencyConstraint3 implements Constraint<RetargetingCondition, Defect> {

    private static final ConditionConsistencyConstraint3 CONSTRAINT = new ConditionConsistencyConstraint3();

    private static final Set<GoalType> ALLOWED_TYPES_IN_NEGATIVE_CONDITION = Set.of(GoalType.GOAL, GoalType.SEGMENT,
            GoalType.AUDIENCE, GoalType.ECOMMERCE, GoalType.CDP_SEGMENT, GoalType.LAL_SEGMENT, GoalType.MOBILE);

    public static ConditionConsistencyConstraint3 conditionIsConsistent() {
        return CONSTRAINT;
    }

    // Не используем retCondition.getNegative() т.к. он может вызвать NPE, если getRules() содержит правило null
    private static boolean isConditionNegative(RetargetingCondition retCondition) {
        return RetargetingCondition.calcNegativeByRules(
                retCondition.getRules().stream().filter(Objects::nonNull).collect(Collectors.toList()));
    }

    @Override
    public Defect apply(RetargetingCondition retCondition) {
        if (retCondition == null || retCondition.getRules() == null) {
            return null;
        }
        if (isConditionNegative(retCondition) && !isValidNegativeCondition(retCondition)) {
            return invalidGoalsForType();
        }
        return null;
    }

    private boolean isValidNegativeCondition(RetargetingCondition retCondition) {
        // условие невалидно, если присутствует хотя бы одна цель неразрешённого типа
        return StreamEx.of(retCondition.getRules()).nonNull()
                .map(Rule::getGoals).nonNull()
                .flatMap(List::stream).nonNull()
                .map(GoalBase::getType).nonNull()
                .allMatch(ALLOWED_TYPES_IN_NEGATIVE_CONDITION::contains);
    }
}
