package ru.yandex.direct.core.entity.retargeting.service.validation2.constraint;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.LongStreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.validation.BidsDefects;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.retargeting.container.AllowedRetargetingComponentsInUserProfile;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCampaignInfo;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingConditionBase;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.requiredAtLeastOneOfFieldsForManualStrategy;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.archivedCampaignModification;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNoRights;
import static ru.yandex.direct.core.entity.retargeting.model.ConditionType.interests;
import static ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService.isRetargetingConditionAllowedForTextAdGroup;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.audienceTargetAllowedOnlyInMobileContentCampaign;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.audienceTargetNotFound;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.badCampaignStatusArchivedOnResume;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.badCampaignStatusArchivedOnSuspend;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.cantRemoveAudienceTargetFromArchivedCampaign;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.cpmYndxFrontpageRetargetingsNotAllowed;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.invalidRetargetingConditionInUserProfileInTgo;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.notFoundRetargeting;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.retargetingConditionIsInvalidForRetargeting;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.unableToDelete;

public class RetargetingConstraints {

    private static final Set<Long> SEX_OR_AGE_CRYPTA_IDS = LongStreamEx.range(2499000001L, 2499000009L).boxed().toSet();

    private static final Set<CampaignType> SUSPEND_FORBIDDEN_CAMPAIGN_TYPES = Set.of(CampaignType.CPM_PRICE);

    private static final Set<CampaignType> DELETE_FORBIDDEN_CAMPAIGN_TYPES = Set.of(CampaignType.CPM_PRICE);

    public static Constraint<Long, Defect> retargetingExistsAndVisible(Map<Long, Long> cidByRetId,
                                                                       Set<Long> visibleCampaigns) {
        return fromPredicate(id -> cidByRetId.containsKey(id) && visibleCampaigns.contains(cidByRetId.get(id)),
                notFoundRetargeting());
    }

    public static Constraint<Long, Defect> audienceTargetInterestExistsAndVisible(Map<Long, Long> cidByRetId,
                                                                                  Set<Long> visibleCampaigns) {
        return fromPredicate(id -> cidByRetId.containsKey(id) && visibleCampaigns.contains(cidByRetId.get(id)),
                audienceTargetNotFound());
    }

    public static Constraint<Long, Defect> operatorHasRightsToChange(Map<Long, Long> cidByRetId,
                                                                     Set<Long> writableCampaigns) {
        return fromPredicate(id -> writableCampaigns.contains(cidByRetId.get(id)), campaignNoRights());
    }

    public static Constraint<Long, Defect> retargetingCampaignIsNotArchivedOnSuspend(
            Map<Long, Long> retargetingIdToArchivedCampaignId) {
        return id -> {
            Long archivedCampaignId = retargetingIdToArchivedCampaignId.get(id);

            return archivedCampaignId != null ?
                    badCampaignStatusArchivedOnSuspend(archivedCampaignId)
                    : null;
        };
    }

    public static Constraint<Long, Defect> retargetingCampaignIsNotArchivedOnResume(
            Map<Long, Long> retargetingIdToArchivedCampaignId) {
        return id -> {
            Long archivedCampaignId = retargetingIdToArchivedCampaignId.get(id);

            return archivedCampaignId != null ?
                    badCampaignStatusArchivedOnResume(archivedCampaignId)
                    : null;
        };
    }

    /**
     * @param retargetingIdToCampaingInfo - отображение retargetingId -> campaingInfo для архивных кампаний.
     */
    public static Constraint<Long, Defect> nonArchivedCampaignOnDelete(
            Map<Long, RetargetingCampaignInfo> retargetingIdToCampaingInfo) {
        return id -> {
            RetargetingCampaignInfo campaignInfo = retargetingIdToCampaingInfo.get(id);
            return campaignInfo != null && campaignInfo.getCampaignIsArchived()
                    ? cantRemoveAudienceTargetFromArchivedCampaign(campaignInfo.getCampaignId())
                    : null;
        };
    }

    public static Constraint<Long, Defect> campaignIsNotArchived(
            Map<Long, Long> retargetingIdToArchivedCampaignId) {
        return id -> retargetingIdToArchivedCampaignId.containsKey(id) ? archivedCampaignModification() : null;
    }

    public static Constraint<Integer, Defect> autoBudgetPriorityIsNotNullForAutoStrategy(
            DbStrategy strategy) {
        return priority -> priority == null && strategy.isAutoBudget()
                ? new Defect<>(BidsDefects.Ids.PRIORITY_IS_NOT_SET_FOR_AUTO_STRATEGY)
                : null;
    }

    public static Constraint<BigDecimal, Defect> isStrategyRequiredUpdatePriceContext(DbStrategy strategy) {
        boolean canUpdatePriceContext = strategy != null && strategy.isDifferentPlaces()
                && !strategy.isAutoBudget() && !strategy.isNetStop();

        return price -> canUpdatePriceContext && price == null
                ? requiredAtLeastOneOfFieldsForManualStrategy(singletonList(SetBidItem.PRICE_CONTEXT))
                : null;
    }

    public static Constraint<Long, Defect> audienceTargetAllowedOnlyInMobileContentTypeCompany(
            Set<Long> interestRetargetingConditionIds, CampaignType campaignType) {
        return id -> interestRetargetingConditionIds.contains(id) && campaignType != CampaignType.MOBILE_CONTENT
                ? audienceTargetAllowedOnlyInMobileContentCampaign() : null;
    }

    public static Constraint<Long, Defect> adGroupTypeValid(TargetInterest targetInterest,
                                                            AdGroupType adGroupType,
                                                            RetargetingCondition retargetingCondition,
                                                            boolean textBannerInterestsRetCondEnabled) {
        EnumSet<AdGroupType> allowableTypes;
        if (targetInterest.getInterestId() != null) {
            // для интересов пока доступны только РМП
            allowableTypes = EnumSet.of(AdGroupType.MOBILE_CONTENT);
        } else if (retargetingCondition != null && retargetingCondition.getType() == interests) {
            allowableTypes = EnumSet.of(AdGroupType.CPM_BANNER, AdGroupType.CPM_VIDEO, AdGroupType.CPM_OUTDOOR,
                    AdGroupType.CPM_INDOOR, AdGroupType.CPM_AUDIO, AdGroupType.CPM_GEOPRODUCT,
                    AdGroupType.CPM_GEO_PIN, AdGroupType.CPM_YNDX_FRONTPAGE);
            if (textBannerInterestsRetCondEnabled) {
                allowableTypes.add(AdGroupType.BASE);
            }
        } else {
            allowableTypes = EnumSet.of(AdGroupType.BASE, AdGroupType.MOBILE_CONTENT);
        }
        return fromPredicate(adGroupId -> allowableTypes.contains(adGroupType),
                RetargetingDefects.notEligibleAdGroup());
    }

    /**
     * Проверка на то, что в условии список getRules() пустой, вызываем только при выключенной фиче CPM_YNDX_FRONTPAGE_PROFILE
     */
    public static <T> Constraint<T, Defect> retConditionIsValidForCpmYndxFrontpage(
            RetargetingCondition retargetingCondition) {
        return fromPredicate(constraintArg -> retargetingCondition == null || retargetingCondition.getRules().isEmpty(),
                cpmYndxFrontpageRetargetingsNotAllowed());
    }

    private static boolean isDemographicInterest(Long interestId) {
        return SEX_OR_AGE_CRYPTA_IDS.contains(interestId);
    }

    public static boolean onlyToDemographics(List<? extends RetargetingConditionBase> retargetingConditions) {
        if (isEmpty(retargetingConditions)) {
            return true;
        }
        return retargetingConditions.stream()
                .flatMap(retargetingCondition -> retargetingCondition.getRules().stream())
                .flatMap(rule -> rule.getGoals().stream())
                .allMatch(goal -> isDemographicInterest(goal.getId()));
    }

    public static Constraint<Long, Defect> onlyToDemographics(RetargetingCondition retargetingCondition) {
        return fromPredicate(id -> retargetingCondition == null || onlyToDemographics(
                Collections.singletonList(retargetingCondition)),
                retargetingConditionIsInvalidForRetargeting());
    }

    public static <T> Constraint<T, Defect> conditionIsAllowedForTextGroup(
            RetargetingCondition retargetingCondition,
            AllowedRetargetingComponentsInUserProfile allowedComponents) {
        return fromPredicate(id -> retargetingCondition == null || retargetingCondition.getType() != interests
                        || isRetargetingConditionAllowedForTextAdGroup(retargetingCondition, allowedComponents),
                invalidRetargetingConditionInUserProfileInTgo());
    }

    public static Constraint<Boolean, Defect> suspendIsAllowed(CampaignType campaignType) {
        return suspendIsAllowed(singletonList(campaignType));
    }

    public static Constraint<Boolean, Defect> suspendIsAllowed(Collection<CampaignType> campaignTypes) {
        return Constraint.fromPredicate(isSuspended -> {
                    boolean suspendForbidden = campaignTypes.stream()
                            .anyMatch(SUSPEND_FORBIDDEN_CAMPAIGN_TYPES::contains);
                    return !isSuspended || !suspendForbidden;
                },
                invalidValue());
    }

    public static Constraint<Long, Defect> deleteIsAllowed(
            Map<Long, RetargetingCampaignInfo> retargetingsCampaignInfo) {
        return Constraint.fromPredicate(retargetingId -> {
                    var campaignType = retargetingsCampaignInfo.get(retargetingId).getCampaignType();
                    return !DELETE_FORBIDDEN_CAMPAIGN_TYPES.contains(campaignType);
                },
                unableToDelete());
    }
}
