package ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupPriceSales;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.retargeting.container.RetargetingAdGroupInfo;
import ru.yandex.direct.core.entity.retargeting.model.ConditionType;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Component
public class RetargetingConditionsCpmPriceValidationDataFactory {

    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final RetargetingConditionRepository retConditionRepository;
    private final PricePackageService pricePackageService;

    @Autowired
    public RetargetingConditionsCpmPriceValidationDataFactory(
            CampaignTypedRepository campaignTypedRepository,
            CampaignRepository campaignRepository,
            AdGroupRepository adGroupRepository,
            RetargetingConditionRepository retConditionRepository, PricePackageService pricePackageService) {
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.retConditionRepository = retConditionRepository;
        this.pricePackageService = pricePackageService;
    }

    public RetargetingConditionsCpmPriceValidationData createForAddRetargetingsWithNonExistentAdGroups(
            int shard,
            Map<Long, List<RetargetingAdGroupInfo>> retargetingsInfoByRetConditionId) {
        Map<Long, List<Long>> adGroupIdsByRetConditionId = EntryStream.of(retargetingsInfoByRetConditionId)
                .mapValues(retargetingsInfo -> mapList(retargetingsInfo, RetargetingAdGroupInfo::getAdGroupFakeId))
                .toMap();

        Map<Long, AdGroupPriceSales> cpmPriceSalesAdGroups =
                StreamEx.of(retargetingsInfoByRetConditionId.values())
                        .flatMap(Collection::stream)
                        .mapToEntry(RetargetingAdGroupInfo::getAdGroupFakeId, RetargetingAdGroupInfo::getAdGroup)
                        .distinctKeys()
                        .selectValues(AdGroupPriceSales.class)
                        .toMap();

        Map<Long, Long> adGroupsPriority = EntryStream.of(cpmPriceSalesAdGroups)
                .mapValues(AdGroupPriceSales::getPriority)
                .filterValues(Objects::nonNull)
                .toMap();
        Map<Long, Long> adGroupsCampaignId = EntryStream.of(cpmPriceSalesAdGroups)
                .mapValues(AdGroupPriceSales::getCampaignId)
                .toMap();
        Map<Long, AdGroupType> adGroupsAdGroupType =  EntryStream.of(cpmPriceSalesAdGroups)
                .mapValues(AdGroupPriceSales::getType)
                .toMap();
        Map<Long, CampaignType> campaignsType =
                campaignRepository.getCampaignsTypeMap(shard, adGroupsCampaignId.values());
        Map<Long, Long> adGroupsCpmPriceCampaignId = EntryStream.of(adGroupsCampaignId)
                .filterValues(campaignId -> campaignsType.get(campaignId) == CampaignType.CPM_PRICE)
                .toMap();
        Map<Long, CpmPriceCampaign> cpmPriceCampaigns =
                getCpmPriceCampaigns(shard, adGroupsCpmPriceCampaignId.values());
        var pricePackageByCampaignIds =
                pricePackageService.getPricePackageByCampaignIds(shard, adGroupsCampaignId.values());
        var defaultAdGroupRetargetingConditionByCampaignIds = getDefaultAdGroupRetargetingConditionByCampaignIds(shard,
                StreamEx.of(adGroupsCpmPriceCampaignId.values()).toSet());

        return new RetargetingConditionsCpmPriceValidationData(
                cpmPriceCampaigns,
                pricePackageByCampaignIds, defaultAdGroupRetargetingConditionByCampaignIds, adGroupsCpmPriceCampaignId,
                adGroupsPriority,
                adGroupIdsByRetConditionId,
                adGroupsAdGroupType
        );
    }

    public RetargetingConditionsCpmPriceValidationData createForAddRetargetingsWithExistentAdGroups(
            int shard,
            Collection<TargetInterest> targetInterests) {
        Map<Long, List<Long>> adGroupIdsByRetConditionId = StreamEx.of(targetInterests)
                .mapToEntry(Retargeting::getRetargetingConditionId, Retargeting::getAdGroupId)
                .filterKeys(Objects::nonNull)
                .filterValues(Objects::nonNull)
                .grouping();

        return createForExistentAdGroups(shard, adGroupIdsByRetConditionId);
    }

    public RetargetingConditionsCpmPriceValidationData createForUpdateRetargetings(
            int shard,
            Collection<Retargeting> retargetings) {
        Map<Long, List<Long>> adGroupIdsByRetConditionId = StreamEx.of(retargetings)
                .mapToEntry(Retargeting::getRetargetingConditionId, Retargeting::getAdGroupId)
                .filterKeys(Objects::nonNull)
                .filterValues(Objects::nonNull)
                .grouping();

        return createForExistentAdGroups(shard, adGroupIdsByRetConditionId);
    }

    public RetargetingConditionsCpmPriceValidationData createForUpdateRetargetingConditions(
            int shard,
            Collection<RetargetingCondition> retargetingConditions) {
        Map<Long, List<Long>> adGroupIdsByRetConditionId = retConditionRepository.getAdGroupIds(shard,
                mapList(retargetingConditions, RetargetingCondition::getId));

        return createForExistentAdGroups(shard, adGroupIdsByRetConditionId);
    }

    private RetargetingConditionsCpmPriceValidationData createForExistentAdGroups(
            int shard,
            Map<Long, List<Long>> adGroupIdsByRetConditionId) {
        Set<Long> adGroupsId = adGroupIdsByRetConditionId.values().stream()
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

        Map<Long, Long> adGroupsPriority = adGroupRepository.getAdGroupsPriority(shard, adGroupsId);
        Map<Long, AdGroupType> adGroupsAdGroupType = adGroupRepository.getAdGroupTypesByIds(shard, adGroupsId);

        Map<Long, CampaignSimple> adGroupsCampaign =
                campaignRepository.getCampaignsSimpleByAdGroupIds(shard, adGroupsId);
        Map<Long, Long> adGroupsCpmPriceCampaignId = EntryStream.of(adGroupsCampaign)
                .filterValues(campaign -> campaign.getType() == CampaignType.CPM_PRICE)
                .mapValues(CampaignSimple::getId)
                .toMap();

        Map<Long, CpmPriceCampaign> cpmPriceCampaigns =
                getCpmPriceCampaigns(shard, adGroupsCpmPriceCampaignId.values());

        var pricePackageByCampaignIds = pricePackageService.getPricePackageByCampaignIds(shard,
                StreamEx.of(adGroupsCpmPriceCampaignId.values()).toList());
        var defaultAdGroupRetargetingConditionByCampaignIds = getDefaultAdGroupRetargetingConditionByCampaignIds(shard,
                StreamEx.of(adGroupsCpmPriceCampaignId.values()).toSet());

        return new RetargetingConditionsCpmPriceValidationData(
                cpmPriceCampaigns,
                pricePackageByCampaignIds, defaultAdGroupRetargetingConditionByCampaignIds, adGroupsCpmPriceCampaignId,
                adGroupsPriority,
                adGroupIdsByRetConditionId,
                adGroupsAdGroupType
        );
    }

    @SuppressWarnings("unchecked")
    private Map<Long, CpmPriceCampaign> getCpmPriceCampaigns(int shard, Collection<Long> cpmPriceCampaignsId) {
        return (Map<Long, CpmPriceCampaign>) campaignTypedRepository.getTypedCampaignsMap(shard, cpmPriceCampaignsId);
    }

    private Map<Long, RetargetingCondition> getDefaultAdGroupRetargetingConditionByCampaignIds(
            int shard, Set<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Collections.emptyMap();
        }
        var defAdGroupIdByCampaignId = adGroupRepository.getDefaultPriceSalesAdGroupIdByCampaignId(shard, campaignIds);
        if (defAdGroupIdByCampaignId.isEmpty()) {
            return Collections.emptyMap();
        }
        var retCondByAdGroupId = retConditionRepository.getRetConditionsByAdGroupIds(
                shard, defAdGroupIdByCampaignId.values());
        return StreamEx.of(campaignIds)
                .mapToEntry(Function.identity(),
                        cid -> {
                            Long pid = defAdGroupIdByCampaignId.get(cid);
                            var retCondList = retCondByAdGroupId.getOrDefault(pid, emptyList());
                            return StreamEx.of(retCondList)
                                    .filter(cond -> !cond.getDeleted() && cond.getType() == ConditionType.interests)
                                    .findAny().orElse(null);
                        })
                .filterValues(Objects::nonNull)
                .toMap();
    }
}
