package ru.yandex.direct.core.entity.retargeting.service.validation2.cpmprice;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isDefaultPriority;
import static ru.yandex.direct.core.entity.adgroup.service.AdGroupCpmPriceUtils.isSpecificPriority;
import static ru.yandex.direct.core.entity.adgroup.service.validation.types.PriceSalesAdGroupContentCategoriesValidator.CONTENT_GENRE_AND_CATEGORY;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.retargetingConditionIsInvalidByDefaultAdGroup;
import static ru.yandex.direct.core.entity.retargeting.service.validation2.RetargetingDefects.retargetingConditionIsInvalidForPricePackage;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@ParametersAreNonnullByDefault
public abstract class RetargetingConditionsCpmPriceValidatorBase<T> implements DefaultValidator<T> {
    private final RetargetingConditionsCpmPriceValidationData data;

    RetargetingConditionsCpmPriceValidatorBase(
            RetargetingConditionsCpmPriceValidationData data) {
        this.data = data;
    }

    @Override
    public final ValidationResult<T, Defect> apply(T objectToValidate) {
        RetargetingCondition condition = getRetargetingConditionForValidatingObject(objectToValidate);
        checkNotNull(condition);

        var vb = ItemValidationBuilder.<T, Defect>of(objectToValidate);

        var adGroupIds = data.getAdGroupIdsForRetargetingCondition(condition.getId());
        for (long adGroupId : adGroupIds) {
            CpmPriceCampaign campaign = data.getCpmPriceCampaignForAdGroup(adGroupId);
            // группа не принадлежит прайсовой кампании - пропускаем
            if (campaign == null) {
                continue;
            }
            var pricePackage = data.getPricePackage(campaign.getId());
            long adGroupPriority = data.getAdGroupPriority(adGroupId);
            AdGroupType adGroupType = data.getAdGroupType(adGroupId);
            var conditionDefaultAdGroup = data.getDefaultAdGroupRetargetingCondition(campaign.getId());
            vb.checkBy(ignored -> validateFor(condition, campaign, pricePackage, conditionDefaultAdGroup,
                    adGroupPriority, adGroupType, objectToValidate),
                    When.isValid());
        }

        return vb.getResult();
    }

    abstract RetargetingCondition getRetargetingConditionForValidatingObject(T objectToValidate);

    private ValidationResult<T, Defect> validateFor(RetargetingCondition condition,
                                                    CpmPriceCampaign campaign,
                                                    PricePackage pricePackage,
                                                    RetargetingCondition conditionDefaultAdGroup,
                                                    long adGroupPriority,
                                                    AdGroupType adGroupType,
                                                    T objectToValidate) {
        if (adGroupType == AdGroupType.CPM_VIDEO && !pricePackage.isFrontpageVideoPackage()) {
            return ItemValidationBuilder.<T, Defect>of(objectToValidate)
                    .checkBy(ignored -> validateByPackage(condition, pricePackage, campaign, objectToValidate))
                    .getResult();
        }

        // валидация для adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE
        return ItemValidationBuilder.<T, Defect>of(objectToValidate)
                .checkBy(ignored -> validateByPackage(condition, pricePackage, campaign, objectToValidate),
                        When.isTrue(isDefaultPriority(adGroupPriority)))
                .check(metrikaGoalsNotAllowedforDefaultAdGroup(condition),
                        When.isTrue(isDefaultPriority(adGroupPriority)))
                .checkBy(ignored -> validateForSpecificAdGroup(
                        condition, conditionDefaultAdGroup, objectToValidate),
                        When.isTrue(isSpecificPriority(adGroupPriority)))
                .getResult();
    }

    private ValidationResult<T, Defect> validateByPackage(RetargetingCondition condition,
                                                                PricePackage pricePackage,
                                                                CpmPriceCampaign campaign,
                                                                T objectToValidate) {
        return ItemValidationBuilder.<T, Defect>of(objectToValidate)
                .check(cryptaGoalsAllowedByPackage(condition, pricePackage, campaign))
                .getResult();
    }

    private Constraint<T, Defect> metrikaGoalsNotAllowedforDefaultAdGroup(RetargetingCondition condition) {
        return fromPredicate(
                ignored -> StreamEx.of(condition.collectGoals())
                        .noneMatch(goal -> goal.getType().isMetrika()),
                retargetingConditionIsInvalidForPricePackage());
    }

    private Constraint<T, Defect> cryptaGoalsAllowedByPackage(RetargetingCondition condition,
                                                        PricePackage pricePackage,
                                                        CpmPriceCampaign campaign) {
        var goalIds = StreamEx.of(nvl(condition.collectGoals(), emptyList()))
                //проверяем только крипту без жанров и тематик
                .filter(goal -> goal.getType().isCrypta() && !CONTENT_GENRE_AND_CATEGORY.contains(goal.getType()))
                .map(GoalBase::getId)
                .toSet();
        var fixedGoalIds = extractFixedGoalIds(pricePackage);
        var optionalGoalIds = extractOptionalGoalIds(pricePackage);
        return fromPredicate(ignored -> goalIds.containsAll(fixedGoalIds) && optionalGoalIds.containsAll(goalIds),
                retargetingConditionIsInvalidForPricePackage());
    }

    private Set<Long> extractOptionalGoalIds(PricePackage pricePackage) {
        var set = extractFixedGoalIds(pricePackage);
        var retargetingCondition = pricePackage.getTargetingsCustom().getRetargetingCondition();
        if(retargetingCondition != null && retargetingCondition.getCryptaSegments() != null) {
            set.addAll(retargetingCondition.getCryptaSegments());
        }
        return set;
    }

    private Set<Long> extractFixedGoalIds(PricePackage pricePackage) {
        var set = new HashSet<Long>();
        if (pricePackage.getTargetingsFixed().getCryptaSegments() != null) {
            set.addAll(pricePackage.getTargetingsFixed().getCryptaSegments());
        }
        return StreamEx.of(set)//фильтруем жанры и категории
                .filter(goalId -> !CONTENT_GENRE_AND_CATEGORY.contains(Goal.computeType(goalId)))
                .toSet();
    }

    private ValidationResult<T, Defect> validateForSpecificAdGroup(RetargetingCondition condition,
                                                                   @Nullable RetargetingCondition conditionDefaultAdGroup,
                                                                   T objectToValidate) {
        return ItemValidationBuilder.<T, Defect>of(objectToValidate)
                .check(rulesIsSubsetOfGoals(
                        condition.getRules(),
                        conditionDefaultAdGroup == null ? emptyList() : conditionDefaultAdGroup.getRules()))
                .getResult();
    }

    /**
     * Проверяет, что rules таргетируются на подмножество аудитории goals.
     */
    private Constraint<T, Defect> rulesIsSubsetOfGoals(List<Rule> rules, List<Rule> rulesDefaultAdGroup) {
        return fromPredicate(ignored -> {
            //Не должно быть правил RuleType.NOT
            if (StreamEx.of(rules).anyMatch(it -> it.getType() == RuleType.NOT)) {
                return false;
            }

            //Для каждого OR из дефолтной группы должно быть правило, где цели - подмножество специфичной и нет других
            for (Rule ruleDef : rulesDefaultAdGroup) {
                if (StreamEx.of(rules).noneMatch(
                        rule -> {
                            var ruleDefGoalIds = StreamEx.of(ruleDef.getGoals()).map(Goal::getId).toSet();
                            var ruleGoalIds = StreamEx.of(rule.getGoals()).map(Goal::getId).toSet();
                            return ruleDefGoalIds.containsAll(ruleGoalIds)
                                    && validateByCryptaInterestType(ruleDef.getInterestType(), rule.getInterestType());
                        }
                )) {
                    return false;
                }
            }
            return true;
        }, retargetingConditionIsInvalidByDefaultAdGroup());
    }

    private static boolean validateByCryptaInterestType(@Nullable CryptaInterestType defType, @Nullable CryptaInterestType specificType) {
        return defType == null || defType == CryptaInterestType.all
                || specificType != null && specificType == defType;
    }
}
