package ru.yandex.direct.core.entity.showcondition.container;

import java.math.BigDecimal;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;

/**
 * Контейнер с фиксированными ставками для условий показа, где они не указаны
 * явно, но нужны в текущей стратегии. Используется при автоматическом
 * выставлении недостающих ставок при добавлении/обновлении условий.
 * <p>
 * Можно задать глобальную ставку, или для каждой группы объявлений по
 * отдельности.
 * <p>
 * Также можно указать отсутствие глобальной ставки, если контейнер нужен,
 * а фиксированной ставки нет, см {@link #ofGlobalFixedPrice(BigDecimal)}.
 */
@ParametersAreNonnullByDefault
public class ShowConditionFixedAutoPrices {
    private Map<Long, BigDecimal> perAdGroupFixedPrices;
    private BigDecimal globalFixedPrice;

    /**
     * Создание контейнера с глобальной фиксированной ставкой.
     *
     * @param globalFixedPrice глобальная ставка. Может быть {@code null}, тогда в контейнере не будет фиксированной
     *                         ставки
     */
    public static ShowConditionFixedAutoPrices ofGlobalFixedPrice(@Nullable BigDecimal globalFixedPrice) {
        return new ShowConditionFixedAutoPrices(null, globalFixedPrice);
    }

    /**
     * Создание контейнера с фиксированными ставками для отдельных групп.
     */
    public static ShowConditionFixedAutoPrices ofPerAdGroupFixedPrices(Map<Long, BigDecimal> perAdGroupFixedPrices) {
        return new ShowConditionFixedAutoPrices(perAdGroupFixedPrices, null);
    }

    private ShowConditionFixedAutoPrices(
            @Nullable Map<Long, BigDecimal> perAdGroupFixedPrices,
            @Nullable BigDecimal globalFixedPrice) {
        this.perAdGroupFixedPrices = perAdGroupFixedPrices;
        this.globalFixedPrice = globalFixedPrice;
    }

    public boolean hasAdGroupFixedPrice(Long adGroupId) {
        return perAdGroupFixedPrices != null && perAdGroupFixedPrices.containsKey(adGroupId);
    }


    public BigDecimal getAdGroupFixedPrice(Long adGroupId) {
        checkState(perAdGroupFixedPrices != null && perAdGroupFixedPrices.containsKey(adGroupId),
                "don't have a fixed price for adGroup " + adGroupId);
        return perAdGroupFixedPrices.get(adGroupId);
    }

    public boolean hasGlobalFixedPrice() {
        return globalFixedPrice != null;
    }

    public BigDecimal getGlobalFixedPrice() {
        checkNotNull(globalFixedPrice, "don't have a global fixed price");
        return globalFixedPrice;
    }
}
