package ru.yandex.direct.core.entity.showcondition.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.advq.SearchRequest;
import ru.yandex.direct.advq.search.AdvqRequestKeyword;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.MinusKeywordPreparingTool;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.keyword.model.KeywordWithMinuses;
import ru.yandex.direct.core.entity.minuskeywordspack.model.MinusKeywordsPack;
import ru.yandex.direct.core.entity.minuskeywordspack.repository.MinusKeywordsPackRepository;
import ru.yandex.direct.core.entity.showcondition.model.ShowStatRequest;
import ru.yandex.direct.core.entity.stopword.service.StopWordService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.libs.keywordutils.StopWordMatcher;
import ru.yandex.direct.libs.keywordutils.inclusion.KeywordInclusionUtils;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;

@Service
public class ShowStatConverter {

    private final MinusKeywordPreparingTool minusKeywordPreparingTool;
    private final KeywordWithLemmasFactory keywordWithLemmasFactory;
    private final StopWordMatcher stopWordMatcher;
    private final MinusKeywordsPackRepository minusKeywordsPackRepository;
    private final ShardHelper shardHelper;
    private final AdGroupRepository adGroupRepository;
    private final CampaignService campaignService;

    @Autowired
    public ShowStatConverter(MinusKeywordPreparingTool minusKeywordPreparingTool,
                             KeywordWithLemmasFactory keywordWithLemmasFactory,
                             StopWordService stopWordService,
                             MinusKeywordsPackRepository minusKeywordsPackRepository,
                             ShardHelper shardHelper,
                             AdGroupRepository adGroupRepository,
                             CampaignService campaignService) {
        this.minusKeywordPreparingTool = minusKeywordPreparingTool;
        this.keywordWithLemmasFactory = keywordWithLemmasFactory;
        this.stopWordMatcher = stopWordService::isStopWord;
        this.minusKeywordsPackRepository = minusKeywordsPackRepository;
        this.shardHelper = shardHelper;
        this.adGroupRepository = adGroupRepository;
        this.campaignService = campaignService;
    }

    /**
     * Преобразовать запрос на получение статистики показа нескольких фраз в запрос для Advq.
     * Добавляет к каждой фразе минус-слова, переданные в запросе, а также минус-слова кампании,
     * если в запросе передан идентификатор кампании. При этом минус-фраза не добавляется,
     * если все ее минус-слова входят в ключевую фразу.
     */
    public SearchRequest convertRequest(ShowStatRequest request) {
        List<String> allMinusPhrases = collectMinusPhrases(request.getCommonMinusPhrases(),
                request.getAdGroupId(),
                request.getCampaignId(),
                request.getLibraryMinusPhrasesIds());
        List<String> cleanedMinusPhrases = minusKeywordPreparingTool.removeDuplicatesAndSort(allMinusPhrases);

        List<AdvqRequestKeyword> phrasesWithAllMinusWords = StreamEx.of(request.getKeywords())
                .mapToEntry(identity(), keyword -> KeywordWithMinuses.fromPhrase(keyword.getPhrase()))
                .mapValues(phrase -> {
                    Set<String> intersectingMinusPhrases =
                            KeywordInclusionUtils.getIncludedMinusKeywords(keywordWithLemmasFactory,
                            stopWordMatcher, List.of(phrase.getPlusKeyword()), cleanedMinusPhrases);
                    List<String> nonIntersectingMinusPhrases = new ArrayList<>(cleanedMinusPhrases);
                    nonIntersectingMinusPhrases.removeAll(intersectingMinusPhrases);
                    phrase.addMinusKeywords(nonIntersectingMinusPhrases);
                    return phrase;
                })
                .mapKeyValue((keyword, phrase) -> new AdvqRequestKeyword(phrase.toString().trim(), keyword.getKeywordId()))
                .toList();

        List<Long> geos = emptyList();
        if (StringUtils.isNotBlank(request.getGeo())) {
            geos = StreamEx.split(request.getGeo(), ",")
                    .map(Long::valueOf)
                    .toList();
        }

        return new SearchRequest(phrasesWithAllMinusWords, geos)
                .withDeviceTypes(request.getDeviceTypes());
    }

    /**
     * Собрать минус-фразы из разных источников в один набор для формирования запроса в Advq.
     * <p>
     * В случае, если передан идентификатор группы {@code adGroupId}, то извлекаем:
     * фразы группы, фразы всех библиотек этой группы, фразы кампании этой группы.
     * <p>
     * В случае, если передан идентификатор кампании {@code campaignId}, то извлекаем фразы этой кампании.
     * <p>
     * Поле {@code libraryIds} -- не самостоятельное. Если оно передано, то минус-фразы этих библиотек учитываются
     * только в случае, если передан {@code adGroupId} или {@code campaignId}.
     *
     * @param commonMinusPhrases минус-фразы из запроса
     * @param adGroupId          идентификатор группы
     * @param campaignId         идентфикатор кампании
     * @param libraryIds         идентификаторы библиотек минус-фраз
     * @return все минус-фразы, которые нужно учитывать при формировании запроса в Advq
     */
    private List<String> collectMinusPhrases(List<String> commonMinusPhrases,
                                             Long adGroupId,
                                             Long campaignId,
                                             List<Long> libraryIds) {
        List<String> minusPhrases = new ArrayList<>();
        if (commonMinusPhrases != null) {
            minusPhrases.addAll(commonMinusPhrases);
        }
        if (adGroupId == null && campaignId == null) {
            return minusPhrases;
        }
        ClientId clientId = getClientId(adGroupId, campaignId);
        int shard = shardHelper.getShardByClientId(clientId);
        Long actualCampaignId;
        if (adGroupId == null) {
            actualCampaignId = campaignId;
        } else {
            AdGroup adGroup = adGroupRepository.getAdGroups(shard, List.of(adGroupId)).get(0);
            minusPhrases.addAll(adGroup.getMinusKeywords());
            minusPhrases.addAll(getMinusPhrasesFromLibraries(shard, clientId, adGroup.getLibraryMinusKeywordsIds()));
            actualCampaignId = adGroup.getCampaignId();
        }
        minusPhrases.addAll(campaignService.getMinusKeywordsByCampaignId(shard, actualCampaignId));
        minusPhrases.addAll(getMinusPhrasesFromLibraries(shard, clientId, libraryIds));
        return minusPhrases;
    }

    private List<String> getMinusPhrasesFromLibraries(int shard, ClientId clientId, List<Long> libraryIds) {
        if (libraryIds == null) {
            return emptyList();
        }
        Map<Long, MinusKeywordsPack> libMinusPhrases =
                minusKeywordsPackRepository.getMinusKeywordsPacks(shard, clientId, libraryIds);
        return StreamEx.ofValues(libMinusPhrases).toFlatList(MinusKeywordsPack::getMinusKeywords);
    }

    /**
     * Получить идентфикатор клиента по идентификатору группы или кампании
     * (хотя бы один идентификатор должен быть равен {@code null})
     */
    private ClientId getClientId(Long adGroupId, Long campaignId) {
        long clientId = adGroupId != null
                ? shardHelper.getClientIdByAdGroupId(adGroupId)
                : shardHelper.getClientIdByCampaignId(campaignId);
        return ClientId.fromLong(clientId);
    }
}
