package ru.yandex.direct.core.entity.sitelink.repository;

import java.math.BigInteger;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.BinaryOperator;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.dbschema.ppc.tables.records.SitelinksLinksRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toMap;
import static org.apache.commons.lang3.StringUtils.defaultString;
import static ru.yandex.bolts.function.forhuman.Comparator.naturalComparator;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_LINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_SET_TO_LINK;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8;

@Repository
public class SitelinkRepository {

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final JooqMapperWithSupplier<Sitelink> sitelinkMapper;
    private final List<Field<?>> fieldsToRead;


    @Autowired
    public SitelinkRepository(ShardHelper shardHelper, DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;

        sitelinkMapper = createSitelinkMapper();

        this.fieldsToRead = StreamEx.of(sitelinkMapper.getFieldsToRead())
                .remove(SITELINKS_SET_TO_LINK.ORDER_NUM::equals)
                .toList();
    }

    /**
     * Добавляет сайтлинки, и выставляет ид в модели
     *
     * @param shard     шард для запроса
     * @param sitelinks список сайтлинков для добавления
     */
    public void add(int shard, Collection<Sitelink> sitelinks) {
        sitelinks.stream().filter(sl -> sl.getHash() == null)
                .forEach(sitelink -> sitelink.withHash(calcHash(sitelink)));

        addToSitelinksTable(shard, sitelinks);
    }

    /**
     * Получает id сайтлинка по hash
     *
     * @param shard    шард для запроса
     * @param sitelink сайтлинк
     * @return ид сайтлинка
     */
    public Long getSitelinkId(int shard, Sitelink sitelink) {
        return dslContextProvider.ppc(shard)
                .select(SITELINKS_LINKS.SL_ID)
                .from(SITELINKS_LINKS)
                .where(SITELINKS_LINKS.HASH.eq(ULong.valueOf(sitelink.getHash())))
                .fetchOne(SITELINKS_LINKS.SL_ID);
    }

    /**
     * Получает мапу hash -> sitelink id
     *
     * @param shard     шард для запроса
     * @param sitelinks список сайтлинков
     * @return мапа hash -> sitelink id
     */
    public Map<BigInteger, Long> getSitelinkIdsByHashes(int shard, Collection<Sitelink> sitelinks) {
        List<ULong> sitelinkHashes = mapList(sitelinks, sl -> ULong.valueOf(sl.getHash()));

        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(SITELINKS_LINKS.SL_ID, SITELINKS_LINKS.HASH))
                .from(SITELINKS_LINKS)
                .where(SITELINKS_LINKS.HASH.in(sitelinkHashes))
                .fetch();

        return result.stream().collect(toMap(
                rec -> rec.getValue(SITELINKS_LINKS.HASH).toBigInteger(),
                rec -> rec.getValue(SITELINKS_LINKS.SL_ID),
                BinaryOperator.minBy(naturalComparator())));
    }

    /**
     * Получает сайтлинки по списку идентификаторов
     *
     * @param shard       шард для запроса
     * @param sitelinkIds список id сайтлинков (sl_id)
     * @return список сайтлинков
     */
    public List<Sitelink> get(int shard, Collection<Long> sitelinkIds) {
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(SITELINKS_LINKS)
                .where(SITELINKS_LINKS.SL_ID.in(sitelinkIds))
                .orderBy(SITELINKS_LINKS.SL_ID)
                .fetch()
                .map(this::constructSitelinkFromRecord);
    }

    /**
     * Удаляет сайтлинки по списку id
     *
     * @param shard шард для запроса
     * @param ids   список id сайтлинков (sl_id)
     */
    public void delete(int shard, Collection<Long> ids) {
        dslContextProvider.ppc(shard)
                .deleteFrom(SITELINKS_LINKS)
                .where(SITELINKS_LINKS.SL_ID.in(ids))
                .execute();
    }

    private void addToSitelinksTable(int shard, Collection<Sitelink> sitelinks) {
        if (sitelinks.isEmpty()) {
            return;
        }
        generateSitelinkIds(sitelinks);
        InsertHelper<SitelinksLinksRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), SITELINKS_LINKS);
        sitelinks.forEach(sitelink -> insertHelper
                .add(sitelinkMapper, sitelink)
                .newRecord()
        );
        insertHelper.execute();
    }

    /**
     * Генерирует новые id сайтлинка (sl_id) и выставляет эти id каждому сайтлинку в переданном списке
     *
     * @param sitelinks список сайтлинков, которым нужно сгенерировать новые id (sl_id)
     */
    private void generateSitelinkIds(Collection<Sitelink> sitelinks) {
        List<Long> ids = shardHelper.generateSitelinkIds(sitelinks.size());
        StreamEx.of(sitelinks).zipWith(ids.stream())
                .forKeyValue(Sitelink::setId);
    }

    /**
     * Вычисляет хеш по полям title, href, desc, tl_id и tl_is_disabled
     *
     * @param sitelink сайтлинк
     * @return хеш сайтлинка
     */
    public static BigInteger calcHash(Sitelink sitelink) {
        checkNotNull(sitelink.getTitle(), "sitelink title cannot be null");

        String sitelinkData = sitelink.getTitle()
                + defaultString(sitelink.getHref())
                + defaultString(sitelink.getDescription());
        if (sitelink.getTurboLandingId() != null) {
            sitelinkData = sitelinkData
                    .concat(sitelink.getTurboLandingId().toString());
        } else {
            checkNotNull(sitelink.getHref(), "required one of: sitelink href or turbolanding");
        }

        return getMd5HalfHashUtf8(sitelinkData);
    }

    /**
     * Безопасно вычисляет хеш сайтлинка, возвращает null, если сайтлинк не валиден
     *
     * @param sitelink Сайтлинк, для которого вычисляется хеш
     * @return хеш сайтлинка, или null, если он не может быть вычислен
     */
    @Nullable
    public static BigInteger calcHashSafe(Sitelink sitelink) {
        if (sitelink.getTitle() == null || (sitelink.getHref() == null && sitelink.getTurboLandingId() == null)) {
            return null;
        }

        return calcHash(sitelink);
    }

    public JooqMapperWithSupplier<Sitelink> createSitelinkMapper() {
        return JooqMapperWithSupplierBuilder.builder(Sitelink::new)
                .map(property(Sitelink.ID, SITELINKS_LINKS.SL_ID))
                .map(convertibleProperty(Sitelink.HREF, SITELINKS_LINKS.HREF,
                        href -> href.isEmpty() ? null : href,
                        StringUtils::defaultString))
                .map(property(Sitelink.TITLE, SITELINKS_LINKS.TITLE))
                .map(property(Sitelink.DESCRIPTION, SITELINKS_LINKS.DESCRIPTION))
                .map(convertibleProperty(Sitelink.HASH, SITELINKS_LINKS.HASH,
                        hash -> hash != null ? hash.toBigInteger() : null,
                        hash -> hash != null ? ULong.valueOf(hash) : null))
                .map(property(Sitelink.ORDER_NUM, SITELINKS_SET_TO_LINK.ORDER_NUM))
                .map(property(Sitelink.TURBO_LANDING_ID, SITELINKS_LINKS.TL_ID))
                .build();
    }

    /*
     * Создаем запись
     */
    @Nonnull
    Sitelink constructSitelinkFromRecord(Record record) {
        return sitelinkMapper.fromDb(record);
    }

}
