package ru.yandex.direct.core.entity.sitelink.repository;

import java.math.BigInteger;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.Function;

import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Result;
import org.jooq.Select;
import org.jooq.SelectConditionStep;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.GuavaCollectors;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.turbolanding.model.SitelinkTurboLanding;
import ru.yandex.direct.dbschema.ppc.enums.BannersMinusGeoType;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.tables.records.SitelinksSetToLinkRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.SitelinksSetsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.bolts.function.forhuman.Comparator.naturalComparator;
import static ru.yandex.direct.common.util.GuavaCollectors.toMultimap;
import static ru.yandex.direct.core.entity.sitelink.repository.mapper.SitelinkTurboLandingMapperProvider.getSitelinkTurboLandingMapper;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MINUS_GEO;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_LINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_SETS;
import static ru.yandex.direct.dbschema.ppc.Tables.SITELINKS_SET_TO_LINK;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8;

@Repository
public class SitelinkSetRepository {

    public static final Function<List<Sitelink>, List<Sitelink>> SORT_SITELINKS_BY_ORDER_NUM =
            l -> l.stream()
                    .sorted(Comparator.comparing(Sitelink::getOrderNum))
                    .collect(toList());

    private final DslContextProvider dslContextProvider;
    private final SitelinkRepository sitelinkRepository;
    private final ShardHelper shardHelper;
    private final ShardSupport shardSupport;
    private final JooqMapperWithSupplier<SitelinkSet> sitelinksSetMapper;
    private final JooqMapperWithSupplier<Sitelink> sitelinkMapper;
    private final JooqMapperWithSupplier<SitelinkTurboLanding> turboLandingMapper;
    private final Collection<Field<?>> sitelinkFieldsToRead;
    private final Collection<Field<?>> sitelinkSetFieldsToRead;

    @Autowired
    public SitelinkSetRepository(ShardHelper shardHelper, ShardSupport shardSupport,
                                 SitelinkRepository sitelinkRepository,
                                 DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.shardSupport = shardSupport;
        this.dslContextProvider = dslContextProvider;
        this.sitelinkRepository = sitelinkRepository;

        sitelinkMapper = sitelinkRepository.createSitelinkMapper();
        turboLandingMapper = getSitelinkTurboLandingMapper();
        sitelinksSetMapper = createSitelinksSetMapper();
        sitelinkFieldsToRead = sitelinkMapper.getFieldsToRead();
        sitelinkSetFieldsToRead = sitelinksSetMapper.getFieldsToRead();
    }

    /**
     * Добавляет сайтлинк сеты. Ожидает, что сайтлинки в составе уже добавлены в базу
     * Новым сайтлинк сетам выставляет ид в модели
     *
     * @param shard        шард для запроса
     * @param sitelinkSets список сайтлинк сетов
     */
    public void add(int shard, Collection<SitelinkSet> sitelinkSets) {
        sitelinkSets.stream().filter(slSet -> slSet.getLinksHash() == null)
                .forEach(slSet -> slSet.withLinksHash(SitelinkSetRepository.calcHash(slSet)));

        StreamEx.of(sitelinkSets)
                .groupingBy(SitelinkSet::getClientId)
                .forEach(this::generateSitelinksSetIds);
        addSitelinkSetToTableSitelinkSet(shard, sitelinkSets);
        addSitelinkSetToTableSitelinkSetToLink(shard, sitelinkSets);
    }

    /**
     * Получаем список id сайтлинк сетов по clientId и linksHash
     *
     * @param shard        шард для запроса
     * @param sitelinkSets список сайтлинк сетов
     * @return список ид сайтлинк сетов
     */
    public Collection<Long> getSitelinkSetIds(int shard, Collection<SitelinkSet> sitelinkSets) {
        Condition condition = getClientIdWithHashCondition(sitelinkSets);

        return dslContextProvider.ppc(shard)
                .select(SITELINKS_SETS.SITELINKS_SET_ID)
                .from(SITELINKS_SETS)
                .where(condition)
                .fetch(SITELINKS_SETS.SITELINKS_SET_ID);
    }

    /**
     * Получает мапу links_hash -> sitelinks_set_id.
     *
     * @param shard        шард для запроса
     * @param clientId     ид клиента, для которого ищутся наборы сайтлинков
     * @param sitelinkSets список сайтлинк сетов
     * @return мапа links_hash -> sitelinks_set_id
     */
    public Map<BigInteger, Long> getSitelinkSetIdsByHashes(int shard,
                                                           ClientId clientId,
                                                           Collection<SitelinkSet> sitelinkSets) {
        List<BigInteger> hashes = mapList(sitelinkSets, SitelinkSet::getLinksHash);

        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(SITELINKS_SETS.LINKS_HASH, SITELINKS_SETS.SITELINKS_SET_ID))
                .from(SITELINKS_SETS)
                .where(SITELINKS_SETS.LINKS_HASH.in(hashes))
                .and(SITELINKS_SETS.CLIENT_ID.eq(clientId.asLong()))
                .fetch();

        return result.stream().collect(toMap(
                rec -> rec.getValue(SITELINKS_SETS.LINKS_HASH).toBigInteger(),
                rec -> rec.getValue(SITELINKS_SETS.SITELINKS_SET_ID),
                BinaryOperator.minBy(naturalComparator())));
    }

    /**
     * Получает мапу ид (sitelinks_set_id) -> (is_used)
     * Сайтлинк сет считается используемым, когда он привязан хоть к одной кампании с statusEmpty = No
     *
     * @param shard           шард для запроса
     * @param clientId        id клиента
     * @param sitelinkSetsIds список id сайтлинк сетов
     * @return возвращает мапу (sitelinks_set_id) -> (is_used)
     */
    public Map<Long, Boolean> getSitelinkSetIdsMapUsed(int shard, ClientId clientId,
                                                       Collection<Long> sitelinkSetsIds) {
        Field<Boolean> usedInCampCond = DSL.field(DSL.exists(getUsedSiteLinkSetQuery(shard))).as("used");
        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(SITELINKS_SETS.SITELINKS_SET_ID, usedInCampCond))
                .from(SITELINKS_SETS)
                .where(SITELINKS_SETS.CLIENT_ID.eq(clientId.asLong())
                        .and(SITELINKS_SETS.SITELINKS_SET_ID.in(sitelinkSetsIds)))
                .fetch();
        return StreamEx.of(result)
                .toMap(rec -> rec.getValue(SITELINKS_SETS.SITELINKS_SET_ID),
                        rec -> rec.getValue("used", Boolean.class));
    }

    /**
     * Получаем мапу, ключом является id сайтлик сета, значением список сайтликов
     *
     * @param shard           шард для запроса
     * @param sitelinksSetIds список id сайтлик сетов (sitelinks_set_id)
     * @return мапа sitelinks_set_id -> список сайтлинков
     */
    public ListMultimap<Long, Sitelink> getSitelinksBySetIds(int shard, Collection<Long> sitelinksSetIds) {
        Result<Record> result = dslContextProvider.ppc(shard)
                .select(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID)
                .select(sitelinkFieldsToRead)
                .from(SITELINKS_SET_TO_LINK)
                .join(SITELINKS_LINKS).using(SITELINKS_SET_TO_LINK.SL_ID)
                //  .leftJoin(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(SITELINKS_LINKS.TL_ID))
                .where(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID.in(sitelinksSetIds))
                .fetch();

        return StreamEx.of(result)
                .mapToEntry(rec -> rec.getValue(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID), sitelinkMapper::fromDb)
                .collapseKeys(collectingAndThen(toList(), SORT_SITELINKS_BY_ORDER_NUM))
                .collect(GuavaCollectors.toListMultimap());
    }

    /**
     * Получить сайтлинк сеты по их идентификаторам
     *
     * @param shard          шард для запроса
     * @param clientId       идентификатор клиента, для которого выбираются сайтлинк сеты
     * @param sitelinkSetIds коллекция идентификаторов
     * @return список сайтлинк сетов
     */
    public List<SitelinkSet> get(int shard, ClientId clientId, Collection<Long> sitelinkSetIds) {
        Result<Record> result = prepareSelect(dslContextProvider.ppc(shard), sitelinkSetIds)
                .and(SITELINKS_SETS.CLIENT_ID.eq(clientId.asLong()))
                .orderBy(SITELINKS_SETS.SITELINKS_SET_ID)
                .fetch();
        return collectToSitelinkSets(result);
    }

    /**
     * Получить сайтлинк сеты по их идентификаторам, без учета clientId
     *
     * @param shard          шард для запроса
     * @param sitelinkSetIds коллекция идентификаторов
     * @return список сайтлинк сетов
     */
    public List<SitelinkSet> get(int shard, Collection<Long> sitelinkSetIds) {
        return get(dslContextProvider.ppc(shard), sitelinkSetIds);
    }

    /**
     * Получить сайтлинк сеты по их идентификаторам, без учета clientId
     */
    public List<SitelinkSet> get(DSLContext context, Collection<Long> sitelinkSetIds) {
        Result<Record> result = prepareSelect(context, sitelinkSetIds)
                .orderBy(SITELINKS_SETS.SITELINKS_SET_ID)
                .fetch();
        return collectToSitelinkSets(result);
    }

    private SelectConditionStep<Record> prepareSelect(DSLContext context, Collection<Long> sitelinkSetIds) {
        return context
                .select(sitelinkSetFieldsToRead)
                .select(sitelinkFieldsToRead)
                .select(turboLandingMapper.getFieldsToRead())
                .from(SITELINKS_SETS)
                .join(SITELINKS_SET_TO_LINK).using(SITELINKS_SETS.SITELINKS_SET_ID)
                .join(SITELINKS_LINKS).using(SITELINKS_SET_TO_LINK.SL_ID)
                .leftJoin(TURBOLANDINGS).on(SITELINKS_LINKS.TL_ID.eq(TURBOLANDINGS.TL_ID))
                .where(SITELINKS_SETS.SITELINKS_SET_ID.in(sitelinkSetIds));
    }

    /**
     * Получить страницу списка идентификаторов сайтлинк сетов по их идентификаторам.
     * Метод нужен для корректной поддержки постраничной выдачи
     *
     * @param shard          шард для запроса
     * @param clientId       идентификатор клиента, для которого выбираются сайтлинк сеты
     * @param sitelinkSetIds коллекция идентификаторов
     * @param limitOffset    размер и смещение страницы выдачи
     * @return список идентификаторов сайтлинк сетов
     */
    public List<Long> getIds(int shard, ClientId clientId, Collection<Long> sitelinkSetIds,
                             LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(SITELINKS_SETS.SITELINKS_SET_ID)
                .from(SITELINKS_SETS)
                .where(SITELINKS_SETS.SITELINKS_SET_ID.in(sitelinkSetIds))
                .and(SITELINKS_SETS.CLIENT_ID.eq(clientId.asLong()))
                .orderBy(SITELINKS_SETS.SITELINKS_SET_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .map(Record1::value1);
    }

    /**
     * Получаем страницу списка идентификаторов сайтлинк сетов, принадлежащих определенному клиенту
     *
     * @param shard       шард для запроса
     * @param clientId    идентификатор клиента, для которого выбираются сайтлинк сеты
     * @param limitOffset размер и смещение страницы выдачи
     * @return список идентификаторов сайтлинк сетов
     */
    public List<Long> getIdsByClientId(int shard, ClientId clientId, LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(SITELINKS_SETS.SITELINKS_SET_ID)
                .from(SITELINKS_SETS)
                .where(SITELINKS_SETS.CLIENT_ID.eq(clientId.asLong()))
                .orderBy(SITELINKS_SETS.SITELINKS_SET_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .map(Record1::value1);
    }

    private List<SitelinkSet> collectToSitelinkSets(Result<Record> result) {
        List<SitelinkSet> sitelinkSets = StreamEx.of(result)
                .map(sitelinksSetMapper::fromDb)
                .distinct()
                .toList();
        ListMultimap<Long, Sitelink> slSetIdToSitelinks = StreamEx.of(result)
                .mapToEntry(rec -> rec.getValue(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID),
                        sitelinkRepository::constructSitelinkFromRecord)
                .collapseKeys(collectingAndThen(toList(), SORT_SITELINKS_BY_ORDER_NUM))
                .collect(GuavaCollectors.toListMultimap());
        sitelinkSets.forEach(sitelinkSet -> sitelinkSet.withSitelinks(slSetIdToSitelinks.get(sitelinkSet.getId())));
        return sitelinkSets;
    }

    /**
     * Удаление сайтлинк сетов по ид.
     * Не удаляет сеты, которые привязаны к баннеру, за исключением, когда camp.statusEmpty = 'Yes'
     *
     * @param shard           шард для запроса
     * @param clientId        id клиента
     * @param sitelinksSetIds список id сайтлик сетов (sitelinks_set_id)
     * @return список удаленных id сайтлинк сетов
     */
    public Set<Long> delete(int shard, ClientId clientId, Collection<Long> sitelinksSetIds) {
        if (sitelinksSetIds.isEmpty()) {
            return Collections.emptySet();
        }

        deleteFromSiteLinkSetsTable(shard, clientId, sitelinksSetIds);

        Set<Long> deletedSiteLinkSetIds = dslContextProvider.ppc(shard)
                .selectDistinct(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID)
                .from(SITELINKS_SET_TO_LINK)
                .leftJoin(SITELINKS_SETS).on(SITELINKS_SETS.SITELINKS_SET_ID.eq(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID))
                .where(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID.in(sitelinksSetIds)
                        .and(SITELINKS_SETS.SITELINKS_SET_ID.isNull()))
                .fetchSet(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID);

        shardSupport.deleteValues(ShardKey.SITELINKS_SET_ID, deletedSiteLinkSetIds);

        deleteFromSiteinksSetsToLinkTable(shard, deletedSiteLinkSetIds);

        return deletedSiteLinkSetIds;
    }

    private void addSitelinkSetToTableSitelinkSet(int shard, Collection<SitelinkSet> sitelinkSets) {
        if (sitelinkSets.isEmpty()) {
            return;
        }
        InsertHelper<SitelinksSetsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), SITELINKS_SETS);
        sitelinkSets.forEach(sitelinksSet -> insertHelper.add(sitelinksSetMapper, sitelinksSet).newRecord());
        insertHelper.execute();
    }

    private void addSitelinkSetToTableSitelinkSetToLink(int shard, Collection<SitelinkSet> sitelinkSets) {
        if (sitelinkSets.isEmpty()) {
            return;
        }
        InsertHelper<SitelinksSetToLinkRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), SITELINKS_SET_TO_LINK);
        for (SitelinkSet sitelinkSet : sitelinkSets) {
            long orderNum = 0;
            for (Sitelink sitelink : sitelinkSet.getSitelinks()) {
                insertHelper.set(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID, sitelinkSet.getId())
                        .set(SITELINKS_SET_TO_LINK.SL_ID, sitelink.getId())
                        .set(SITELINKS_SET_TO_LINK.ORDER_NUM, orderNum++)
                        .newRecord();
            }
        }
        insertHelper.execute();
    }

    private Condition getClientIdWithHashCondition(Collection<SitelinkSet> sitelinkSets) {
        Multimap<Long, ULong> clientIdToHashes = sitelinkSets.stream()
                .collect(toMultimap(SitelinkSet::getClientId, slSet -> ULong.valueOf(slSet.getLinksHash())));

        return StreamEx.of(sitelinkSets)
                .map(slSet -> SITELINKS_SETS.CLIENT_ID.eq(slSet.getClientId())
                        .and(SITELINKS_SETS.LINKS_HASH.in(clientIdToHashes.get(slSet.getClientId()))))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
    }

    private void generateSitelinksSetIds(Long clientId, Collection<SitelinkSet> sitelinkSets) {
        Iterator<Long> ids = shardHelper.generateSitelinkSetIds(clientId, sitelinkSets.size()).iterator();
        sitelinkSets.forEach(sitelinkSet -> sitelinkSet.setId(ids.next()));
    }

    private void deleteFromSiteLinkSetsTable(int shard, ClientId clientId, Collection<Long> sitelinkSetIds) {
        Table<Record1<Long>> tmpTable = dslContextProvider.ppc(shard)
                .select(SITELINKS_SETS.SITELINKS_SET_ID)
                .from(SITELINKS_SETS)
                .where(SITELINKS_SETS.CLIENT_ID.eq(clientId.asLong()))
                .and(SITELINKS_SETS.SITELINKS_SET_ID.in(sitelinkSetIds))
                .and(DSL.exists(getUsedSiteLinkSetQuery(shard)))
                .asTable("tmp");
        Select<Record1<Long>> select = dslContextProvider.ppc(shard)
                .select(tmpTable.field(SITELINKS_SETS.SITELINKS_SET_ID))
                .from(tmpTable);
        dslContextProvider.ppc(shard)
                .delete(SITELINKS_SETS)
                .where(SITELINKS_SETS.CLIENT_ID.eq(clientId.asLong())
                        .and(SITELINKS_SETS.SITELINKS_SET_ID.in(sitelinkSetIds))
                        .and(SITELINKS_SETS.SITELINKS_SET_ID.notIn(select)))
                .execute();
    }

    private Select getUsedSiteLinkSetQuery(int shard) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNERS.SITELINKS_SET_ID.eq(SITELINKS_SETS.SITELINKS_SET_ID)
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)));
    }

    private void deleteFromSiteinksSetsToLinkTable(int shard, Collection<Long> sitelinkSetIds) {
        dslContextProvider.ppc(shard)
                .deleteFrom(SITELINKS_SET_TO_LINK)
                .where(SITELINKS_SET_TO_LINK.SITELINKS_SET_ID.in(sitelinkSetIds))
                .execute();
    }

    /**
     * Вычисляет хеш по списку id сайтликов в сете
     *
     * @param sitelinkSet сайтлинк сет
     * @return хеш сайтлинк сета
     */
    public static BigInteger calcHash(SitelinkSet sitelinkSet) {
        List<Long> sitelinkIds = mapList(sitelinkSet.getSitelinks(), Sitelink::getId);
        checkArgument(!sitelinkIds.contains(null), "sitelink id in set cannot ne null");
        return getMd5HalfHashUtf8(StringUtils.join(sitelinkIds, ','));
    }

    public JooqMapperWithSupplier<SitelinkSet> createSitelinksSetMapper() {
        return JooqMapperWithSupplierBuilder.builder(SitelinkSet::new)
                .map(property(SitelinkSet.ID, SITELINKS_SETS.SITELINKS_SET_ID))
                .map(property(SitelinkSet.CLIENT_ID, SITELINKS_SETS.CLIENT_ID))
                .map(convertibleProperty(SitelinkSet.LINKS_HASH, SITELINKS_SETS.LINKS_HASH,
                        hash -> hash != null ? hash.toBigInteger() : null,
                        hash -> hash != null ? ULong.valueOf(hash) : null))
                .build();
    }

    public List<Long> selectGroupsForResync(Configuration configuration, List<Long> bids) {

        var current = BANNERS_MINUS_GEO.as("current");
        var synced = BANNERS_MINUS_GEO.as("synced");

        return configuration.dsl()
                .select(BANNERS.PID)
                .from(BANNERS)
                .leftJoin(current).on(BANNERS.BID.eq(current.BID), current.TYPE.eq(BannersMinusGeoType.current))
                .leftJoin(synced).on(BANNERS.BID.eq(synced.BID), synced.TYPE.eq(BannersMinusGeoType.bs_synced))
                .where(BANNERS.BID.in(bids),
                        DSL.or(
                                BANNERS.BANNER_ID.eq(0L),
                                DSL.ifnull(synced.MINUS_GEO, "").ne(DSL.ifnull(current.MINUS_GEO, ""))
                        )
                )
                .fetch(Record1::value1);
    }
}
