package ru.yandex.direct.core.entity.sitelink.service;

import java.math.BigInteger;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.Multimaps;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkRepository;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkSetValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.collect.Multimaps.asMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class SitelinkSetAddOperation extends SimpleAbstractAddOperation<SitelinkSet, Long> {

    private final SitelinkSetRepository sitelinkSetRepository;
    private final SitelinkRepository sitelinkRepository;
    private final SitelinkSetValidationService sitelinkSetValidationService;

    private ClientId clientId;
    private int shard;

    @Autowired
    public SitelinkSetAddOperation(Applicability applicability,
                                   List<SitelinkSet> models,
                                   SitelinkSetRepository sitelinkSetRepository,
                                   SitelinkRepository sitelinkRepository,
                                   SitelinkSetValidationService sitelinkSetValidationService,
                                   ClientId clientId, int shard) {
        super(applicability, models);
        this.sitelinkSetRepository = sitelinkSetRepository;
        this.sitelinkRepository = sitelinkRepository;
        this.sitelinkSetValidationService = sitelinkSetValidationService;
        this.clientId = clientId;
        this.shard = shard;
    }

    /**
     * Метод нужен только для частичного обновления баннера
     */
    @Deprecated
    public ValidationResult<List<SitelinkSet>, Defect> getValidationResult() {
        return validationResult;
    }

    @Override
    protected ValidationResult<List<SitelinkSet>, Defect> preValidate(List<SitelinkSet> models) {
        setIdsOfExistingSitelinkSets(shard, clientId, models);
        return super.preValidate(models);
    }

    @Override
    protected void validate(
            ValidationResult<List<SitelinkSet>, Defect> preValidationResult) {
        new ItemValidationBuilder<>(preValidationResult)
                .checkBy(sets -> sitelinkSetValidationService.validateSitelinkSets(sets, clientId));
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<SitelinkSet> modelsValidatedStep) {
        prepareSystemFields(modelsValidatedStep.getValidModelsMap().values());
    }

    private void prepareSystemFields(Collection<SitelinkSet> validSitelinkSets) {
        validSitelinkSets.forEach(s -> s.setClientId(clientId.asLong()));
    }

    @Override
    protected List<Long> execute(List<SitelinkSet> validModelsToApply) {
        // Вся эта хитрая логика нужна для обработки сценария, когда от пользователя нам приходят два одинаковых
        // набора сайтлинков в одном запросе, которых при этом еще нет в базе, и мы должны один вставить нормально,
        // а второму проставить идентификатор первого (который будет известен только после вставки), и повесить на
        // второй предупреждение, что такой набор уже был вставлен
        List<Sitelink> sitelinks = StreamEx.of(validModelsToApply)
                .flatMap(sls -> sls.getSitelinks().stream())
                .filter(sl -> sl.getId() == null)
                .toList();
        addSitelinks(sitelinks);

        // дедуплицирует наборы сайтлинков, вставляет только уникальные, затем у дубликатов проставляет правильные
        // идентификаторы.
        List<SitelinkSet> sitelinkSets = StreamEx.of(validModelsToApply)
                .filter(sls -> sls.getId() == null)
                .map(slSet -> slSet.withLinksHash(SitelinkSetRepository.calcHash(slSet)))
                .toList();
        ImmutableListMultimap<BigInteger, SitelinkSet> sets = Multimaps.index(sitelinkSets, SitelinkSet::getLinksHash);
        Map<BigInteger, SitelinkSet> uniqueValidModels = asMap(sets).values().stream()
                .map(v -> v.get(0))
                .collect(toMap(SitelinkSet::getLinksHash, identity()));


        sitelinkSetRepository.add(shard, uniqueValidModels.values());


        List<SitelinkSet> allModels = validationResult.getValue();
        for (SitelinkSet currentModel : allModels) {
            SitelinkSet validModel = uniqueValidModels.get(currentModel.getLinksHash());
            if (!(validModel == null || currentModel == validModel)) {
                currentModel.setId(validModel.getId());
            }
        }

        return mapList(validModelsToApply, SitelinkSet::getId);
    }

    /**
     * Дедуплицирует сайтлинки из списка, вставляет только уникальные сайтлинки, затем у дубликатов проставляет
     * правильные идентификаторы
     *
     * @param sitelinks сайтлинки, которые нужно вставить
     */
    private void addSitelinks(List<Sitelink> sitelinks) {
        ImmutableListMultimap<BigInteger, Sitelink> links = Multimaps.index(sitelinks, Sitelink::getHash);
        Map<BigInteger, Sitelink> uniqueSitelinks = asMap(links).values().stream()
                .map(sl -> sl.get(0))
                .collect(toMap(Sitelink::getHash, identity()));

        sitelinkRepository.add(shard, uniqueSitelinks.values());

        for (Sitelink current : sitelinks) {
            Sitelink unique = uniqueSitelinks.get(current.getHash());
            if (unique != current) {
                current.setId(unique.getId());
            }
        }
    }


    private void setIdsOfExistingSitelinkSets(int shard, ClientId clientId, List<SitelinkSet> sitelinkSets) {
        List<Sitelink> sitelinksWithHashes = sitelinkSets.stream().flatMap(set -> set.getSitelinks().stream())
                .map(sl -> sl.withHash(SitelinkRepository.calcHashSafe(sl)))
                .filter(sl -> sl.getHash() != null)
                .collect(toList());

        Map<BigInteger, Long> existingSitelinks
                = sitelinkRepository.getSitelinkIdsByHashes(shard, sitelinksWithHashes);
        for (Sitelink sitelink : sitelinksWithHashes) {
            Long existingSitelinkId = existingSitelinks.get(sitelink.getHash());
            sitelink.setId(existingSitelinkId);
        }

        List<SitelinkSet> sitelinksExist =
                sitelinkSets.stream().filter(set -> set.getSitelinks().stream().allMatch(sl -> sl.getId() != null))
                        .map(sls -> sls.withLinksHash(SitelinkSetRepository.calcHash(sls)))
                        .collect(toList());
        Map<BigInteger, Long> existingSitelinkSets =
                sitelinkSetRepository.getSitelinkSetIdsByHashes(shard, clientId, sitelinksExist);
        for (SitelinkSet sitelinkSet : sitelinkSets) {
            Long existingSetId = existingSitelinkSets.get(sitelinkSet.getLinksHash());
            sitelinkSet.setId(existingSetId);
        }
    }
}
