package ru.yandex.direct.core.entity.sitelink.service;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkRepository;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkSetValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkSetDefects.sitelinkSetInUse;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

/**
 * Сервис сайтлинк сетов (сетов уточнений).
 */
@Service
@ParametersAreNonnullByDefault
public class SitelinkSetService implements EntityService<SitelinkSet, Long> {

    private final ShardHelper shardHelper;
    private final SitelinkSetRepository sitelinkSetRepository;
    private final SitelinkRepository sitelinkRepository;
    private final SitelinkSetValidationService sitelinkSetValidationService;

    @Autowired
    public SitelinkSetService(ShardHelper shardHelper,
                              SitelinkSetRepository sitelinkSetRepository,
                              SitelinkRepository sitelinkRepository,
                              SitelinkSetValidationService sitelinkSetValidationService) {
        this.shardHelper = shardHelper;
        this.sitelinkSetRepository = sitelinkSetRepository;
        this.sitelinkRepository = sitelinkRepository;
        this.sitelinkSetValidationService = sitelinkSetValidationService;
    }

    /**
     * Выбирает наборы сайтлинков постранично по их идентификаторам, или все для данного клиента
     *
     * @param ids         идентификаторы наборов сайтлинков
     * @param clientId    идентификатор клиента, которому принадлежат сайтлинки
     * @param limitOffset размер и смещение страницы выдачи
     * @return Список наборов сайтлинков данного клиента, соответствующих условию выборки
     */
    public List<SitelinkSet> getSitelinkSets(@Nullable Collection<Long> ids, ClientId clientId,
                                             LimitOffset limitOffset) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<Long> pagedIds;
        if (ids != null && !ids.isEmpty()) {
            pagedIds = sitelinkSetRepository.getIds(shard, clientId, ids, limitOffset);
        } else {
            pagedIds = sitelinkSetRepository.getIdsByClientId(shard, clientId, limitOffset);
        }
        return sitelinkSetRepository.get(shard, clientId, pagedIds);
    }

    /**
     * Добавление сайтлинк сетов, если все сеты прошли этап валидации
     *
     * @param clientId     - ид клиента, которому будут добавлены сайтлинк сеты
     * @param sitelinkSets - список сайтлинк сетов
     */
    public MassResult<Long> addSitelinkSetsFull(ClientId clientId, List<SitelinkSet> sitelinkSets) {
        return addSitelinkSets(clientId, sitelinkSets, Applicability.FULL);
    }

    /**
     * Добавление сайтлинк сетов. Добавил только те сеты, которые прошли этап валидации
     *
     * @param clientId     - ид клиента, которому будут добавлены сайтлинк сеты
     * @param sitelinkSets - список сайтлинк сетов
     */
    public MassResult<Long> addSitelinkSetsPartial(ClientId clientId, List<SitelinkSet> sitelinkSets) {
        return addSitelinkSets(clientId, sitelinkSets, Applicability.PARTIAL);
    }

    /**
     * Массовое удаление сайтлинк сетов для клиента.
     * Считается успешным, если удален хотя бы один сайтлинк сет.
     *
     * @param clientId       id клиента
     * @param sitelinkSetIds удаляемые id сайтлинк сетов
     * @return результат удаления сайтлинк сетов
     */
    public MassResult<Long> deleteSiteLinkSets(ClientId clientId, List<Long> sitelinkSetIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        ValidationResult<List<Long>, Defect> vr =
                sitelinkSetValidationService.validateDelete(shard, clientId, sitelinkSetIds);
        List<Long> validSiteLinkSetIds = getValidItems(vr);
        if (vr.hasErrors() || validSiteLinkSetIds.isEmpty()) {
            return MassResult.brokenMassAction(sitelinkSetIds, vr);
        }

        Set<Long> deletedSiteLinkSetIds = sitelinkSetRepository.delete(shard, clientId, validSiteLinkSetIds);

        vr.getSubResults().forEach((pathNode, subResult) -> {
            int idx = ((PathNode.Index) pathNode).getIndex();
            Long value = sitelinkSetIds.get(idx);
            if (!subResult.hasAnyErrors() && !deletedSiteLinkSetIds.contains(value)) {
                subResult.addError(sitelinkSetInUse());
            }
        });

        return MassResult.successfulMassAction(sitelinkSetIds, vr);
    }

    public SitelinkSetAddOperation createAddOperation(ClientId clientId, List<SitelinkSet> sitelinkSets,
                                                      Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return createAddOperation(shard, clientId, sitelinkSets, applicability);
    }

    public SitelinkSetAddOperation createAddOperation(int shard, ClientId clientId, List<SitelinkSet> sitelinkSets,
                                                      Applicability applicability) {
        return new SitelinkSetAddOperation(applicability,
                sitelinkSets,
                sitelinkSetRepository,
                sitelinkRepository,
                sitelinkSetValidationService,
                clientId,
                shard);
    }

    private MassResult<Long> addSitelinkSets(ClientId clientId, List<SitelinkSet> sitelinkSets,
                                             Applicability applicability) {
        return createAddOperation(clientId, sitelinkSets, applicability).prepareAndApply();
    }

    private MassResult<Long> addSitelinkSets(int shard, ClientId clientId, List<SitelinkSet> sitelinkSets,
                                             Applicability applicability) {
        return createAddOperation(shard, clientId, sitelinkSets, applicability).prepareAndApply();
    }

    @Override
    public List<SitelinkSet> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return sitelinkSetRepository.get(shard, clientId, ids);
    }

    @Override
    public MassResult<Long> add(ClientId clientId, Long operatorUid, List<SitelinkSet> entities,
                                Applicability applicability) {
        return addSitelinkSets(clientId, entities, applicability);
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer,
                                 List<SitelinkSet> entities, Applicability applicability) {
        return addSitelinkSets(copyContainer.getShardTo(), copyContainer.getClientIdTo(), entities, applicability);
    }
}
