package ru.yandex.direct.core.entity.sitelink.service.validation;

import java.util.List;
import java.util.function.Predicate;

import com.google.common.base.CharMatcher;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static com.google.common.base.CharMatcher.anyOf;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstants.ALLOW_SITELINK_LETTERS;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstants.SITELINKS_MAX_LENGTH;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstants.SITELINKS_PER_BLOCK;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.maxSetSitelinkTitlesSize;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.maxSitelinksFirstBlockTitlesLength;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.maxSitelinksSecondBlockTitlesLength;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.requiredSitelinkWithTurboHrefOrTurbolandingId;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.StringDefects.admissibleChars;

public final class SitelinkConstraints {

    private static final CharMatcher ALLOW_SITELINK_LETTERS_MATCHER = anyOf(ALLOW_SITELINK_LETTERS);

    private SitelinkConstraints() {
    }

    /**
     * Проверка допустимых символов
     */
    static Constraint<String, Defect> allowedSitelinkChars() {
        return fromPredicate(ALLOW_SITELINK_LETTERS_MATCHER::matchesAllOf, admissibleChars());
    }

    /* Валидация блока сайтлинков с 1го по 4й */
    static Constraint<List<Sitelink>, Defect> sitelinksFirstBlockMaxTitleLengths(int sitelinksListSize) {
        Predicate<List<Sitelink>> predicate = sitelinks -> {
            int sumLength = StreamEx.of(sitelinks)
                    .limit(SITELINKS_PER_BLOCK)
                    .nonNull()
                    .map(Sitelink::getTitle)
                    .nonNull()
                    .mapToInt(String::length)
                    .sum();
            return sumLength <= SITELINKS_MAX_LENGTH;
        };
        return fromPredicate(predicate, sitelinksListSize > 4 ?
                maxSitelinksFirstBlockTitlesLength(SITELINKS_MAX_LENGTH) :
                maxSetSitelinkTitlesSize(SITELINKS_MAX_LENGTH));
    }

    /* Валидация блока сайтлинков с 5го по 8й */
    static Constraint<List<Sitelink>, Defect> sitelinksSecondBlockMaxTitleLengths() {
        Predicate<List<Sitelink>> predicate = sitelinks -> {
            int sumLength = StreamEx.of(sitelinks)
                    .skip(SITELINKS_PER_BLOCK)
                    .nonNull()
                    .map(Sitelink::getTitle)
                    .nonNull()
                    .mapToInt(String::length)
                    .sum();
            return sumLength <= SITELINKS_MAX_LENGTH;
        };
        return fromPredicate(predicate, maxSitelinksSecondBlockTitlesLength(SITELINKS_MAX_LENGTH));
    }

    public static Constraint<Sitelink, Defect> sitelinkContainsTurbolandingOnly() {
        return fromPredicate(SitelinkConstraints::sitelinkContainsTurbolandingOnly,
                requiredSitelinkWithTurboHrefOrTurbolandingId());
    }

    public static boolean sitelinkContainsTurbolandingOnly(Sitelink sitelink) {
        return sitelink.getHref() == null && sitelink.getTurboLandingId() != null;
    }
}
