package ru.yandex.direct.core.entity.sitelink.service.validation;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstraints.sitelinksFirstBlockMaxTitleLengths;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstraints.sitelinksSecondBlockMaxTitleLengths;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.duplicateSitelinkDescs;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.duplicateSitelinkTitles;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkSetConstraints.sitelinkSetDoesNotHaveId;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkSetConstraints.sitelinkSetNotUsed;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkSetDefects.sitelinkCountMustBeBetween;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.listSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class SitelinkSetValidationService {
    public static final int MIN_SITELINKS_PER_SET = 1;
    public static final int MAX_SITELINKS_PER_SET = 8;
    //ToDo: оторвать старые константы после включения фичи на 100% (DIRECT-97398)
    public static final int MAX_SITELINKS_PER_SET_OLD = 4;

    private final SitelinkValidationService sitelinkValidationService;
    private final SitelinkSetRepository sitelinkSetRepository;
    private final TurboLandingService turboLandingService;
    private final FeatureService featureService;

    @Autowired
    public SitelinkSetValidationService(
            SitelinkValidationService sitelinkValidationService,
            TurboLandingService turboLandingService,
            SitelinkSetRepository sitelinkSetRepository,
            FeatureService featureService) {
        this.sitelinkValidationService = sitelinkValidationService;
        this.sitelinkSetRepository = sitelinkSetRepository;
        this.turboLandingService = turboLandingService;
        this.featureService = featureService;
    }

    public ValidationResult<List<SitelinkSet>, Defect> validateSitelinkSets(
            List<SitelinkSet> sitelinkSets,
            ClientId clientId
    ) {
        Set<Long> turbolandingIds = new HashSet<>();
        sitelinkSets.forEach(s -> s.getSitelinks().forEach(
                sl -> {
                    if (sl.getTurboLandingId() != null) {
                        turbolandingIds.add(sl.getTurboLandingId());
                    }
                })
        );

        Set<Long> existingTurbolandingIds = turboLandingService.fetchTurboLandingIds(clientId, turbolandingIds);
        Boolean clientHasTextCampaign8SitelinksFeature = featureService.isEnabledForClientId(clientId,
                FeatureName.TEXT_CAMPAIGN_8_SITELINKS);
        boolean isTotalSitelinksLengthValidationDisabled = featureService.isEnabledForClientId(clientId,
                FeatureName.DISABLE_TOTAL_SITELINKS_LENGTH_VALIDATION);

        return ListValidationBuilder.of(sitelinkSets, Defect.class)
                .checkEachBy(set -> validateOneSitelinksSet(set, existingTurbolandingIds,
                        clientHasTextCampaign8SitelinksFeature,
                        isTotalSitelinksLengthValidationDisabled))
                .getResult();
    }

    public ValidationResult<SitelinkSet, Defect> validateOneSitelinksSet(
            SitelinkSet sitelinkSet,
            Set<Long> existingTurbolandingsids,
            Boolean clientHasTextCampaign8SitelinksFeature,
            boolean isTotalSitelinksLengthValidationDisabled
    ) {
        ModelItemValidationBuilder<SitelinkSet> v = ModelItemValidationBuilder.of(sitelinkSet);
        int maxSitelinkPerSetListSize = clientHasTextCampaign8SitelinksFeature ? MAX_SITELINKS_PER_SET :
                MAX_SITELINKS_PER_SET_OLD;
        v.list(SitelinkSet.SITELINKS)
                .check(notNull())
                .check(listSize(MIN_SITELINKS_PER_SET, maxSitelinkPerSetListSize),
                        sitelinkCountMustBeBetween(MIN_SITELINKS_PER_SET, maxSitelinkPerSetListSize))
                .checkEach(notNull())
                .check(sitelinksFirstBlockMaxTitleLengths(sitelinkSet.getSitelinks() != null ?
                        sitelinkSet.getSitelinks().size() : 0), When.isFalse(isTotalSitelinksLengthValidationDisabled))
                .check(sitelinksSecondBlockMaxTitleLengths(), When.isTrue(clientHasTextCampaign8SitelinksFeature &&
                        !isTotalSitelinksLengthValidationDisabled))
                .checkEach(unique(Sitelink::getTitle), duplicateSitelinkTitles())
                .checkEach(unique(Sitelink::getDescription), duplicateSitelinkDescs())

                .checkEachBy(
                        s -> sitelinkValidationService.validateOneSitelink(s, existingTurbolandingsids),
                        When.notNull()
                );

        v.weakCheck(sitelinkSetDoesNotHaveId());// нет id - значит не существует такой в базе. В перле эта проверка
        // делается уже после вставки, в Яве addOperation не позволяет так
        // обработать результат вставки
        return v.getResult();
    }

    public ValidationResult<List<Long>, Defect> validateDelete(int shard, ClientId clientId,
                                                               List<Long> sitelinkSetIds) {
        Map<Long, Boolean> existsSiteLinks =
                sitelinkSetRepository.getSitelinkSetIdsMapUsed(shard, clientId, sitelinkSetIds);
        return ListValidationBuilder.<Long, Defect>of(sitelinkSetIds)
                .check(notNull())
                .checkEach(notNull())
                .checkEach(unique(), SitelinkDefects.duplicateObject(), When.isValid())
                .checkEach(inSet(existsSiteLinks.keySet()), objectNotFound(), When.isValid())
                .checkEach(sitelinkSetNotUsed(existsSiteLinks), When.isValid())
                .getResult();
    }
}
