package ru.yandex.direct.core.entity.sitelink.service.validation;

import java.util.Set;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.type.turbolanding.BannerWithTurboLandingConstraints.turboLandingExists;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkConstraints.allowedSitelinkChars;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.allowedSymbolsSitelinkDesc;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.allowedSymbolsSitelinkTitle;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.invalidSitelinkHref;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.sitelinkDescriptionTooLong;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.sitelinkHrefTooLong;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.sitelinkTitleTooLong;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.sitelinksDescriptionEmpty;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefects.sitelinksTitleEmpty;
import static ru.yandex.direct.core.validation.defects.Defects.hrefOrTurboRequired;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;

@Service
public class SitelinkValidationService {

    static ModelItemValidationBuilder<Sitelink> baseChecks(Sitelink sitelink) {
        ModelItemValidationBuilder<Sitelink> v = ModelItemValidationBuilder.of(sitelink);

        if (sitelink.getHref() != null) {
            v.item(Sitelink.HREF)
                    .check(notBlank(), invalidSitelinkHref(), When.isValid())
                    .check(maxStringLength(SitelinkConstants.MAX_SITELINK_HREF_LENGTH),
                            sitelinkHrefTooLong(SitelinkConstants.MAX_SITELINK_HREF_LENGTH),
                            When.isValid())
                    .check(StringConstraints.validHref(), invalidSitelinkHref(), When.isValid());
        } else {
            v.item(Sitelink.HREF)
                    .check(notNull(), hrefOrTurboRequired(), When.isTrue(sitelink.getTurboLandingId() == null));
        }

        v.item(Sitelink.TITLE)
                .check(notNull())
                .check(notEmpty(), sitelinksTitleEmpty(), When.isValid())
                .check(maxStringLength(SitelinkConstants.MAX_SITELINK_TITLE_LENGTH),
                        sitelinkTitleTooLong(SitelinkConstants.MAX_SITELINK_TITLE_LENGTH),
                        When.isValid())
                .check(allowedSitelinkChars(), allowedSymbolsSitelinkTitle(), When.isValid());
        v.item(Sitelink.DESCRIPTION)
                .check(notEmpty(), sitelinksDescriptionEmpty())
                .check(maxStringLength(SitelinkConstants.MAX_SITELINK_DESC_LENGTH),
                        sitelinkDescriptionTooLong(SitelinkConstants.MAX_SITELINK_DESC_LENGTH),
                        When.isValid())
                .check(allowedSitelinkChars(), allowedSymbolsSitelinkDesc(), When.isValid());
        return v;
    }

    public ValidationResult<Sitelink, Defect> validateOneSitelink(
            Sitelink sitelink,
            Set<Long> existingTurboLandings
    ) {
        ModelItemValidationBuilder<Sitelink> v = baseChecks(sitelink);

        v.item(Sitelink.TURBO_LANDING_ID)
                .check(turboLandingExists(existingTurboLandings), When.notNull());

        return v.getResult();
    }
}
