package ru.yandex.direct.core.entity.statistics.repository;

import java.util.Collection;
import java.util.List;
import java.util.function.Predicate;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.statistics.container.ProceededActiveOrder;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.INTERNAL_FREE;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;

@Repository
public class ActiveOrdersRepository {

    private final DslContextProvider dslContextProvider;

    private static final Campaigns CAMP = CAMPAIGNS.as("camp");
    private static final Campaigns WALLET_CAMP = CAMPAIGNS.as("wallet_camp");
    private static final int UPDATE_CHUNK_SIZE = 1000;

    public ActiveOrdersRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public int updateCampaigns(int shard, Collection<ProceededActiveOrder> proceededActiveOrder) {
        Iterable<List<ProceededActiveOrder>> chunks = Iterables.partition(proceededActiveOrder, UPDATE_CHUNK_SIZE);
        int result = 0;
        for (var chunk : chunks) {
            result += (updateCampaignsChunk(shard, chunk));
        }
        return result;
    }

    private int updateCampaignsChunk(int shard, Collection<ProceededActiveOrder> activeOrdersProceeded) {

        var showsStatement = JooqMapperUtils.makeCaseStatement(CAMP.CID, CAMP.SHOWS,
                StreamEx.of(activeOrdersProceeded).toMap(ProceededActiveOrder::getCid,
                        ProceededActiveOrder::getNewShows));
        var clicksStatement = JooqMapperUtils.makeCaseStatement(CAMP.CID, CAMP.CLICKS,
                StreamEx.of(activeOrdersProceeded).toMap(ProceededActiveOrder::getCid,
                        ProceededActiveOrder::getNewClicks));

        var sumSpentStatement = JooqMapperUtils.makeCaseStatement(CAMP.CID, CAMP.SUM_SPENT,
                StreamEx.of(activeOrdersProceeded).toMap(ProceededActiveOrder::getCid,
                        ProceededActiveOrder::getNewSumSpent));

        // Выражение для записи sum_units для внутренних бесплатных кампаний.
        // Должно использовать только с internalFreeCondition, чтобы гарантировать,
        // что юниты будут перезаписаны только для внутренних бесплатных кампаний.
        var unitsStatement = JooqMapperUtils.makeCaseStatement(CAMP.CID, CAMP.SUM_UNITS,
                StreamEx.of(activeOrdersProceeded).toMap(ProceededActiveOrder::getCid, ProceededActiveOrder::getUnits));

        var sumSpentUnitsStatement = JooqMapperUtils.makeCaseStatement(CAMP.CID, CAMP.SUM_SPENT_UNITS,
                StreamEx.of(activeOrdersProceeded).toMap(ProceededActiveOrder::getCid,
                        order -> order.getType() == INTERNAL_FREE ? order.getNewSumSpentUnits() : order.getNewShows()));

        var rollbackedCidCondition = getCidsCondition(ProceededActiveOrder::isRollbacked, activeOrdersProceeded);
        var newShowsCondition = getCidsCondition(ProceededActiveOrder::isNewShows, activeOrdersProceeded);
        var finishedCampaignsCondition = getCidsCondition(ProceededActiveOrder::isFinished, activeOrdersProceeded);
        var internalFreeCondition = getCidsCondition(order -> order.getType() == INTERNAL_FREE, activeOrdersProceeded);

        var cidsToUpdate = activeOrdersProceeded.stream()
                .map(ProceededActiveOrder::getCid)
                .collect(toList());

        return dslContextProvider.ppc(shard)
                .transactionResult(conf ->
                {
                    int updatedCamp = conf.dsl().update(CAMP)
                            .set(CAMP.CLICKS, clicksStatement)
                            .set(CAMP.SHOWS, showsStatement)
                            .set(CAMP.SUM_SPENT, sumSpentStatement)
                            .set(CAMP.SUM_UNITS, DSL.iif(internalFreeCondition, unitsStatement, CAMP.SUM_UNITS))
                            .set(CAMP.SUM_SPENT_UNITS, sumSpentUnitsStatement)
                            .set(CAMP.STATUS_BS_SYNCED, DSL.iif(rollbackedCidCondition, CampaignsStatusbssynced.No,
                                    CAMP.STATUS_BS_SYNCED))
                            .set(CAMP.LAST_SHOW_TIME, DSL.iif(newShowsCondition, DSL.currentLocalDateTime(),
                                    CAMP.LAST_SHOW_TIME))
                            .where(CAMP.CID.in(cidsToUpdate))
                            .execute();

                    // обновление статуса активности зависит от нового значения sum_spent, поэтому обновление нужно
                    // выполнять отдельным запросом
                    // сделать как в старом скрипте обновление статуса последним в update'е нельзя, потому что при
                    // multiple-table updates не гарантируется порядок обновления полей
                    // https://docs.oracle.com/cd/E17952_01/mysql-5.7-en/update.html
                    conf.dsl().update(CAMP.leftJoin(WALLET_CAMP).on(CAMP.WALLET_CID.eq(WALLET_CAMP.CID)))
                            .set(CAMP.STATUS_ACTIVE, DSL.iif(CampaignRepository.campaignStatusActiveCondition(CAMP,
                                    WALLET_CAMP), CampaignsStatusactive.Yes, CampaignsStatusactive.No))
                            .where(finishedCampaignsCondition)
                            .execute();


                    return updatedCamp;
                });
    }

    private Condition getCidsCondition(Predicate<? super ProceededActiveOrder> predicate,
                                       Collection<ProceededActiveOrder> orders) {
        var cids = StreamEx.of(orders)
                .filter(predicate)
                .map(ProceededActiveOrder::getCid)
                .toList();
        return CAMP.CID.in(cids);
    }


}
