package ru.yandex.direct.core.entity.statistics.repository;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.jooq.Field;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.statistics.AdvqHits;
import ru.yandex.direct.core.entity.statistics.AdvqHitsDeviceType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.AdvqHits.ADVQ_HITS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class AdvqHitsRepository {

    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<AdvqHits> mapper;
    private final Set<Field<?>> fieldsToRead;

    @Autowired
    public AdvqHitsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        mapper = createAdvqHitsMapper();
        fieldsToRead = mapper.getFieldsToRead();
    }

    private static JooqMapperWithSupplier<AdvqHits> createAdvqHitsMapper() {
        return JooqMapperWithSupplierBuilder.builder(AdvqHits::new)
                .map(property(AdvqHits.REGION_ID, ADVQ_HITS.REGION_ID))
                .map(convertibleProperty(AdvqHits.DEVICE_TYPE, ADVQ_HITS.DEVICE_TYPE,
                        AdvqHitsDeviceType::fromSource, AdvqHitsDeviceType::toSource))
                .map(property(AdvqHits.SHOWS, ADVQ_HITS.SHOWS))
                .build();
    }

    public List<AdvqHits> getAdvqHits(Collection<Long> regionIds, Collection<AdvqHitsDeviceType> deviceTypes) {
        return dslContextProvider.ppcdict()
                .select(fieldsToRead)
                .from(ADVQ_HITS)
                .where(ADVQ_HITS.REGION_ID.in(regionIds))
                .and(ADVQ_HITS.DEVICE_TYPE.in(mapList(deviceTypes, AdvqHitsDeviceType::toSource)))
                .fetch(mapper::fromDb);
    }

    public void deleteAllAdvqHits() {
        dslContextProvider.ppcdict()
                .deleteFrom(ADVQ_HITS)
                .execute();
    }

    public void addAdvqHits(Collection<AdvqHits> advqHits) {
        new InsertHelper<>(dslContextProvider.ppcdict(), ADVQ_HITS)
                .addAll(mapper, advqHits)
                .executeIfRecordsAdded();
    }
}
