package ru.yandex.direct.core.entity.statistics.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Row2;
import org.jooq.conf.ParamType;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.statistics.container.AdGroupIdAndPhraseIdPair;
import ru.yandex.direct.core.entity.statistics.container.ChangedPhraseIdInfo;
import ru.yandex.direct.core.entity.statistics.container.ProcessedAuctionStat;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.dbschema.ppc.tables.Bids.BIDS;
import static ru.yandex.direct.dbschema.ppc.tables.BsAuctionStat.BS_AUCTION_STAT;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement;

@Repository
public class BsAuctionStatRepository {
    private final DslContextProvider dslContextProvider;

    private static final Set<Field<?>> BS_AUCTION_STAT_PRIMARY_KEYS = Set.of(
            BS_AUCTION_STAT.PID, BS_AUCTION_STAT.PHRASE_ID);

    private static final List<Field<?>> BS_AUCTION_STAT_FIELDS = List.of(
            BS_AUCTION_STAT.PID, BS_AUCTION_STAT.PHRASE_ID, BS_AUCTION_STAT.STATTIME, BS_AUCTION_STAT.CLICKS,
            BS_AUCTION_STAT.PCLICKS, BS_AUCTION_STAT.SHOWS, BS_AUCTION_STAT.PSHOWS, BS_AUCTION_STAT.RANK);

    private static final int INSERT_CHUNK_SIZE = 1000;
    private static final int UPDATE_PHRASEID_CHUNK_SIZE = 200;

    private static final Map<Field<?>, Field<?>> FIELDS_TO_MYSQLDSL_VALUES = getFieldsToMysqldslValues();

    public BsAuctionStatRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Возвращает неиспользуемые пары id группы и фразы с момента {@param statTimeLimit}, не больше чем
     * {@param limit} штук
     */
    public List<AdGroupIdAndPhraseIdPair> getUnusedIds(int shard, LocalDateTime statTimeLimit, int limit) {
        return dslContextProvider.ppc(shard)
                .select(BS_AUCTION_STAT.PID, BS_AUCTION_STAT.PHRASE_ID)
                .from(BS_AUCTION_STAT)
                .leftJoin(BIDS).on(BIDS.PID.eq(BS_AUCTION_STAT.PID).and(BIDS.PHRASE_ID.eq(BS_AUCTION_STAT.PHRASE_ID)))
                .where(BIDS.ID.isNull())
                .and(BS_AUCTION_STAT.STATTIME.lessThan(statTimeLimit))
                .limit(limit)
                .fetch(record -> new AdGroupIdAndPhraseIdPair(record.get(BS_AUCTION_STAT.PID),
                        record.get(BS_AUCTION_STAT.PHRASE_ID)));
    }

    /**
     * Удаляет записи из таблицы по id если они не были использованы с момента {@param statTimeLimit}
     */
    public int deleteUnusedByIds(int shard, List<AdGroupIdAndPhraseIdPair> unusedIds, LocalDateTime statTimeLimit) {
        return deleteUnusedByIds(dslContextProvider.ppc(shard), unusedIds, statTimeLimit);
    }

    int deleteUnusedByIds(DSLContext dslContext, List<AdGroupIdAndPhraseIdPair> unusedIds,
                          LocalDateTime statTimeLimit) {
        Condition deleteCondition = StreamEx.of(unusedIds)
                .mapToEntry(AdGroupIdAndPhraseIdPair::getAdGroupId, AdGroupIdAndPhraseIdPair::getPhraseId)
                .mapKeyValue((pid, phraseId) -> BS_AUCTION_STAT.PID.eq(pid).and(BS_AUCTION_STAT.PHRASE_ID.eq(phraseId)))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        return dslContext
                .deleteFrom(BS_AUCTION_STAT)
                .where(BS_AUCTION_STAT.STATTIME.lessThan(statTimeLimit)
                        .and(deleteCondition))
                .execute();
    }

    public void updateBsAuctionStat(int shard, List<ProcessedAuctionStat> processedAuctionStatList) {
        Iterable<List<ProcessedAuctionStat>> chunks = Iterables.partition(processedAuctionStatList, INSERT_CHUNK_SIZE);
        for (var chunk : chunks) {
            updateBsAuctionStatChunk(shard, chunk);
        }
    }

    /**
     * Обновить PhraseID в таблице. Конфликты Primary key игнорируются
     */
    public void updatePhraseId(int shard, Collection<ChangedPhraseIdInfo> updateInfo) {
        Iterables.partition(updateInfo, UPDATE_PHRASEID_CHUNK_SIZE).forEach(chunk -> updatePhraseIdChunk(shard, chunk));
    }

    private void updatePhraseIdChunk(int shard, Iterable<ChangedPhraseIdInfo> updateInfo) {
        Map<ULong, ULong> changes = new HashMap<>();
        List<Row2<Long, ULong>> changedIds = new ArrayList<>();

        for (var item : updateInfo) {
            changes.put(ULong.valueOf(item.getOldPhraseId()), ULong.valueOf(item.getNewPhraseId()));
            changedIds.add(row(item.getAdGroupId(), ULong.valueOf(item.getOldPhraseId())));
        }
        Field<ULong> phraseIdCase = makeCaseStatement(BS_AUCTION_STAT.PHRASE_ID, BS_AUCTION_STAT.PHRASE_ID, changes);
        DSLContext dslContext = dslContextProvider.ppc(shard);
        String query = dslContext.update(BS_AUCTION_STAT)
                .set(BS_AUCTION_STAT.STATTIME, BS_AUCTION_STAT.STATTIME)
                .set(BS_AUCTION_STAT.PHRASE_ID, phraseIdCase)
                .where(row(BS_AUCTION_STAT.PID, BS_AUCTION_STAT.PHRASE_ID).in(changedIds))
                // sad, but true: jooq не умеет UPDATE IGNORE https://github.com/jOOQ/jOOQ/issues/4529
                .getSQL(ParamType.INLINED)
                .replaceFirst(" ", " IGNORE ");

        dslContext.execute(query);
    }

    public Map<Long, List<ProcessedAuctionStat>> getAdGroupStat(int shard, Collection<Long> adGroupsIds) {
        Map<Long, List<ProcessedAuctionStat>> adgroupStat = new HashMap<>();
        if (adGroupsIds.isEmpty()) {
            return adgroupStat;
        }
        dslContextProvider.ppc(shard)
                .select(BS_AUCTION_STAT_FIELDS)
                .from(BS_AUCTION_STAT)
                .where(BS_AUCTION_STAT.PID.in(adGroupsIds))
                .forEach(record -> {
                    var stat = new ProcessedAuctionStat.Builder()
                                .withPid(record.get(BS_AUCTION_STAT.PID))
                                .withPhraseId(record.get(BS_AUCTION_STAT.PHRASE_ID).toBigInteger())
                                .withClicks(record.get(BS_AUCTION_STAT.CLICKS))
                                .withShows(record.get(BS_AUCTION_STAT.SHOWS))
                                .withPclicks(record.get(BS_AUCTION_STAT.PCLICKS))
                                .withPshows(record.get(BS_AUCTION_STAT.PSHOWS))
                                .withRank(record.get(BS_AUCTION_STAT.RANK))
                                .withPshows(record.get(BS_AUCTION_STAT.PSHOWS))
                                .build();
                    adgroupStat.computeIfAbsent(stat.getPid(), v -> new ArrayList<>()).add(stat);
                });
        return adgroupStat;
    }

    private void updateBsAuctionStatChunk(int shard, List<ProcessedAuctionStat> processedAuctionStatList) {
        var insertStep = dslContextProvider.ppc(shard)
                .insertInto(BS_AUCTION_STAT, BS_AUCTION_STAT_FIELDS);
        processedAuctionStatList
                .forEach(processedAuctionStat -> insertStep.values(getValues(processedAuctionStat)));

        insertStep.onDuplicateKeyUpdate().set(FIELDS_TO_MYSQLDSL_VALUES).execute();
    }

    private static Map<Field<?>, Field<?>> getFieldsToMysqldslValues() {
        return BS_AUCTION_STAT_FIELDS.stream()
                .filter(field -> !BS_AUCTION_STAT_PRIMARY_KEYS.contains(field))
                .collect(toMap(
                        field -> field,
                        MySQLDSL::values
                ));
    }

    private List<?> getValues(ProcessedAuctionStat processedAuctionStat) {
        return List.of(processedAuctionStat.getPid(), ULong.valueOf(processedAuctionStat.getPhraseId()),
                processedAuctionStat.getStatTime(), processedAuctionStat.getClicks(),
                processedAuctionStat.getPclicks(), processedAuctionStat.getShows(), processedAuctionStat.getPshows(),
                processedAuctionStat.getRank());
    }
}
