package ru.yandex.direct.core.entity.statistics.repository;

import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.statistics.container.ClusterToFreshnessTimestamp;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytcomponents.repository.StatsDynClusterFreshnessRepository;
import ru.yandex.direct.ytcomponents.repository.YtClusterFreshnessRepository;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.direct.ytwrapper.model.YtOperator;

import static java.util.Collections.shuffle;
import static java.util.Comparator.comparingLong;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.db.PpcPropertyNames.ORDER_STAT_DISABLED_CLUSTERS;
import static ru.yandex.direct.common.db.PpcPropertyNames.ORDER_STAT_SKIP_SHARDS;
import static ru.yandex.direct.grid.schema.yt.Tables.ORDERSTAT_BS;

/**
 * Выбирает кластер на основе свежести таблицы {@link ru.yandex.direct.grid.schema.yt.tables.OrderstatBs} и таблицы
 * {@link ru.yandex.direct.grid.schema.yt.tables.CampaignstableDirect}
 * 1) все доступные кластеры, в которых таблицы директа не старее суток
 * 2) все доступные кластеры, в которых OrderStat свежее, чем при прошлом запуске(userAttribute
 * last_sync_time_bs-chevent-log больше)
 * 3) из пересечения п1 и п2 кластер с самой свежей OrderStat
 */
@Repository
@ParametersAreNonnullByDefault
public class OrderStatClusterChooseRepository {

    private static final Logger logger = LoggerFactory.getLogger(OrderStatClusterChooseRepository.class);

    private final YtClusterFreshnessRepository ytClusterFreshnessRepository;
    private final StatsDynClusterFreshnessRepository statsDynClusterFreshnessRepository;
    private final YtProvider ytProvider;
    private volatile Collection<YtCluster> clusters;
    private final long syncTablesFreshnessThresholdSec;
    private final PpcProperty<Set<String>> disabledClustersProperty;
    private final PpcProperty<List<Integer>> skipShardsProperty;

    OrderStatClusterChooseRepository(
            YtClusterFreshnessRepository ytClusterFreshnessRepository,
            StatsDynClusterFreshnessRepository statsDynClusterFreshnessRepository,
            YtProvider ytProvider,
            DirectYtDynamicConfig directYtDynamicConfig,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.ytClusterFreshnessRepository = ytClusterFreshnessRepository;
        this.statsDynClusterFreshnessRepository = statsDynClusterFreshnessRepository;
        this.ytProvider = ytProvider;
        this.clusters = directYtDynamicConfig.getOrderStatClusters();
        this.syncTablesFreshnessThresholdSec = directYtDynamicConfig.getFreshnessThresholdForOrderStat().toSeconds();
        this.disabledClustersProperty = ppcPropertiesSupport.get(ORDER_STAT_DISABLED_CLUSTERS);
        this.skipShardsProperty = ppcPropertiesSupport.get(ORDER_STAT_SKIP_SHARDS);
    }

    @Nonnull
    public List<ClusterToFreshnessTimestamp> getClustersByPriority(long previousBsCheventLog) {
        var disabledClusters = getDisabledClusters();
        logger.info("Clusters disabled by ppc_properties {}", disabledClusters);
        Map<YtCluster, Operators> clusterToOperators = getClusterOperators().entrySet().stream()
                .filter(e -> !disabledClusters.contains(e.getKey()))
                .collect(toMap(Map.Entry::getKey, Map.Entry::getValue));
        var suitableClusters = getSuitableClusters(previousBsCheventLog, clusterToOperators);

        shuffle(suitableClusters);
        suitableClusters.sort(comparingLong(ClusterToFreshnessTimestamp::getTimestamp).reversed());
        return suitableClusters;
    }

     Set<YtCluster> getDisabledClusters() {
        return disabledClustersProperty.getOrDefault(Set.of())
                .stream()
                .map(cluster -> {
                    try {
                        return YtCluster.parse(cluster);
                    } catch (RuntimeException e) {
                        logger.error("Failed to parse cluster {} from ppc property", cluster);
                        return null;
                    }
                })
                .filter(Objects::nonNull)
                .collect(toSet());
    }

    /**
     * Выбирает пересечение кластеров для таблиц директа и для таблицы
     * {@link ru.yandex.direct.grid.schema.yt.tables.OrderstatBs}
     *
     * @return список кластеров и их last_sync_time_bs-chevent-log
     */
    List<ClusterToFreshnessTimestamp> getSuitableClusters(long previousBsCheventLog,
                                                          Map<YtCluster, Operators> clustersToOperators) {
        Set<YtCluster> syncTablesFreshClusters = loadSyncTablesFreshClusters(clustersToOperators);
        List<ClusterToFreshnessTimestamp> bsClustersByPriority =
                loadBsTablesFreshClusters(previousBsCheventLog, clustersToOperators);
        return bsClustersByPriority.stream()
                .filter(clusterToFreshnessTimestamp ->
                        syncTablesFreshClusters.contains(clusterToFreshnessTimestamp.getCluster()))
                .collect(toList());
    }

    /**
     * Выбирает все доступные кластера, на которых таблицы директа для всех шардов свежее now - {@link -
     * #syncTablesFreshnessThresholdSec}. Если таких нет, то берутся кластера с максимальным значением свежести.
     */
    Set<YtCluster> loadSyncTablesFreshClusters(Map<YtCluster, Operators> clustersToOperators) {
        long minSyncTime = System.currentTimeMillis() - syncTablesFreshnessThresholdSec * 1000;

        List<ClusterToFreshnessTimestamp> clustersToFreshness = clustersToOperators.entrySet().stream()
                .map(e -> getSyncTablesClusterFreshnessTimestamp(e.getKey(), e.getValue().ytDynamicOperator))
                .filter(Objects::nonNull)
                .collect(toList());

        Set<YtCluster> freshClusters = clustersToFreshness.stream()
                .filter(clusterToFreshnessTimestamp -> clusterToFreshnessTimestamp.getTimestamp() > minSyncTime)
                .map(ClusterToFreshnessTimestamp::getCluster)
                .collect(toSet());

        if (!freshClusters.isEmpty()) {
            return freshClusters;
        }

        // если ни один кластер не удовлетворяет условию свежести (например из-за какого-то одного шарда),
        // то берем все кластера с максимальным значением свежести, несмотря на условие
        long maxFreshness = clustersToFreshness.stream()
                .mapToLong(ClusterToFreshnessTimestamp::getTimestamp)
                .max()
                .orElse(0L);

        return clustersToFreshness.stream()
                .filter(clusterToFreshness -> clusterToFreshness.getTimestamp() == maxFreshness)
                .map(ClusterToFreshnessTimestamp::getCluster)
                .collect(toSet());
    }

    // для логирования; отдельной функцией, чтобы можно было замокать
    String getSkipShardsPropertyName() {
        return skipShardsProperty.getName().getName();
    }

    Set<Integer> getShardsToSkip() {
        return skipShardsProperty.getOrDefault(List.of()).stream().collect(toSet());
    }

    /**
     * Возвращает кластер вместе с минимальным временем синхронизации таблиц директа среди всех шардов,
     * или null, если кластер недоступен
     */
    @Nullable
    private ClusterToFreshnessTimestamp getSyncTablesClusterFreshnessTimestamp(YtCluster ytCluster,
                                                                               YtDynamicOperator ytDynamicOperator) {
        if (!getShardsToSkip().isEmpty()) {
            logger.info("going to skip shards from property {}: {}", getSkipShardsPropertyName(), getShardsToSkip());
        }
        Map<Integer, Long> shardToTimestamp =
                ytClusterFreshnessRepository.loadShardToTimestamp(ytDynamicOperator, getShardsToSkip());

        if (shardToTimestamp == null) {
            return null;
        }

        long minFreshness = shardToTimestamp.values().stream().mapToLong(i -> i).min().orElse(0L);
        return new ClusterToFreshnessTimestamp(ytCluster, minFreshness);
    }

    /**
     * Выбирает все доступные кластера, на которых у таблицы
     * {@link ru.yandex.direct.grid.schema.yt.tables.OrderstatBs} значение аттрибута last_sync_time_bs-chevent-log
     * больше, чем previousBsSyncTime
     *
     * @param previousBsSyncTime значение аттрибута, с которым выгружались данные в предыдущий раз
     * @return список кластеров, удовлетворяющих условиям, и их значения аттрибута last_sync_time_bs-chevent-log
     */
    List<ClusterToFreshnessTimestamp> loadBsTablesFreshClusters(long previousBsSyncTime,
                                                                Map<YtCluster, Operators> clustersToOperators) {
        return clustersToOperators.entrySet().stream()
                .map(e -> {
                    var cluster = e.getKey();
                    var operator = e.getValue().ytOperator;
                    ZonedDateTime clusterFreshnessTime;
                    try {
                        clusterFreshnessTime =
                                statsDynClusterFreshnessRepository.getClusterFreshnessTimeForTable(operator,
                                        ORDERSTAT_BS);
                    } catch (RuntimeException ex) {
                        logger.warn("Can't get OrderStat table attribute on cluster {}: {}", cluster, ex);
                        return null;
                    }
                    return new ClusterToFreshnessTimestamp(cluster, clusterFreshnessTime.toEpochSecond());
                })
                .filter(Objects::nonNull)
                .filter(clusterToFreshnessTimestamp -> clusterToFreshnessTimestamp.getTimestamp() > previousBsSyncTime)
                .collect(toList());
    }

    Map<YtCluster, Operators> getClusterOperators() {
        var clustersToOperators = new EnumMap<YtCluster, Operators>(YtCluster.class);
        clusters.forEach(
                cluster -> {
                    YtOperator ytOperator = null;
                    YtDynamicOperator ytDynamicOperator = null;
                    try {
                        ytOperator = ytProvider.getOperator(cluster);
                        ytDynamicOperator = ytProvider.getDynamicOperator(cluster);
                    } catch (RuntimeException ex) {
                        logger.error("Can't init operator for cluster {}", cluster, ex);
                    }
                    if (Objects.nonNull(ytOperator) && Objects.nonNull(ytDynamicOperator)) {
                        clustersToOperators.put(cluster, new Operators(ytOperator, ytDynamicOperator));
                    }
                }
        );
        return clustersToOperators;
    }

    static class Operators {
        YtOperator ytOperator;
        YtDynamicOperator ytDynamicOperator;

        Operators(YtOperator ytOperator, YtDynamicOperator ytDynamicOperator) {
            this.ytOperator = ytOperator;
            this.ytDynamicOperator = ytDynamicOperator;
        }
    }

}
