package ru.yandex.direct.core.entity.statistics.repository;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Select;

import ru.yandex.direct.core.entity.statistics.container.OrderIdWithPeriod;
import ru.yandex.direct.core.entity.statistics.model.SpentInfo;
import ru.yandex.direct.core.entity.statistics.model.StatisticsDateRange;
import ru.yandex.direct.grid.schema.yt.tables.CaesarorderinfoBs;
import ru.yandex.direct.grid.schema.yt.tables.OrderstatdayBs;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.DateTimeUtils;
import ru.yandex.direct.utils.TimeProvider;
import ru.yandex.direct.ytcomponents.service.OrderStatDayDynContextProvider;
import ru.yandex.direct.ytcomponents.service.StatsDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static com.google.common.base.Preconditions.checkArgument;
import static java.time.temporal.ChronoUnit.DAYS;
import static org.jooq.impl.DSL.max;
import static org.jooq.impl.DSL.min;
import static org.jooq.impl.DSL.noCondition;
import static org.jooq.impl.DSL.sum;
import static ru.yandex.direct.grid.schema.yt.Tables.CAESARORDERINFO_BS;
import static ru.yandex.direct.grid.schema.yt.Tables.ORDERSTATDAY_BS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.DateTimeUtils.instantToMoscowDate;
import static ru.yandex.direct.utils.DateTimeUtils.startOfDayInMoscowTime;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.YtTableUtils.instantValueGetter;
import static ru.yandex.direct.ytwrapper.YtTableUtils.longValueGetter;
import static ru.yandex.direct.ytwrapper.YtTableUtils.toEntryStream;

/**
 * Репозиторий для работы с динамической таблицей {@link OrderstatdayBs},
 * хранящей статистику БК в разрезе дата/заказ/тип площадки (BSDEV-73553)
 * Даты хранятся в UpdateTime как UTC timestamp начало суток по Москве
 * В интерфейсе YT: https://yt.yandex-team.ru/arnold/navigation?path=//home/yabs/stat/OrderStatDay&offsetMode=key
 * Таблица есть почти на всех кластерах.
 * <p>
 * БК уведомлены о том, что директ использует указанную таблицу и обещают не изменять ее без предупреждения
 * https://st.yandex-team.ru/BSDEV-73553#5cdadca7b1a407002081fe90
 */
public class OrderStatDayRepository {

    private static final OrderstatdayBs STAT = ORDERSTATDAY_BS.as("S");
    private static final Field<Long> ORDER_ID = aliased(STAT.ORDER_ID);
    private static final Field<Long> UPDATE_TIME = aliased(STAT.UPDATE_TIME);
    private static final Field<Long> MAX_UPDATE_TIME = max(STAT.UPDATE_TIME).as("MaxUpdateTime");
    private static final Field<Long> MIN_UPDATE_TIME = min(STAT.UPDATE_TIME).as("MinUpdateTime");
    private static final Field<BigDecimal> SUM_SESSION_NUM = sum(STAT.SESSION_NUM).as("SumSessionNum");
    private static final Field<BigDecimal> SUM_COST_CUR = sum(STAT.COST_CUR).as("SumCostCur");
    private static final Field<BigDecimal> SUM_CLICKS = sum(STAT.CLICKS).as("SumClicks");

    private static final CaesarorderinfoBs INFO = CAESARORDERINFO_BS.as("INFO");
    private static final Field<Long> TAX_ID = aliased(INFO.TAX_ID);
    private static final Field<Long> CURRENCY_ID = aliased(INFO.CURRENCY_ID);

    private final StatsDynContextProvider dynContextProvider;
    private final TimeProvider timeProvider;

    public OrderStatDayRepository(OrderStatDayDynContextProvider dynContextProvider) {
        this(dynContextProvider, new TimeProvider());
    }

    OrderStatDayRepository(OrderStatDayDynContextProvider dynContextProvider, TimeProvider timeProvider) {
        this.dynContextProvider = dynContextProvider;
        this.timeProvider = timeProvider;
    }

    /**
     * Возвращает словарь OrderID -> последняя дата, за которую есть статистика
     * для переданного списка OrderID {@param orderIds}
     */
    public Map<Long, LocalDate> getOrdersToLastDay(Collection<Long> orderIds) {

        if (orderIds.isEmpty()) {
            return Collections.emptyMap();
        }

        try (TraceProfile ignored = Trace.current().profile("orderstat:getLastDayOfCampaigns", "yt",
                orderIds.size())) {

            Select query = YtDSL.ytContext()
                    .select(ORDER_ID, MAX_UPDATE_TIME)
                    .from(STAT)
                    .where(ORDER_ID.in(orderIds))
                    .groupBy(ORDER_ID);
            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);

            return toEntryStream(rowset, ORDER_ID, MAX_UPDATE_TIME)
                    .mapKeys(UnversionedValue::longValue)
                    .mapValues(UnversionedValue::longValue)
                    .mapValues(Instant::ofEpochSecond)
                    .mapValues(DateTimeUtils::instantToMoscowDate)
                    .toMap();
        }
    }

    /**
     * Возвращает словарь OrderID -> суммарное количество сессий за указанное количество дней
     * для переданного списка OrderID {@param orderIds}
     */
    public Map<Long, Long> getOrdersSessionsNumSum(Collection<Long> orderIds, int minusDays) {
        if (orderIds.isEmpty()) {
            return Collections.emptyMap();
        }

        Long nDaysAgoTimestamp = timeProvider.instantNow().minus(minusDays, DAYS).getEpochSecond();

        try (TraceProfile ignored = Trace.current().profile("orderstat:getOrderSessionsNumForWeek", "yt",
                orderIds.size())) {

            Select query = YtDSL.ytContext()
                    .select(ORDER_ID, SUM_SESSION_NUM)
                    .from(STAT)
                    .where(ORDER_ID.in(orderIds)
                            .and(STAT.UPDATE_TIME.greaterOrEqual(nDaysAgoTimestamp)))
                    .groupBy(ORDER_ID);
            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);
            TableSchema schema = rowset.getSchema();

            return toEntryStream(rowset, ORDER_ID, longValueGetter(schema, SUM_SESSION_NUM))
                    .mapKeys(UnversionedValue::longValue)
                    .nonNullValues()
                    .toMap();
        }
    }

    /**
     * Возвращает словарь OrderID -> суммарное количество кликов за сегодня
     * для переданного списка OrderID {@param orderIds}
     */
    public Map<Long, Long> getOrdersClicksToday(Collection<Long> orderIds) {
        if (orderIds.isEmpty()) {
            return Collections.emptyMap();
        }

        LocalDate today = timeProvider.now().toLocalDate();
        Long leftBorderTimestamp = startOfDayInMoscowTime(today);
        Long rightBorderTimestamp = startOfDayInMoscowTime(today.plusDays(1));

        try (TraceProfile ignored = Trace.current().profile("orderstat:getOrdersClicksToday", "yt", orderIds.size())) {

            Select query = YtDSL.ytContext()
                    .select(ORDER_ID, SUM_CLICKS)
                    .from(STAT)
                    .where(ORDER_ID.in(orderIds))
                    .and(STAT.UPDATE_TIME.ge(leftBorderTimestamp))
                    .and(STAT.UPDATE_TIME.lt(rightBorderTimestamp))
                    .groupBy(ORDER_ID);
            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);
            TableSchema schema = rowset.getSchema();

            return toEntryStream(rowset, ORDER_ID, longValueGetter(schema, SUM_CLICKS))
                    .mapKeys(UnversionedValue::longValue)
                    .nonNullValues()
                    .toMap();
        }
    }

    /**
     * Возвращает словарь OrderID -> первые и последние даты, за которые есть статистика
     * для переданного списка OrderID {@param orderIds}
     */
    public Map<Long, StatisticsDateRange> getOrdersFirstLastDay(Collection<Long> orderIds) {

        if (orderIds.isEmpty()) {
            return Collections.emptyMap();
        }

        try (TraceProfile ignored = Trace.current().profile("orderstat:getOrdersFirstLastDay", "yt",
                orderIds.size())) {

            Select query = YtDSL.ytContext().
                    select(ORDER_ID, MAX_UPDATE_TIME, MIN_UPDATE_TIME)
                    .from(STAT)
                    .where(ORDER_ID.in(orderIds))
                    .groupBy(ORDER_ID);
            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);
            List<UnversionedRow> rows = rowset.getRows();

            TableSchema schema = rowset.getSchema();
            Map<Long, StatisticsDateRange> result = new HashMap<>();
            for (UnversionedRow row : rows) {
                Long orderId = longValueGetter(schema, ORDER_ID).apply(row);
                LocalDate maxDate = instantToMoscowDate(instantValueGetter(schema, MAX_UPDATE_TIME).apply(row));
                LocalDate minDate = instantToMoscowDate(instantValueGetter(schema, MIN_UPDATE_TIME).apply(row));
                result.put(orderId, new StatisticsDateRange(minDate, maxDate));
            }

            return result;
        }
    }

    /**
     * Возвращает число - количество дней за которые есть статистика в указанном диапазоне дат
     * для переданного списка OrderID {@param orderIds}, начальной даты startDate {@param startDate} и
     * конечной даты {@param endDate}
     */
    public Long getOrdersDaysNum(Collection<Long> orderIds, LocalDate startDate, LocalDate endDate) {
        if (orderIds.isEmpty()) {
            return null;
        }
        if (startDate == null || endDate == null) {
            throw new NullPointerException("startDate and endDate must be not null");
        }

        Long startTimestamp = startOfDayInMoscowTime(startDate);
        Long endTimestamp = startOfDayInMoscowTime(endDate);

        try (TraceProfile ignored = Trace.current().profile("orderstat:getOrdersDaysNum", "yt",
                orderIds.size())) {

            Select query = YtDSL.ytContext()
                    .select(UPDATE_TIME)
                    .from(STAT)
                    .where(STAT.ORDER_ID.in(orderIds).and(UPDATE_TIME.between(startTimestamp, endTimestamp)))
                    .groupBy(UPDATE_TIME);
            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);
            List<UnversionedRow> rows = rowset.getRows();
            return (long) rows.size();
        }
    }

    /**
     * Получаем stream из данных о тратах на заказах {@param orderIds} в период от {@param startDate} до
     * {@param endDate}
     */
    StreamEx<SpentInfo> getOrdersSpentInfo(Collection<Long> orderIds, LocalDate startDate, LocalDate endDate) {

        if (orderIds.isEmpty()) {
            return StreamEx.empty();
        }

        Long leftBorderTimestamp = startOfDayInMoscowTime(startDate);
        Long rightBorderTimestamp = startOfDayInMoscowTime(endDate);
        Condition condition = ORDER_ID.in(orderIds)
                .and(UPDATE_TIME.between(leftBorderTimestamp, rightBorderTimestamp));

        try (TraceProfile ignored = Trace.current().profile("orderstat:getOrdersSpentInfo", "yt", orderIds.size())) {
            return getSpentInfos(condition);
        }
    }
    /**
     * Получаем stream из данных о тратах на заказах {@param orderIds} в период от {@param startDate} до
     * {@param endDate}
     */
    private StreamEx<SpentInfo> getOrdersSpentInfo(List<OrderIdWithPeriod> orderIdWithPeriodList) {

        if (orderIdWithPeriodList.isEmpty()) {
            return StreamEx.empty();
        }
        Condition condition = StreamEx.of(orderIdWithPeriodList)
                .map(orderIdWithPeriod -> ORDER_ID.eq(orderIdWithPeriod.getOrderId())
                        .and(UPDATE_TIME.between(startOfDayInMoscowTime(orderIdWithPeriod.getStart()),
                                startOfDayInMoscowTime(orderIdWithPeriod.getFinish()))))
                .foldLeft(noCondition(), Condition::or);

        try (TraceProfile ignored = Trace.current().profile("orderstat:getOrdersSpentInfo", "yt",
                orderIdWithPeriodList.size())) {
            return getSpentInfos(condition);
        }
    }

    private StreamEx<SpentInfo> getSpentInfos(Condition condition) {
        Select query = YtDSL.ytContext()
                .select(ORDER_ID, UPDATE_TIME, TAX_ID, CURRENCY_ID, SUM_COST_CUR)
                .from(STAT.join(INFO)
                        .on(ORDER_ID.eq(INFO.ORDER_ID)))
                .where(condition)
                .groupBy(ORDER_ID, UPDATE_TIME, TAX_ID, CURRENCY_ID);

        UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);
        TableSchema schema = rowset.getSchema();

        Function<UnversionedRow, SpentInfo> valueGetter = row -> getSpentInfoFromRow(schema, row);

        return StreamEx.of(rowset.getRows())
                .map(valueGetter);
    }


    private SpentInfo getSpentInfoFromRow(TableSchema schema, UnversionedRow row) {
        return new SpentInfo(
                longValueGetter(schema, ORDER_ID).apply(row),
                instantValueGetter(schema, UPDATE_TIME).apply(row),
                nvl(longValueGetter(schema, SUM_COST_CUR).apply(row), 0L),
                longValueGetter(schema, TAX_ID).apply(row),
                longValueGetter(schema, CURRENCY_ID).apply(row)
        );
    }

    /**
     * Возвращает словарь orderId из {@param orderIds} -> максимальнная сумма трат за последние {@param minusDays} дней,
     * не включая сегодняшний. То есть: спрашиваем в пн - смотрим на траты за всю предыдущую неделю с пн по вт
     * Сумма - как CostCur у БК (в миллионных долях)
     */
    public Map<Long, Long> getOrderIdToMaxSum(Collection<Long> orderIds, int minusDays) {
        LocalDate today = timeProvider.now().toLocalDate();

        LocalDate startDate = today.minusDays(minusDays);
        LocalDate endDate = today.minusDays(1);

        return getOrdersSpentInfo(orderIds, startDate, endDate)
                .mapToEntry(SpentInfo::getOrderId, SpentInfo::getCost)
                .toMap(Long::max);
    }

    /**
     * Возвращает словарь дата из периода от {@param startDate} до {@param endDate} -> суммарные траты
     * по заказам {@param orderIds}
     */
    public Map<LocalDate, BigDecimal> getDateToSumSpent(Collection<Long> orderIds, LocalDate startDate,
                                                        LocalDate endDate) {
        return getOrdersSpentInfo(orderIds, startDate, endDate)
                .mapToEntry(SpentInfo::getStatDate, SpentInfo::getCost)
                .mapKeys(DateTimeUtils::instantToMoscowDate)
                .mapValues(BigDecimal::valueOf)
                .toMap(BigDecimal::add);
    }

    /**
     * Возвращает информацию о потраченных деньгах в день {@param day} для заказов {@param orderIds}
     */
    public Map<Long, SpentInfo> getOrdersSpent(Collection<Long> orderIds, LocalDate day) {
        return getOrdersSpentInfo(orderIds, day, day)
                .mapToEntry(SpentInfo::getOrderId, Function.identity())
                .toMap();
    }

    /**
     * Возвращает информацию о потраченных деньгах для каждого заказа по соответствующему периоду
     */
    public Map<Long, List<SpentInfo>> getOrdersSpent(List<Long> orderIds, List<LocalDate> startDates,
                                                     List<LocalDate> finishDates) {
        checkArgument(orderIds.size() == startDates.size() && orderIds.size() == finishDates.size(),
                "For every order should be date of start and date of finish");

        List<OrderIdWithPeriod> orderIdWithPeriodList = IntStreamEx.range(0, orderIds.size())
                .boxed()
                .map(index -> new OrderIdWithPeriod(orderIds.get(index), startDates.get(index), finishDates.get(index)))
                .toList();

        Map<Long, OrderIdWithPeriod> orderIdWithPeriodByOrderId = listToMap(orderIdWithPeriodList,
                OrderIdWithPeriod::getOrderId, Function.identity());

        List<SpentInfo> ordersSpentInfo = getOrdersSpentInfo(orderIdWithPeriodList).toList();

        return StreamEx.of(ordersSpentInfo)
                .mapToEntry(SpentInfo::getOrderId, Function.identity())
                .filterKeyValue((orderId, spentInfo) -> {
                    OrderIdWithPeriod orderIdWithPeriod = orderIdWithPeriodByOrderId.get(orderId);
                    LocalDate statDate = DateTimeUtils.instantToMoscowDate(spentInfo.getStatDate());
                    return !statDate.isBefore(orderIdWithPeriod.getStart()) && !statDate.isAfter(orderIdWithPeriod.getFinish());
                })
                .grouping();
    }

}
