package ru.yandex.direct.core.entity.statistics.repository;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import org.jooq.Field;
import org.jooq.Select;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.statistics.model.FraudAndGiftClicks;
import ru.yandex.direct.grid.schema.yt.tables.OrderstatfraudBs;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.ytcomponents.service.OrderStatFraudDynContextProvider;
import ru.yandex.direct.ytcomponents.service.StatsDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static org.jooq.impl.DSL.sum;
import static org.jooq.impl.DSL.val;
import static ru.yandex.direct.grid.schema.yt.Tables.ORDERSTATFRAUD_BS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.longValueGetter;

/**
 * Репозиторий для работы с {@link OrderstatfraudBs}, хранящей стастику БК по фроду (BSDEV-73748)
 * Даты хранятся в UpdateTime как UTC timestamp начало суток по Москве
 * В интерфейсе YT: https://yt.yandex-team.ru/arnold/navigation?path=//home/yabs/stat/OrderStatFraud&offsetMode=key
 * Таблица есть почти на всех кластерах.
 */
@Repository
public class OrderStatFraudRepository {

    private static final OrderstatfraudBs STAT = ORDERSTATFRAUD_BS.as("S");
    private static final Field<BigDecimal> FRAUD_CLICKS_SUM = sum(STAT.FRAUD_CLICKS).as("FraudClicksSum");
    private static final Field<BigDecimal> GIFT_CLICKS_SUM = sum(STAT.GIFT_CLICKS).as("GiftClicksSum");
    private static final Field<BigDecimal> FRAUD_SHOWS_GENERAL_SUM =
            sum(STAT.FRAUD_SHOWS_GENERAL).as("FraudShowsGeneralSum");
    private static final Field<BigDecimal> FRAUD_SHOWS_SOPHISTICATED_SUM =
            sum(STAT.FRAUD_SHOWS_SOPHISTICATED).as("FraudShowsSophisticatedSum");

    private static final Field<Long> ZERO = val(0L);

    private final StatsDynContextProvider dynContextProvider;

    @Autowired
    public OrderStatFraudRepository(OrderStatFraudDynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    /**
     * Возвращает пару (сумма FraudСlicks; сумма GiftClicks) за даты между {@param startDate} и
     * {@param endDate} включительно для переданного списка OrderID {@param orderIds}.
     * Если статистики нет, то возвращаются нули.
     */
    public FraudAndGiftClicks getFraudClicks(Collection<Long> orderIds, LocalDate startDate, LocalDate endDate) {

        if (orderIds.isEmpty() || endDate.isBefore(startDate)) {
            return FraudAndGiftClicks.ZERO;
        }

        Field<Long> startTimestamp = YtDSL.toEpochSecondsAtStartOfDate(startDate);
        Field<Long> endTimestamp = YtDSL.toEpochSecondsAtStartOfDate(endDate);

        try (TraceProfile ignored = Trace.current().profile("orderstat:getFraudClicks", "", orderIds.size())) {
            Select query = YtDSL.ytContext()
                    .select(
                            FRAUD_CLICKS_SUM,
                            GIFT_CLICKS_SUM,
                            FRAUD_SHOWS_GENERAL_SUM,
                            FRAUD_SHOWS_SOPHISTICATED_SUM)
                    .from(STAT)
                    .where(STAT.ORDER_ID.in(orderIds)
                            .and(STAT.UPDATE_TIME.between(startTimestamp, endTimestamp)))
                    .groupBy(ZERO);

            UnversionedRowset rowset = dynContextProvider.getContext().executeTimeoutSafeSelect(query);
            List<UnversionedRow> rows = rowset.getRows();

            if (rows.isEmpty()) {
                return FraudAndGiftClicks.ZERO;
            }

            TableSchema schema = rowset.getSchema();
            UnversionedRow row = rows.get(0);
            Long fraudShowsGeneral = Optional.ofNullable(
                    longValueGetter(schema, FRAUD_SHOWS_GENERAL_SUM).apply(row)
            ).orElse(0L);
            Long fraudShowsSophisticated = Optional.ofNullable(
                    longValueGetter(schema, FRAUD_SHOWS_SOPHISTICATED_SUM).apply(row)
            ).orElse(0L);
            return new FraudAndGiftClicks(
                    Optional.ofNullable(longValueGetter(schema, FRAUD_CLICKS_SUM).apply(row)).orElse(0L),
                    Optional.ofNullable(longValueGetter(schema, GIFT_CLICKS_SUM).apply(row)).orElse(0L),
                    fraudShowsGeneral,
                    fraudShowsSophisticated
            );
        }
    }
}
