package ru.yandex.direct.core.entity.statistics.repository;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import org.jooq.Field;
import org.jooq.Select;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.statistics.model.ActiveOrderChanges;
import ru.yandex.direct.core.entity.statistics.model.YtHashBorders;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.grid.schema.yt.tables.CaesarorderinfoBs;
import ru.yandex.direct.grid.schema.yt.tables.CampaignstableDirect;
import ru.yandex.direct.grid.schema.yt.tables.CurrencyBs;
import ru.yandex.direct.grid.schema.yt.tables.OrderstatBs;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.context.YtDynamicContext;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static java.util.stream.Collectors.toList;
import static java.util.stream.StreamSupport.stream;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.db.PpcPropertyNames.ORDER_STAT_MAX_SUBQUERIES;
import static ru.yandex.direct.grid.schema.yt.Tables.CAESARORDERINFO_BS;
import static ru.yandex.direct.grid.schema.yt.Tables.CAMPAIGNSTABLE_DIRECT;
import static ru.yandex.direct.grid.schema.yt.Tables.CURRENCY_BS;
import static ru.yandex.direct.grid.schema.yt.Tables.ORDERSTAT_BS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.Case.of;

/**
 * Репозиторий для работы с {@link OrderstatBs}, хранящей стастику БК по показам, кликам, потраченным деньгам
 */
@Repository
public class OrderStatRepository {
    private static final Logger logger = LoggerFactory.getLogger(OrderStatRepository.class);

    private final YtProvider ytProvider;
    private final DirectYtDynamicConfig directDynamicConfig;
    private final PpcProperty<Integer> maxSubQueriesProperty;
    private static final CampaignstableDirect CAMP = CAMPAIGNSTABLE_DIRECT.as("CAMP");
    private static final CaesarorderinfoBs ORDERINFO = CAESARORDERINFO_BS.as("ORDERINFO");
    private static final OrderstatBs ORDERSTAT = ORDERSTAT_BS.as("ORDERSTAT");
    private static final CurrencyBs CURRENCY = CURRENCY_BS.as("CURRENCY");
    private static final Field<Long> ORDER_ID = aliased(CAMP.ORDER_ID);
    private static final Field<Long> CID = aliased(CAMP.CID);
    private static final Field<String> TYPE = aliased(CAMP.TYPE);
    private static final Field<Long> OLD_SHOWS = YtDSL.ytIfNull(CAMP.SHOWS, 0L).as("oldShows");
    private static final Field<Long> NEW_SHOWS = ORDERSTAT.SHOWS.as("newShows");
    private static final Field<Long> NEW_CLICKS = ORDERSTAT.CLICKS.as("newClicks");
    private static final Field<Long> OLD_SUM_SPENT = CAMP.SUM_SPENT.as("oldSumSpent");
    private static final Field<Long> NEW_SUM_SPENT = YtDSL.ytIf(CURRENCY.CURRENCY_ID.eq(0L), ORDERSTAT.COST,
            ORDERSTAT.COST_CUR.mul(CURRENCY.RATIO)).as("newSumSpent");
    private static final Field<Long> OLD_SPENT_UNITS = YtDSL.ytIf(
            CAMP.TYPE.eq(CampaignsType.internal_free.getLiteral()),
            YtDSL.ytIfNull(CAMP.SUM_SPENT_UNITS, 0L), DSL.val(0L)).as("oldSpentUnits");
    private static final Field<Long> NEW_SPENT_UNITS = YtDSL.ytIfNull(YtDSL.ytIf(
            CAMP.TYPE.eq(CampaignsType.internal_free.getLiteral()),
            YtDSL.ytSwitch(CAMP.RESTRICTION_TYPE, List.of(
                    of("shows", NEW_SHOWS),
                    of("clicks", NEW_CLICKS),
                    of("days", ORDERSTAT.DAYS)
                    // Тип 'money' для бесплатных кампаний неакутален
            ), 0L), DSL.val(0L)), 0L).as("newSpentUnits");
    private static final Field<Long> SUM = aliased(CAMP.SUM);
    private static final Field<Long> UNITS = YtDSL.ytIfNull(CAMP.RESTRICTION_VALUE, 0L).as("units");
    private static final Field<Long> WALLET_CID = aliased(CAMP.WALLET_CID);
    private static final Field<String> ARCHIVED = aliased(CAMP.ARCHIVED);
    private static final Field<Long> SHARD = aliased(CAMP.__SHARD__);

    /**
     * Поля Cost, CostCur, UpdateTime нужны для отображения в logviewer'е
     **/
    private static final Field<Long> COST = aliased(ORDERSTAT.COST);
    private static final Field<Long> COST_CUR = aliased(ORDERSTAT.COST_CUR);
    private static final Field<Long> UPDATE_TIME = aliased(ORDERSTAT.UPDATE_TIME);

    /* 67 - внутренняя реклама, 7 - директ */
    private static final List<Long> ENGINE_IDS_TO_LOAD = List.of(7L, 67L);
    private static final Duration QUERY_TIMEOUT = Duration.ofMinutes(5);

    private static final Pattern YT_HASH_EXPRESSION_REGEX =
            Pattern.compile("int\\d+\\(.+%(\\d+)\\)", Pattern.CASE_INSENSITIVE);


    public OrderStatRepository(YtProvider ytProvider,
                               PpcPropertiesSupport ppcPropertiesSupport,
                               DirectYtDynamicConfig directDynamicConfig) {
        this.ytProvider = ytProvider;
        this.directDynamicConfig = directDynamicConfig;
        // see https://st.yandex-team.ru/YTADMINREQ-18749
        this.maxSubQueriesProperty = ppcPropertiesSupport.get(ORDER_STAT_MAX_SUBQUERIES);
    }

    public long getCampaignsYtHashMaxValue(YtCluster ytCluster) {
        YtClient ytClient = ytProvider.getDynamicOperator(ytCluster).getYtClient();
        YTreeListNode schema = ytClient
                .getNode(directDynamicConfig.tables().direct().campaignsTablePath() + "/@schema")
                .join() // IGNORE-BAD-JOIN DIRECT-149116
                .listNode();

        String expression = stream(schema.spliterator(), false)
                .map(YTreeNode::mapNode)
                .filter(node -> node.get("name").isPresent() &&
                        node.get("name").get().stringValue().equals(CAMP.__HASH__.getName()))
                .map(node -> node.get("expression"))
                .filter(Optional::isPresent)
                .map(expr -> expr.get().stringValue())
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("Can't find YtHash expression"))
                .replace(" ", "");

        Matcher expressionMatcher = YT_HASH_EXPRESSION_REGEX.matcher(expression);
        if (!expressionMatcher.matches()) {
            throw new IllegalStateException("Unexpected YtHash expression: " + expression);
        }

        return Long.parseLong(expressionMatcher.group(1)) - 1;
    }

    @Nullable
    public static YtHashBorders getYtHashBorders(long workerNum, long workersCount,
                                                 long batchNum, long batchesCount,
                                                 long ytHashMaxValue) {
        long sizePerWorker = (ytHashMaxValue + workersCount) / workersCount;
        long sizePerBatch = (sizePerWorker + batchesCount - 1) / batchesCount;

        if (workerNum * sizePerWorker + batchNum * sizePerBatch > ytHashMaxValue) {
            return null;
        }

        return new YtHashBorders(sizePerWorker * workerNum + sizePerBatch * batchNum,
                Math.min(ytHashMaxValue,
                        Math.min(sizePerWorker * (workerNum + 1) - 1,
                                sizePerWorker * workerNum + sizePerBatch * (batchNum + 1) - 1)));
    }

    /**
     * Выгружает данные по кампаниям, для которых данные по показам, кликам или потраченным деньгам в таблице
     * campaigns Директа для нового интерфейса {@link CampaignstableDirect} отличаются от данных в таблице со
     * статистикой в БК {@link OrderstatBs}
     *
     * @param ytHashBorders   диапазон значений поля __hash__
     * @return список кампаний, по которым нашлись отличия, содержащие старые данные по показам, кликам,
     * деньгам и новые данные
     */
    public List<ActiveOrderChanges> getChangedActiveOrders(YtHashBorders ytHashBorders, List<YtCluster> ytClusters) {
        Select query = getChangedCampaignQuery(ytHashBorders);
        long timestampBeforeQuery = System.currentTimeMillis() / 1000;
        var maxSubQueries = maxSubQueriesProperty.getOrDefault(1);

        var dynamicContext = maxSubQueries == 0 ?
                new YtDynamicContext(ytProvider, ytClusters, QUERY_TIMEOUT)
                : new YtDynamicContext(ytProvider, ytClusters, QUERY_TIMEOUT, maxSubQueries);

        UnversionedRowset rows = dynamicContext.executeTimeoutSafeSelect(query);
        long queryDuration = System.currentTimeMillis() / 1000 - timestampBeforeQuery;
        logger.info("YT hash borders {}: query duration {} sec", ytHashBorders, queryDuration);

        return rows.getYTreeRows().stream()
                .map(r -> new ActiveOrderChanges.Builder()
                        .withOrderId(r.getLong(ORDER_ID.getName()))
                        .withCid(r.getLong(CID.getName()))
                        .withShard(r.getInt(SHARD.getName()))
                        .withType(CampaignType.fromSource(CampaignsType.valueOf(r.getString(TYPE.getName()))))
                        .withOldShows(r.getLong(OLD_SHOWS.getName()))
                        .withOldSumSpent(r.getLong(OLD_SUM_SPENT.getName()))
                        .withOldSumSpentUnits(r.getLong(OLD_SPENT_UNITS.getName()))
                        .withNewShows(r.getLong(NEW_SHOWS.getName()))
                        .withNewClicks(r.getLong(NEW_CLICKS.getName()))
                        .withNewSumSpent(r.getLong(NEW_SUM_SPENT.getName()))
                        .withNewSumSpentUnits(r.getLong(NEW_SPENT_UNITS.getName()))
                        .withSum(r.getLong(SUM.getName()))
                        .withUnits(r.getLong(UNITS.getName()))
                        .withWalletCid(r.getLong(WALLET_CID.getName()))
                        .withArchived(r.getString(ARCHIVED.getName()))
                        .withCost(r.getLong(COST.getName()))
                        .withCostCur(r.getLong(COST_CUR.getName()))
                        .withUpdateTime(r.getLong(UPDATE_TIME.getName()))
                        .build())
                .collect(toList());
    }

    Select getChangedCampaignQuery(YtHashBorders ytHashBorders) {
        return YtDSL.ytContext()
                .select(ORDER_ID, CID, SHARD, TYPE, OLD_SHOWS, OLD_SUM_SPENT, OLD_SPENT_UNITS,
                        NEW_CLICKS, NEW_SHOWS, NEW_SUM_SPENT, NEW_SPENT_UNITS,
                        SUM, UNITS, WALLET_CID, ARCHIVED, COST, COST_CUR, UPDATE_TIME)
                .from(CAMP)
                .join(ORDERINFO).on(CAMP.ORDER_ID.eq(ORDERINFO.ORDER_ID))
                .join(ORDERSTAT).on(row(ORDERINFO.EXPORT_ID, ORDERINFO.ENGINE_ID).eq(row(ORDERSTAT.EXPORT_ID,
                        ORDERSTAT.ENGINE_ID)).and(ORDERSTAT.ENGINE_ID.in(ENGINE_IDS_TO_LOAD)))
                .leftJoin(CURRENCY).on(ORDERINFO.CURRENCY_ID.eq(CURRENCY.CURRENCY_ID))
                .where(CAMP.__HASH__.between(ytHashBorders.getFirst(), ytHashBorders.getSecond()))
                .and(ORDERSTAT.SHOWS.ne(CAMP.SHOWS).or(ORDERSTAT.CLICKS.ne(CAMP.CLICKS).or(NEW_SUM_SPENT.ne(CAMP.SUM_SPENT))
                        .or(NEW_SPENT_UNITS.ne(OLD_SPENT_UNITS))));
    }
}
