package ru.yandex.direct.core.entity.statistics.service;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.base.Functions;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.statistics.model.FraudAndGiftClicks;
import ru.yandex.direct.core.entity.statistics.model.Period;
import ru.yandex.direct.core.entity.statistics.model.SpentInfo;
import ru.yandex.direct.core.entity.statistics.model.StatisticsDateRange;
import ru.yandex.direct.core.entity.statistics.repository.OrderStatDayRepository;
import ru.yandex.direct.core.entity.statistics.repository.OrderStatFraudRepository;
import ru.yandex.direct.core.entity.tax.model.TaxInfo;
import ru.yandex.direct.core.entity.tax.service.TaxHistoryService;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.utils.DateTimeUtils;
import ru.yandex.direct.utils.TimeProvider;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.tax.service.TaxHistoryUtils.getTaxInfoForDay;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Сервис для получения статистики по заказу
 */
@Service
public class OrderStatService {

    /**
     * "Начало всех времен" для статистики. В перле - Stat::Const::BEGIN_OF_TIME_FOR_STAT
     */
    public static final LocalDate BEGIN_OF_TIME_FOR_STAT = LocalDate.of(2001, 1, 1);
    public static final int MILLION = 1_000_000;
    /**
     * За сколько последних лет доступна статистика БК
     */
    private static final int BS_STAT_ALLOW_YEARS = 3;
    /**
     * Количество последних дней, за которые ищем максимальную сумму для грубого прогноза
     */
    private static final int DAYS_FOR_FORECAST_COUNT = 7;
    private static final String NOT_FOR_SANDBOX = "Сan't be used in Sandbox";
    private final OrderStatDayRepository orderStatDayRepository;
    private final OrderStatFraudRepository orderStatFraudRepository;
    private final CampaignRepository campaignRepository;
    private final TaxHistoryService taxHistoryService;
    private final ShardHelper shardHelper;
    private final EnvironmentType environmentType;
    private final TimeProvider timeProvider;

    @Autowired
    public OrderStatService(OrderStatDayRepository orderStatDayRepository,
                            OrderStatFraudRepository orderStatFraudRepository,
                            CampaignRepository campaignRepository,
                            TaxHistoryService taxHistoryService,
                            ShardHelper shardHelper,
                            EnvironmentType environmentType) {
        this(orderStatDayRepository, orderStatFraudRepository, campaignRepository,
                taxHistoryService, shardHelper, environmentType,
                new TimeProvider());
    }

    public OrderStatService(OrderStatDayRepository orderStatDayRepository,
                            OrderStatFraudRepository orderStatFraudRepository,
                            CampaignRepository campaignRepository,
                            TaxHistoryService taxHistoryService,
                            ShardHelper shardHelper,
                            EnvironmentType environmentType,
                            TimeProvider timeProvider) {
        this.orderStatDayRepository = orderStatDayRepository;
        this.orderStatFraudRepository = orderStatFraudRepository;
        this.campaignRepository = campaignRepository;
        this.taxHistoryService = taxHistoryService;
        this.shardHelper = shardHelper;
        this.environmentType = environmentType;
        this.timeProvider = timeProvider;
    }

    /**
     * Возращает словарь OrderID -> последняя дата, за которую есть статистика по переданному списку OrderID
     * {@param orderIds}.
     * Значения статистики берутся из yt - {@link OrderStatDayRepository#getOrdersFirstLastDay(Collection)}
     */
    public Map<Long, LocalDate> getLastDayOfCampaigns(Collection<Long> orderIds) {
        Map<Long, StatisticsDateRange> ordersFirstLastDay = getOrdersFirstLastDay(orderIds);
        Map<Long, LocalDate> result = new HashMap<>();
        for (Map.Entry<Long, StatisticsDateRange> entry : ordersFirstLastDay.entrySet()) {
            result.put(entry.getKey(), ifNotNull(entry.getValue(), StatisticsDateRange::getLastDate));
        }
        return result;
    }

    /**
     * Возвращает суммарное количество сессий за последние семь дней для списка OrderID.
     * Если статистики за этот период нет, то возвращаются нули.
     *
     * @param orderIds список идентификаторов заказов в БК
     * @return словарь OrderID -> суммарное количество сессий за последние семь дней
     */
    public Map<Long, Long> getOrdersSessionsNumForWeek(Collection<Long> orderIds) {
        // в песочнице отдавать настоящую статистику из продакшена мы не можем
        checkState(!environmentType.isSandbox(), NOT_FOR_SANDBOX);

        Map<Long, Long> orderSessionsNumSum = orderStatDayRepository.getOrdersSessionsNumSum(orderIds, 7);
        Map<Long, Long> result = new HashMap<>();
        for (Long orderId : orderIds) {
            result.put(orderId, orderSessionsNumSum.getOrDefault(orderId, 0L));
        }
        return result;
    }

    /**
     * Возвращает суммарное количество сессий за последние семь дней для переданного OrderID
     *
     * @param orderId идентификатор заказа в БК
     * @return количество сессий за последние семь дней
     */
    public Long getOrderSessionsNumForWeek(Long orderId) {
        return getOrdersSessionsNumForWeek(Collections.singleton(orderId)).get(orderId);
    }

    /**
     * Возвращает суммарное количество кликов за сегодня для списка OrderID.
     * Если статистики за этот период нет, то возвращаются нули.
     *
     * @param orderIds список идентификаторов заказов в БК
     * @return словарь OrderID -> суммарное количество кликов за сегодня
     */
    public Map<Long, Long> getOrdersClicksToday(Collection<Long> orderIds) {
        // в песочнице отдавать настоящую статистику из продакшена мы не можем
        checkState(!environmentType.isSandbox(), NOT_FOR_SANDBOX);

        Map<Long, Long> orderClicksToday = orderStatDayRepository.getOrdersClicksToday(orderIds);
        Map<Long, Long> result = new HashMap<>();
        for (Long orderId : orderIds) {
            result.put(orderId, orderClicksToday.getOrDefault(orderId, 0L));
        }
        return result;
    }

    /**
     * Возвращает суммарное количество кликов за сегодня для переданного OrderID
     *
     * @param orderId идентификатор заказа в БК
     * @return количество кликов за сегодня
     */
    public Long getOrderClicksToday(Long orderId) {
        return getOrdersClicksToday(Collections.singleton(orderId)).get(orderId);
    }

    /**
     * Возвращает пару (сумма FraudClicks; сумма GiftClicks) за даты между {@param startDate} и {@param endDate}
     * включительно для переданного списка OrderID {@param orderIds}
     * Если статистики не было, то возвращаются нули
     */
    public FraudAndGiftClicks getFraudClicks(Collection<Long> orderIds, LocalDate startDate, LocalDate endDate) {
        // если метод нужен в песочнице - можно возвращать нули
        checkState(!environmentType.isSandbox(), "Ходим в прод yt, в песочнице эти данные не должны быть доступны");
        startDate = restrictToBsStatStartDate(startDate);
        return orderStatFraudRepository.getFraudClicks(orderIds, startDate, endDate);
    }

    /**
     * Ограничиваем до даты, с начала которой доступна статистика БК - начало текущего месяца минус
     * {@value BS_STAT_ALLOW_YEARS} лет (начало месяца приехало из DIRECT-51668)
     */
    public LocalDate restrictToBsStatStartDate(LocalDate date) {
        LocalDate statStartDate = timeProvider.now().toLocalDate().withDayOfMonth(1).minusYears(BS_STAT_ALLOW_YEARS);
        return date.isBefore(statStartDate) ? statStartDate : date;
    }

    /**
     * Возвращает для каждого переданного OrderID первую и последнюю дату, за которые есть статистика.
     *
     * @param orderIds список идентификаторов заказов в БК
     * @return словарь OrderID -> первая и последняя дата, за которые есть статистика,
     * если статистики нет, то возвращается null
     */
    public Map<Long, StatisticsDateRange> getOrdersFirstLastDay(Collection<Long> orderIds) {
        Map<Long, StatisticsDateRange> result = orderStatDayRepository.getOrdersFirstLastDay(orderIds);
        orderIds.forEach(o -> result.putIfAbsent(o, null));
        return result;
    }

    /**
     * Возвращает число - количество дней за которые есть статистика в указанном диапазоне дат
     * хотя бы у одного из переданных заказов
     *
     * @param orderIds  список идентификаторов заказов в БК
     * @param startDate дата начала периода, если указана дата ранее {@value BS_STAT_ALLOW_YEARS},
     *                  то используем {@value BS_STAT_ALLOW_YEARS}
     * @param endDate   дата окончания периода
     */
    public Long getOrdersDaysNum(Collection<Long> orderIds, LocalDate startDate, LocalDate endDate) {
        // в песочнице отдавать настоящую статистику из продакшена мы не можем
        checkState(!environmentType.isSandbox(), NOT_FOR_SANDBOX);

        return orderStatDayRepository.getOrdersDaysNum(orderIds, startDate, endDate);
    }

    /**
     * Возвращает словарь campaignId из {@param campaignIds} -> предсказание трат по кампании как максимум из трат
     * за последние семь дней
     *
     * @param currencyCode - валюта переданных кампаний (одна на всех)
     */
    public Map<Long, Money> getCampBsstatForecast(Collection<Long> campaignIds, CurrencyCode currencyCode) {
        Map<Long, List<Long>> subCampaignIds = new HashMap<>();
        Map<Long, Long> campaignIdToOrderId = new HashMap<>();

        shardHelper.groupByShard(campaignIds, ShardKey.CID).forEach((shard, shardCampaignIds) -> {
            Map<Long, List<Long>> shardSubCampaignIds =
                    campaignRepository.getSubCampaignIdsByMasterId(shard, shardCampaignIds);
            subCampaignIds.putAll(shardSubCampaignIds);
            Set<Long> shardAllCampaignIds = StreamEx.ofValues(shardSubCampaignIds)
                    .append(shardCampaignIds)
                    .toFlatCollection(Functions.identity(), HashSet::new);
            Map<Long, Long> shardCampaignIdToOrderId =
                    EntryStream.of(campaignRepository.getCampaignsSimple(shard, shardAllCampaignIds))
                            .values()
                            .peek(campaignSimple -> checkState(campaignSimple.getCurrency().equals(currencyCode),
                                    "wrong currency"))
                            .toMap(CampaignSimple::getId, CampaignSimple::getOrderId);
            campaignIdToOrderId.putAll(shardCampaignIdToOrderId);
        });

        Map<Long, Long> orderIdToMaxSum = orderStatDayRepository.getOrderIdToMaxSum(campaignIdToOrderId.values(),
                DAYS_FOR_FORECAST_COUNT);

        return StreamEx.of(campaignIds)
                .mapToEntry(campaignId -> StreamEx.of(campaignId)
                        .append(subCampaignIds.getOrDefault(campaignId, List.of()))
                        .map(campaignIdToOrderId::get)
                        .nonNull())
                .mapValues(orderIds -> orderIds
                        .map(orderIdToMaxSum::get)
                        .nonNull()
                        .map(sum -> applyRatio(sum, currencyCode))
                        .reduce(Money.valueOf(0, currencyCode), Money::add))
                .toMap((a, b) -> a.compareTo(b) > 0 ? a : b); // если в запросе пришло два одинаковых campaignId
    }

    /**
     * Для набора периодов {@param periods} и заказов {@param orderIds} возвращает сумму, потраченную заказами,
     * для каждого периода (словарь id периода -> сумма)
     */
    public Map<String, Money> getOrdersSumSpent(Collection<Long> orderIds, Collection<Period> periods,
                                                CurrencyCode currencyCode) {

        LocalDate startDate = periods.stream().map(Period::getStartDate).min(LocalDate::compareTo).orElseThrow();
        LocalDate endDate = periods.stream().map(Period::getEndDate).max(LocalDate::compareTo).orElseThrow();

        Map<LocalDate, BigDecimal> dateToSumSpent = orderStatDayRepository.getDateToSumSpent(orderIds, startDate,
                endDate);

        return StreamEx.of(periods)
                .mapToEntry(
                        Period::getKey,
                        period -> EntryStream.of(dateToSumSpent)
                                .filterKeys(period::isDateInPeriod)
                                .values()
                                .reduce(BigDecimal::add)
                                .orElse(BigDecimal.ZERO)
                )
                .mapValues(sum -> applyRatio(sum, currencyCode))
                .toMap();
    }

    /**
     * Для каждого заказа с указанным периодом находит потраченные за период деньги с ндс или без
     *
     * @param orderIds     список заказом
     * @param startDates   список начал переодов
     * @param finishDates  список концов периодов
     * @param currencyCode валюта
     * @param withNds      когда включен -- вычитаем ндс
     * @return возвращает мапу id заказа -> потраченная сумма
     */
    public Map<Long, Money> getSpentSumForOrderDuringPeriod(List<Long> orderIds, List<LocalDate> startDates,
                                                            List<LocalDate> finishDates,
                                                            CurrencyCode currencyCode,
                                                            boolean withNds) {

        Map<Long, List<SpentInfo>> ordersSpent = orderStatDayRepository.getOrdersSpent(orderIds, startDates,
                finishDates);

        Set<Long> taxIds = StreamEx.of(ordersSpent.values())
                .flatMap(Collection::stream)
                .map(SpentInfo::getTaxId)
                .toSet();

        Map<Long, List<TaxInfo>> taxInfosByTaxId = taxHistoryService.getTaxInfos(taxIds);
        return StreamEx.of(orderIds)
                .mapToEntry(orderId -> extractMoneyFromSpentInfo(orderId, ordersSpent,
                        taxInfosByTaxId, currencyCode, withNds))
                .toMap();
    }

    private static Money extractMoneyFromSpentInfo(Long orderId, Map<Long, List<SpentInfo>> ordersSpent,
                                                   Map<Long, List<TaxInfo>> taxInfosByTaxId, CurrencyCode currencyCode,
                                                   boolean withNds) {
        return withNds ?
                extractMoneyFromSpentInfosWithNds(ordersSpent.get(orderId), currencyCode,
                        taxInfosByTaxId) :
                extractMoneyFromSpentInfoWithoutNds(ordersSpent.get(orderId), currencyCode);
    }

    private static Money extractMoneyFromSpentInfoWithoutNds(@Nullable List<SpentInfo> spentInfos,
                                                             CurrencyCode currencyCode) {
        if (spentInfos == null || spentInfos.isEmpty()) {
            return Money.valueOf(0, currencyCode);
        }
        return StreamEx.of(spentInfos)
                .map(SpentInfo::getCost)
                .map(BigDecimal::valueOf)
                .foldLeft(BigDecimal::add)
                .map(cost -> applyRatio(cost, currencyCode))
                .orElseThrow();

    }

    private static Money extractMoneyFromSpentInfosWithNds(@Nullable List<SpentInfo> spentInfos,
                                                           CurrencyCode currencyCode,
                                                           Map<Long, List<TaxInfo>> taxInfosByTaxId) {
        if (spentInfos == null) {
            return Money.valueOf(0, currencyCode);
        }

        return StreamEx.of(spentInfos)
                .map(spentInfo -> subtractNds(spentInfo, currencyCode, taxInfosByTaxId.get(spentInfo.getTaxId())))
                .foldLeft(Money::add)
                .orElse(Money.valueOf(0, currencyCode));

    }

    private static Money subtractNds(@Nullable SpentInfo spentInfo, CurrencyCode currencyCode,
                                     List<TaxInfo> taxInfos) {
        if (spentInfo == null) {
            return Money.valueOf(0, currencyCode);
        }

        LocalDate dayOfSpending = DateTimeUtils.instantToMoscowDate(spentInfo.getStatDate());
        TaxInfo taxInfoForDay = getTaxInfoForDay(dayOfSpending, taxInfos);

        Money cost = applyRatio(spentInfo.getCost(), currencyCode);

        return cost.subtractNds(taxInfoForDay.getPercent());
    }

    /**
     * Возвращает количество потраченных за сегодня денег с НДС или без в зависимости от флага {@param withNds}
     * Если по кампании нет статистики, то результат по ней не возвращается
     */
    public Map<Long, Money> getOrdersSpentToday(Collection<Long> orderIds, boolean withNds) {
        LocalDate today = timeProvider.now().toLocalDate();
        Map<Long, SpentInfo> ordersToSpent = orderStatDayRepository.getOrdersSpent(orderIds, today);
        return EntryStream.of(ordersToSpent)
                .mapValues(spentInfo -> orderSpentTodayInternal(spentInfo, withNds, today))
                .toMap();
    }

    private Money orderSpentTodayInternal(SpentInfo spentInfo, boolean withNds, LocalDate day) {
        checkState(spentInfo.getCurrencyId() != null, "CurrencyID must not be null");
        checkState(spentInfo.getTaxId() != null, "TaxID must not be null");

        CurrencyCode currencyCode = Currencies.getCurrency(spentInfo.getCurrencyId().intValue()).getCode();

        Money res = applyRatio(spentInfo.getCost(), currencyCode);

        if (!withNds) {
            Percent percent = taxHistoryService.getTaxInfo(spentInfo.getTaxId(), day).getPercent();
            res = res.subtractNds(percent);
        } else {
            res = res.roundToCentDown();
        }

        return res;
    }

    private static Money applyRatio(Long value, CurrencyCode currencyCode) {
        return applyRatio(BigDecimal.valueOf(value), currencyCode);
    }

    private static Money applyRatio(BigDecimal value, CurrencyCode currencyCode) {
        return Money.valueOf(value, currencyCode)
                .multiply(currencyCode.getCurrency().getYabsRatio())
                .divide(MILLION);
    }
}
