package ru.yandex.direct.core.entity.statisticsgathering.repository;

import java.time.LocalDateTime;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.monitor.tables.MonitorValuesSec.MONITOR_VALUES_SEC;

/**
 * Работа с таргетами мониторинга
 */
@ParametersAreNonnullByDefault
@Repository
public class MonitorTargetsRepository {
    private static final int DELETE_CHUNK_SIZE = 10_000;
    private static final double DELETE_SLEEP_COEF = 1.0;
    private static final Logger logger = LoggerFactory.getLogger(MonitorTargetsRepository.class);

    private final DslContextProvider dslContextProvider;

    @Autowired
    public MonitorTargetsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Удаляет значения старше указанного времени из таблицы monitor_values_sec
     *
     * @param borderDateTime {@link LocalDateTime} граничное время
     * @return число удаленных записей
     */

    public int deleteSecValuesOlderThan(LocalDateTime borderDateTime) {
        int totalDeleted = 0;
        var relaxed = new RelaxedWorker(DELETE_SLEEP_COEF);
        for (int iter = 1; ; iter++) {
            int deleted = relaxed.callAndRelax(() ->
                    dslContextProvider.monitor()
                            .deleteFrom(MONITOR_VALUES_SEC)
                            .where(MONITOR_VALUES_SEC.MEASURE_TIME.lessThan(borderDateTime))
                            .limit(DELETE_CHUNK_SIZE)
                            .execute()
            );

            totalDeleted += deleted;
            logger.info("Iteration {}, deleted {} rows", iter, deleted);
            if (deleted < DELETE_CHUNK_SIZE) {
                break;
            }
        }
        return totalDeleted;
    }

}
