package ru.yandex.direct.core.entity.stopword.repository;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.type.TypeReference;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.dbschema.ppcdict.tables.StopWords.STOP_WORDS;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Repository(StopWordRepository.BEAN_NAME)
public class StopWordRepository {

    public static final String BEAN_NAME = "stopWordRepository";

    private static final long STOP_WORDS_ID = 1L;
    private static final TypeReference<List<String>> STOP_WORDS_TYPE = new TypeReference<List<String>>() {
    };

    private final DslContextProvider dslContextProvider;

    @Autowired
    public StopWordRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Заменить стоп-слова в таблице.
     * Обновляется запись со списком стоп-слов в json представлении.
     *
     * @param newStopWords полный список стоп-слов для обновления
     * @return true если список стоп-слов был изменён
     */
    public boolean replaceStopWords(@Nullable Set<String> newStopWords) {
        return dslContextProvider.ppcdict().transactionResult(cfg -> {
            DSLContext txContext = DSL.using(cfg);

            String savedStopWordsJson = txContext.select(STOP_WORDS.TEXT)
                    .from(STOP_WORDS)
                    .where(STOP_WORDS.ID.eq(STOP_WORDS_ID))
                    .forUpdate()
                    .fetchOne(STOP_WORDS.TEXT);

            Set<String> savedStopWords = extractStopWordsSet(savedStopWordsJson);
            if (!savedStopWords.equals(newStopWords)) {
                String jsonKeywords = JsonUtils.toJson(nvl(newStopWords, emptySet()));
                txContext.insertInto(STOP_WORDS, STOP_WORDS.ID, STOP_WORDS.TEXT)
                        .values(STOP_WORDS_ID, jsonKeywords)
                        .onDuplicateKeyUpdate()
                        .set(STOP_WORDS.TEXT, jsonKeywords)
                        .execute();

                return true;
            }

            return false;
        });
    }

    /**
     * Получить список стоп-слов из таблицы или пустую коллекцию, если записи нет
     *
     * @return полный список стоп-слов
     */
    public Set<String> getStopWords() {
        String stopWordsJson = dslContextProvider.ppcdict()
                .select(STOP_WORDS.TEXT)
                .from(STOP_WORDS)
                .where(STOP_WORDS.ID.eq(STOP_WORDS_ID))
                .fetchOne(STOP_WORDS.TEXT);

        return extractStopWordsSet(stopWordsJson);
    }

    private Set<String> extractStopWordsSet(String stopWordsJson) {
        return stopWordsJson == null
                ? emptySet()
                : new HashSet<>(JsonUtils.fromJson(stopWordsJson, STOP_WORDS_TYPE));
    }

}
