package ru.yandex.direct.core.entity.stopword.service;

import java.time.Duration;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableSet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.advq.stringprep.StringPrep;
import ru.yandex.direct.core.entity.stopword.repository.StopWordRepository;

import static org.apache.commons.lang3.StringUtils.isEmpty;

/**
 * Кеширующий сервис проверки стоп-слов
 */
@Service
@ParametersAreNonnullByDefault
public class StopWordService {
    private static final Duration RELOAD_INTERVAL = Duration.ofMinutes(60);

    private final StopWordRepository stopWordRepository;
    private Supplier<Set<String>> dataSupplier;

    @Autowired
    public StopWordService(StopWordRepository stopWordRepository) {
        this.stopWordRepository = stopWordRepository;

        dataSupplier = Suppliers.memoizeWithExpiration(
                this::computeData, RELOAD_INTERVAL.getSeconds(), TimeUnit.SECONDS);
    }

    private Set<String> computeData() {
        return ImmutableSet.copyOf(stopWordRepository.getStopWords());
    }

    /**
     * Проверка является ли строка стоп-словом.
     * Проверка работает с учетом обработки строки методом {@code StringPrep.toPreparedString()},
     * а так же с учетом того, что слово может оканчиваться на точку.
     *
     * @param text строка
     * @return входит в список стоп-слов или нет
     */
    public boolean isStopWord(String text) {
        return isStopWord(text, true);
    }

    /**
     * Проверка является ли строка стоп-словом.
     * Проверка работает с учетом обработки строки методом {@code StringPrep.toPreparedString()}
     *
     * @param text         строка
     * @param removePeriod удалять ли точку в конце слова
     * @return входит в список стоп-слов или нет
     */
    public boolean isStopWord(String text, boolean removePeriod) {
        if (isEmpty(text)) {
            return false;
        }
        if (removePeriod && (text.charAt(text.length() - 1) == '.')) {
            text = text.substring(0, text.length() - 1);
        }
        return dataSupplier.get().contains(StringPrep.toPreparedString(text));
    }

    public Set<String> getStopWords() {
        return dataSupplier.get();
    }
}
