package ru.yandex.direct.core.entity.strategy.container

import org.jooq.DSLContext
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientAdapter
import ru.yandex.direct.core.entity.campaign.service.WalletService
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.metrika.service.strategygoals.StrategyGoalsService
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants.DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY
import ru.yandex.direct.core.entity.strategy.utils.StrategyModelUtils.campaignIds
import ru.yandex.direct.core.entity.strategy.utils.StrategyModelUtils.metrikaCounterIds
import ru.yandex.direct.dbschema.ppc.tables.Strategies
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.metrika.client.MetrikaClient
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.whereEqFilter
import ru.yandex.direct.rbac.RbacService
import java.util.IdentityHashMap

@Service
class StrategyAddOperationContainerService(
    private val clientService: ClientService,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val strategyGoalsService: StrategyGoalsService,
    private val featureService: FeatureService,
    private val rbacService: RbacService,
    private val walletService: WalletService,
    private val campaignRepository: CampaignRepository,
    private val metrikaClient: MetrikaClient,
    private val strategyTypedRepository: StrategyTypedRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) {

    fun fillContainers(
        container: StrategyAddOperationContainer,
        strategies: List<BaseStrategy>,
        dslContext: DSLContext? = null
    ) {
        val clientRepresentativesUids: List<Long> =
            rbacService.getClientRepresentativesUidsForGetMetrikaCounters(container.clientId)

        val shouldFetchUnavailableGoals = StrategyUpdateOperationUtils.shouldFetchUnavailableGoals(container)

        container.metrikaClientAdapter =
            RequestBasedMetrikaClientAdapter.getRequestBasedMetrikaClientAdapterForStrategies(
                metrikaClient,
                clientRepresentativesUids,
                container.availableFeatures,
                strategies,
                shouldFetchUnavailableGoals
            )
        container.strategyToIndexMap = strategies.withIndex()
            .associate { it.value to it.index }
            .toMap(IdentityHashMap())
        container.currency = clientService.getWorkCurrency(container.clientId)

        container.typedCampaignsMap = getCampaignsWithType(strategies, container.shard, dslContext)

        container.strategyGoalsMap = getAvailableGoals(container, strategies)
        container.availableFeatures = getAvailableFeatures(container.clientId)
        container.operatorRole = rbacService.getUidRole(container.operator)
        container.wallet = getCampaignWallet(container.shard, container.clientId)
        container.options.maxNumberOfCids =
            ppcPropertiesSupport.get(PpcPropertyNames.MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY)
                .getOrDefault(DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY)

        val existingStrategies = getAllClientStrategies(container.shard, container.clientId)

        container.clientStrategiesNumber = existingStrategies.filter { it.isPublic }.size

        container.clientUnarchivedStrategiesNumber =
            existingStrategies.filter { it.isPublic && !it.statusArchived }.size
    }

    private fun getAvailableGoals(container: StrategyAddOperationContainer, strategies: List<BaseStrategy>) =
        if (container.options.isCampaignToPackageStrategyOneshot) {
            IdentityHashMap()
        } else {
            val strategiesWithCampaignTypeMap = strategies
                .groupBy {
                    StrategyGoalsService.Companion.StrategyAvailableGoalsRequest(
                        it.campaignIds().toSet(),
                        it.metrikaCounterIds().toSet(),
                        container.campaignTypesMap[it]
                    )
                }
            strategyGoalsService
                .getAvailableForStrategies(container.clientId, container.operator, strategiesWithCampaignTypeMap.keys)
                .mapNotNull {
                    strategiesWithCampaignTypeMap[it.key]?.let { strategies -> strategies.map { strategy -> strategy to it.value } }
                }.flatten().toMap(IdentityHashMap())
        }

    private fun getCampaignsWithType(strategies: List<BaseStrategy>, shard: Int, dslContext: DSLContext?):
        IdentityHashMap<BaseStrategy, List<CampaignWithPackageStrategy>> {
        val campaignIds = strategies.flatMap { it.campaignIds() }
        val campaignsWithType = getCampaigns(campaignIds.toSet(), shard, dslContext)
        return strategies.map {
            val campaignWithTypeList = it.campaignIds().mapNotNull(campaignsWithType::get)
            it to campaignWithTypeList
        }.toMap(IdentityHashMap())
    }

    private fun getCampaigns(
        campaignIds: Set<Long>,
        shard: Int,
        dslContext: DSLContext?
    ): Map<Long, CampaignWithPackageStrategy> {
        if (campaignIds.isEmpty()) {
            return emptyMap()
        }
        val campaigns = if (dslContext == null) {
            campaignTypedRepository.getTyped(shard, campaignIds)
        } else {
            campaignTypedRepository.getTyped(dslContext, campaignIds)
        }
        return campaigns
            .mapNotNull { it as? CampaignWithPackageStrategy }
            .associateBy(CampaignWithPackageStrategy::getId)
    }

    private fun getAvailableFeatures(clientId: ClientId): Set<FeatureName> =
        featureService.getEnabledForClientId(clientId)
            .mapNotNull(FeatureName::fromString)
            .toSet()

    private fun getCampaignWallet(shard: Int, clientId: ClientId): Campaign? {
        val wallet = walletService.massGetWallets(listOf(clientId)).firstOrNull()
        return wallet?.let {
            campaignRepository.getCampaigns(shard, listOf(it.walletCampaignId)).firstOrNull()
        }
    }

    private fun getAllClientStrategies(shard: Int, clientId: ClientId): List<CommonStrategy> {
        return strategyTypedRepository.getSafely(
            shard,
            whereEqFilter(Strategies.STRATEGIES.CLIENT_ID, clientId.asLong()),
            CommonStrategy::class.java
        )
    }
}
