package ru.yandex.direct.core.entity.strategy.container

import one.util.streamex.StreamEx
import org.jooq.DSLContext
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.campaign.model.Campaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientAdapter
import ru.yandex.direct.core.entity.campaign.service.WalletService
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.metrika.service.commongoals.CommonCountersService
import ru.yandex.direct.core.entity.metrika.service.strategygoals.StrategyGoalsService
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.model.StrategyWithCampaignIds
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMetrikaCounters
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants.DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY
import ru.yandex.direct.core.entity.strategy.service.StrategyConstants.STRATEGY_TYPE_TO_CLASS
import ru.yandex.direct.core.entity.strategy.utils.StrategyModelUtils.campaignIds
import ru.yandex.direct.core.entity.strategy.utils.StrategyModelUtils.metrikaCounterIds
import ru.yandex.direct.dbschema.ppc.tables.Strategies
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.metrika.client.MetrikaClient
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory
import ru.yandex.direct.rbac.RbacService
import java.util.function.Function

@Service
class StrategyUpdateOperationContainerService(
    private val clientService: ClientService,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val strategyGoalsService: StrategyGoalsService,
    private val featureService: FeatureService,
    private val rbacService: RbacService,
    private val walletService: WalletService,
    private val campaignRepository: CampaignRepository,
    private val metrikaClient: MetrikaClient,
    private val commonCountersService: CommonCountersService,
    private val strategyTypedRepository: StrategyTypedRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) {

    fun fillContainers(
        container: StrategyUpdateOperationContainer,
        strategies: Collection<BaseStrategy>,
        modelChanges: List<ModelChanges<out BaseStrategy>>,
        dslContext: DSLContext? = null
    ) {
        container.strategyIdToClass = strategies.associate { it.id to STRATEGY_TYPE_TO_CLASS[it.type] }
        container.typedCampaignsMap = getCampaignsWithType(strategies, modelChanges, container.shard, dslContext)
        container.strategyGoalsMap = getGoalsMap(strategies, modelChanges, container)
        container.availableFeatures = getAvailableFeatures(container.clientId)
        container.currency = clientService.getWorkCurrency(container.clientId)
        container.operatorRole = rbacService.getUidRole(container.operator)
        container.wallet = getCampaignWallet(container.shard, container.clientId)
        container.options.maxNumberOfCids =
            ppcPropertiesSupport.get(PpcPropertyNames.MAX_NUMBER_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY)
                .getOrDefault(DEFAULT_MAX_NUMBERS_OF_CIDS_ABLE_TO_LINK_TO_PACKAGE_STRATEGY)

        val clientRepresentativesUids: List<Long> =
            rbacService.getClientRepresentativesUidsForGetMetrikaCounters(container.clientId)

        val shouldFetchUnavailableGoals = StrategyUpdateOperationUtils.shouldFetchUnavailableGoals(container)

        container.metrikaClientAdapter =
            RequestBasedMetrikaClientAdapter.getRequestBasedMetrikaClientAdapterForStrategies(
                metrikaClient,
                clientRepresentativesUids,
                container.availableFeatures,
                strategies,
                shouldFetchUnavailableGoals
            )

        val existingStrategies = getAllClientStrategies(container.shard, container.clientId)

        container.clientStrategiesNumber = existingStrategies.filter { it.isPublic }.size

        container.clientUnarchivedStrategiesNumber =
            existingStrategies.filter { it.isPublic && !it.statusArchived }.size
    }

    fun <T : BaseStrategy> createRepositoryContainer(
        container: StrategyUpdateOperationContainer,
        applicableAppliedChanges: List<AppliedChanges<T>>?,
        isCampaignToStrategyMigrationOneshot: Boolean
    ): StrategyRepositoryContainer {
        val counterIds: List<Long> = StreamEx.of(applicableAppliedChanges)
            .map { it.model }
            .select(StrategyWithMetrikaCounters::class.java)
            .flatCollection { it.metrikaCounters }
            .distinct()
            .toList()
        val availableCountersFromMetrika: List<UserCountersExtended> =
            commonCountersService.getAvailableCountersFromMetrika(container.clientId, counterIds)
        val counterInfoById = if (!isCampaignToStrategyMigrationOneshot) {
            StreamEx.of(availableCountersFromMetrika)
                .flatCollection { obj -> obj.counters }
                .mapToEntry({ obj -> obj.id }, Function.identity())
                .distinctKeys()
                .mapKeys { obj -> obj.toLong() }
                .toMap()
        } else {
            emptyMap()
        }
        return StrategyRepositoryContainer(
            container.shard, container.clientId, counterInfoById,
            container.options.isStrategyIdsAlreadyFilled
        )
    }

    private fun getCampaignsWithType(
        strategies: Collection<BaseStrategy>,
        modelChanges: List<ModelChanges<out BaseStrategy>>,
        shard: Int,
        dslContext: DSLContext?
    ): Map<Long, List<CampaignWithPackageStrategy>> {
        val campaignIdsBySid = getCampaignIdsBySid(strategies, modelChanges)

        val cids = campaignIdsBySid.values.flatten()
        val campaignsWithType = getCampaigns(cids.toSet(), shard, dslContext)

        return campaignIdsBySid.mapValues { it.value.mapNotNull(campaignsWithType::get) }
    }

    private fun getCampaigns(
        campaignIds: Set<Long>,
        shard: Int,
        dslContext: DSLContext?
    ): Map<Long, CampaignWithPackageStrategy> {
        if (campaignIds.isEmpty()) {
            return emptyMap()
        }
        val campaigns = if (dslContext == null) {
            campaignTypedRepository.getTyped(shard, campaignIds)
        } else {
            campaignTypedRepository.getTyped(dslContext, campaignIds)
        }
        return campaigns
            .mapNotNull { it as? CampaignWithPackageStrategy }
            .associateBy(CampaignWithPackageStrategy::getId)
    }

    private fun getGoalsMap(
        strategies: Collection<BaseStrategy>,
        modelChanges: List<ModelChanges<out BaseStrategy>>,
        container: StrategyUpdateOperationContainer
    ): Map<Long, Set<Goal>> {
        val campaignIdsBySid = getCampaignIdsBySid(strategies, modelChanges)

        val strategiesWithCampaignTypeMap = strategies
            .associateBy { it.id }
            .mapValues {
                StrategyGoalsService.Companion.StrategyAvailableGoalsRequest(
                    campaignIdsBySid[it.key]!!.toSet(),
                    it.value.metrikaCounterIds().toSet(),
                    container.campaignTypesMap[it.key]
                )
            }

        return strategyGoalsService.getAvailableForStrategyIds(
            container.clientId,
            container.operator,
            strategiesWithCampaignTypeMap
        )
    }

    private fun getCampaignIdsBySid(
        strategies: Collection<BaseStrategy>,
        modelChanges: List<ModelChanges<out BaseStrategy>>,
    ): Map<Long, List<Long>> {
        val modelChangesBySid =
            modelChanges.associateBy { it.id }.mapValues { it.value as? ModelChanges<StrategyWithCampaignIds> }

        return strategies.associate {
            val updatedStrategyCids =
                modelChangesBySid[it.id]?.getPropIfChanged(StrategyWithCampaignIds.CIDS) ?: emptyList()
            it.id to updatedStrategyCids.plus(it.campaignIds()).distinct()
        }
    }

    private fun getAvailableFeatures(clientId: ClientId): Set<FeatureName> =
        featureService.getEnabledForClientId(clientId)
            .mapNotNull(FeatureName::fromString)
            .toSet()

    private fun getCampaignWallet(shard: Int, clientId: ClientId): Campaign? {
        val wallet = walletService.massGetWallets(listOf(clientId)).firstOrNull()
        return wallet?.let {
            campaignRepository.getCampaigns(shard, listOf(it.walletCampaignId)).firstOrNull()
        }
    }

    private fun getAllClientStrategies(shard: Int, clientId: ClientId): List<CommonStrategy> {
        return strategyTypedRepository.getSafely(
            shard,
            ConditionFilterFactory.whereEqFilter(Strategies.STRATEGIES.CLIENT_ID, clientId.asLong()),
            CommonStrategy::class.java
        )
    }
}
