package ru.yandex.direct.core.entity.strategy.repository;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.TableField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyAttributionModel;
import ru.yandex.direct.core.entity.strategy.model.StrategyName;
import ru.yandex.direct.dbschema.ppc.enums.StrategiesArchived;
import ru.yandex.direct.dbschema.ppc.enums.StrategiesIsPublic;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.jsonwrite.JsonWriterBuilders;

import static ru.yandex.direct.common.util.RepositoryUtils.booleanToYesNoWithDefault;
import static ru.yandex.direct.dbschema.ppc.Tables.STRATEGIES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Component
@ParametersAreNonnullByDefault
public class CommonStrategyRepositoryTypeSupport extends BaseStrategyDataFieldRepositoryTypeSupport<CommonStrategy> {

    @Autowired
    protected CommonStrategyRepositoryTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<CommonStrategy> getTypeClass() {
        return CommonStrategy.class;
    }

    @Override
    public JooqMapper<CommonStrategy> createMapper() {
        return baseBuilder()
                .writeField(STRATEGIES.STRATEGY_DATA, JsonWriterBuilders.fromConvertableEnumProperty(
                        CommonStrategy.TYPE,
                        strategyName -> StrategyName.toSource(strategyName).getLiteral(),
                        "name"))
                .map(convertibleProperty(CommonStrategy.TYPE, STRATEGIES.TYPE,
                        StrategyName::fromSource,
                        StrategyName::toSource))
                .map(property(CommonStrategy.CLIENT_ID, STRATEGIES.CLIENT_ID))
                .map(property(CommonStrategy.NAME, STRATEGIES.NAME))
                .map(property(CommonStrategy.WALLET_ID, STRATEGIES.WALLET_CID))
                .map(convertibleProperty(
                        CommonStrategy.STATUS_ARCHIVED,
                        STRATEGIES.ARCHIVED,
                        RepositoryUtils::booleanFromYesNo,
                        value -> booleanToYesNoWithDefault(value, StrategiesArchived.class, StrategiesArchived.No)
                ))
                .map(property(CommonStrategy.LAST_CHANGE, STRATEGIES.LAST_CHANGE))
                .map(convertibleProperty(
                        CommonStrategy.ATTRIBUTION_MODEL,
                        STRATEGIES.ATTRIBUTION_MODEL,
                        StrategyAttributionModel::fromSource,
                        StrategyAttributionModel::toSource
                ))
                .map(convertibleProperty(
                        CommonStrategy.IS_PUBLIC,
                        STRATEGIES.IS_PUBLIC,
                        RepositoryUtils::booleanFromYesNo,
                        value -> booleanToYesNoWithDefault(
                                value,
                                StrategiesIsPublic.class,
                                StrategiesIsPublic.No
                        )
                ))
                .build();
    }

    @Override
    public Set<TableField<?, Long>> getPrimaryKeys() {
        return Set.of(STRATEGIES.STRATEGY_ID);
    }
}
