package ru.yandex.direct.core.entity.strategy.repository;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Table;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.repository.TypeModifyRepository;
import ru.yandex.direct.multitype.repository.filter.Filter;

import static ru.yandex.direct.core.entity.strategy.repository.StrategyFilterFactory.strategyIdFilter;
import static ru.yandex.direct.dbschema.ppc.tables.Strategies.STRATEGIES;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Repository
@ParametersAreNonnullByDefault
public class StrategyModifyRepository extends TypeModifyRepository<BaseStrategy, BaseStrategy,
        StrategyRepositoryContainer, StrategyRepositoryContainer> {
    private final DslContextProvider ppcDslContextProvider;

    private final ShardHelper shardHelper;
    private final ShardSupport shardSupport;

    @Autowired
    public StrategyModifyRepository(DslContextProvider ppcDslContextProvider,
                                    ShardHelper shardHelper,
                                    StrategyRepositoryTypeSupportFacade typeSupportFacade,
                                    StrategyTypedRepository typedRepository,
                                    ShardSupport shardSupport) {
        super(ppcDslContextProvider, typeSupportFacade, typedRepository);
        this.shardHelper = shardHelper;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.shardSupport = shardSupport;
    }

    @Override
    protected void generateIds(StrategyRepositoryContainer addContainer, List<? extends BaseStrategy> models) {
        Iterator<Long> strategyIds = shardHelper.generateStrategyIds(addContainer.getClientId().asLong(),
                models.size()).iterator();
        models.forEach(strategy -> strategy.setId(strategyIds.next()));
    }

    public List<Long> add(DSLContext context,
                          StrategyRepositoryContainer addContainer,
                          List<? extends BaseStrategy> models) {
        if (!addContainer.isStrategyIdsAlreadyFilled()) {
            generateIds(addContainer, models);
        } else {
            var strategyIdToClientId = listToMap(models, BaseStrategy::getId, strategy -> addContainer.getClientId());
            shardSupport.saveValues(ShardKey.STRATEGY_ID, ShardKey.CLIENT_ID, strategyIdToClientId);
        }
        return addInternal(context, addContainer, models);
    }

    @Override
    protected Table<?> getLockTable() {
        return STRATEGIES;
    }

    @Override
    protected Filter getIdFilter(Collection<Long> modelIds) {
        return strategyIdFilter(modelIds);
    }

    /**
     * Обновление только таблиц strategies.
     * Для обновления кампаний со всеми связанными сущностями использовать {@link this#updateTable}
     */
    public void updateStrategiesTable(int shard,
                                     Collection<? extends AppliedChanges<? extends BaseStrategy>> appliedChanges) {
        updateTable(ppcDslContextProvider.ppc(shard), appliedChanges);
    }
}
