package ru.yandex.direct.core.entity.strategy.repository;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer;
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy;
import ru.yandex.direct.core.entity.strategy.model.DefaultManualStrategy;
import ru.yandex.direct.core.entity.strategy.model.StrategyName;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithDayBudget;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.StrategiesType;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.repository.RepositoryTypeSupportFacade;
import ru.yandex.direct.multitype.typesupport.TypeSupport;
import ru.yandex.direct.multitype.typesupport.TypeSupportUtils;

import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.INTERFACES_BY_CLASS;
import static ru.yandex.direct.core.entity.strategy.service.StrategyConstants.TYPE_TO_STRATEGY_CLASS_SUPPLIER;
import static ru.yandex.direct.dbschema.ppc.Tables.STRATEGIES;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class StrategyRepositoryTypeSupportFacade extends RepositoryTypeSupportFacade<BaseStrategy, StrategyName,
        StrategyRepositoryContainer, StrategyRepositoryContainer> {

    Map<? extends Class<? extends BaseStrategy>, ? extends JooqMapper<? extends BaseStrategy>> mapperByClass;
    @SuppressWarnings("rawtypes")
    Map<? extends Class<? extends BaseStrategy>, Set<Table>> tablesByClass;
    Map<? extends Class<? extends BaseStrategy>, Set<TableField<?, Long>>> primaryKeysByClass;

    @Autowired
    public StrategyRepositoryTypeSupportFacade(
            List<? extends StrategyRepositoryTypeSupport<? extends BaseStrategy>> repositoryTypeSupports,
            Collection<Field<?>> additionalReadFields,
            Set<Class<? extends BaseStrategy>> whiteListSupportTypeClass) {
        super(repositoryTypeSupports, TYPE_TO_STRATEGY_CLASS_SUPPLIER, additionalReadFields, whiteListSupportTypeClass);

        mapperByClass = StreamEx.ofKeys(INTERFACES_BY_CLASS)
                .mapToEntry(clazz -> getMapperForModel(clazz, repositoryTypeSupports))
                .toMap();

        tablesByClass = getTablesByClass(repositoryTypeSupports);

        primaryKeysByClass = getPrimaryKeysByClass(repositoryTypeSupports);
    }

    private <T extends BaseStrategy> JooqMapper<T> getMapperForModel(
            Class<T> clazz,
            List<? extends StrategyRepositoryTypeSupport<? extends BaseStrategy>> supports) {
        List<? extends StrategyRepositoryTypeSupport<? super T>> supportsForModel =
                TypeSupportUtils.getSupportsByClassWithExactTypes(supports, clazz);
        List<? extends JooqMapper<? super T>> jooqMappers = StreamEx.of(supportsForModel)
                .map(StrategyRepositoryTypeSupport::createMapper)
                .nonNull()
                .toList();

        return JooqMapperBuilder.builder(jooqMappers).build();
    }

    private Map<? extends Class<? extends BaseStrategy>, Set<TableField<?, Long>>> getPrimaryKeysByClass(
            List<? extends StrategyRepositoryTypeSupport<? extends BaseStrategy>> repositoryTypeSupports) {
        var primaryKeysByInterfaces = StreamEx.of(repositoryTypeSupports)
                .mapToEntry(TypeSupport::getTypeClass, x -> x)
                .flatMapValues(x -> x.getPrimaryKeys().stream())
                .grouping();

        //noinspection SuspiciousMethodCalls
        return EntryStream.of(INTERFACES_BY_CLASS)
                .flatMapValues(Collection::stream)
                .mapValues(primaryKeysByInterfaces::get)
                .nonNullValues()
                .flatMapValues(Collection::stream)
                .groupingTo(HashSet::new);
    }

    @SuppressWarnings("rawtypes")
    private Map<? extends Class<? extends BaseStrategy>, Set<Table>> getTablesByClass(
            List<? extends StrategyRepositoryTypeSupport<? extends BaseStrategy>> repositoryTypeSupports) {
        var tablesByInterfaces =
                StreamEx.of(repositoryTypeSupports)
                        .mapToEntry(TypeSupport::getTypeClass, x -> x)
                        .flatMapValues(x -> x.getPrimaryKeys().stream())
                        .mapValues(TableField::getTable)
                        .grouping();

        //noinspection SuspiciousMethodCalls
        return EntryStream.of(INTERFACES_BY_CLASS)
                .flatMapValues(Collection::stream)
                .mapValues(tablesByInterfaces::get)
                .nonNullValues()
                .flatMapValues(Collection::stream)
                .selectValues(Table.class)
                .groupingTo(HashSet::new);
    }

    @Override
    protected StrategyName getModelType(Record record) {
        var strategyDbName = record.get(STRATEGIES.TYPE, String.class);
        var strategyName = Arrays.stream(StrategiesType.values())
                .filter(type -> type.getLiteral().equals(strategyDbName))
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("Unable to parse strategy_name"));

        return StrategyName.fromSource(strategyName);
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateBuilderAggregator,
                              Collection<? extends AppliedChanges<? extends BaseStrategy>> appliedChanges) {
        var appliedChangesByClass = StreamEx.of(appliedChanges)
                .mapToEntry(changes -> changes.getModel().getClass(), Function.identity())
                .grouping();

        appliedChangesByClass.forEach((clazz, models) -> processUpdate(clazz, updateBuilderAggregator, models));
    }


    public <M extends BaseStrategy> void processUpdate(
            Class<M> clazz,
            UpdateHelperAggregator updateBuilderAggregator,
            List<? extends AppliedChanges<? extends BaseStrategy>> appliedChanges) {
        List<AppliedChanges<M>> appliedChangesOfCertainModelType = mapList(appliedChanges,
                changes -> changes.castModelUp(clazz));

        //noinspection unchecked
        JooqMapper<M> jooqMapper = (JooqMapper<M>) mapperByClass.get(clazz);
        primaryKeysByClass.get(clazz).forEach(primaryKey ->
                updateBuilderAggregator.getOrCreate(primaryKey).processUpdateAll(jooqMapper,
                        appliedChangesOfCertainModelType));
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, List<? extends BaseStrategy> modelList) {
        modelList.forEach(model -> pushToInsert(insertHelperAggregator, model));
    }

    private <M extends BaseStrategy> void pushToInsert(InsertHelperAggregator insertHelperAggregator,
                                                       M model) {

        JooqMapper<M> jooqMapper = getJooqMapper(model);
        pushToInsert(insertHelperAggregator, model, jooqMapper);
        insertHelperAggregator.newRecord();
    }

    private <M extends BaseStrategy> JooqMapper<M> getJooqMapper(M model) {
        //noinspection unchecked
        return (JooqMapper<M>) mapperByClass.get(model.getClass());
    }

    private <M extends BaseStrategy> void pushToInsert(
            InsertHelperAggregator insertHelperAggregator,
            M model,
            JooqMapper<M> jooqMapper) {
        Set<Table> tables = tablesByClass.get(model.getClass());
        //noinspection unchecked
        tables.forEach(table -> insertHelperAggregator.getOrCreate(table).add(jooqMapper, model));
        if (tables.contains(Tables.STRATEGIES)) {
            insertStrategiesTableDefaultValues(insertHelperAggregator, model);
        }
    }

    private <M extends BaseStrategy> void insertStrategiesTableDefaultValues(
            InsertHelperAggregator insertHelperAggregator,
            M model) {
        var insertHelper = insertHelperAggregator.getOrCreate(Tables.STRATEGIES);
        insertHelper.set(Tables.STRATEGIES.STRATEGY_ID, model.getId());
        if (!(model instanceof DefaultManualStrategy)) {
            insertHelper.set(STRATEGIES.ENABLE_CPC_HOLD, STRATEGIES.ENABLE_CPC_HOLD.getDataType().defaultValue());
        }
        if (!(model instanceof StrategyWithDayBudget)) {
            insertHelper.set(STRATEGIES.DAY_BUDGET, STRATEGIES.DAY_BUDGET.getDataType().defaultValue());
            insertHelper.set(STRATEGIES.DAY_BUDGET_SHOW_MODE,
                    STRATEGIES.DAY_BUDGET_SHOW_MODE.getDataType().defaultValue());
            insertHelper.set(STRATEGIES.DAY_BUDGET_DAILY_CHANGE_COUNT,
                    STRATEGIES.DAY_BUDGET_DAILY_CHANGE_COUNT.getDataType().defaultValue());
        }
    }

}
