package ru.yandex.direct.core.entity.strategy.service

import org.jooq.DSLContext
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.container.CampaignAdditionalActionsContainer
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusBsSynced
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignAdditionalActionsService
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientFactory
import ru.yandex.direct.core.entity.campaign.service.type.update.BaseCampaignUpdateOperationSupport
import ru.yandex.direct.core.entity.campaign.service.type.update.CampaignUpdateOperationSupportFacade
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer
import ru.yandex.direct.core.entity.campaign.service.validation.type.disabled.DisabledFieldsDataContainer
import ru.yandex.direct.core.entity.strategy.container.AbstractStrategyOperationContainer
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.service.converter.StrategyToCampaignConverterFacade
import ru.yandex.direct.core.entity.strategy.type.common.CommonStrategyUpdateLastChangeService
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.rbac.RbacService

@Service
class CampaignUpdateHelper(
    private val campaignModifyRepository: CampaignModifyRepository,
    private val campaignAdditionalActionsService: CampaignAdditionalActionsService,
    private val strategyToCampaignConverterFacade: StrategyToCampaignConverterFacade,
    private val rbacService: RbacService,
    private val metrikaClientFactory: RequestBasedMetrikaClientFactory,
    private val commonStrategyUpdateLastChangeService: CommonStrategyUpdateLastChangeService,
    campaignUpdateOperationSupportList: List<BaseCampaignUpdateOperationSupport<*>>
) {

    private val campaignUpdateOperationSupportFacade: CampaignUpdateOperationSupportFacade =
        CampaignUpdateOperationSupportFacade(
            campaignUpdateOperationSupportList,
            emptySet()
        )

    fun updateCampaignsInTransaction(
        strategyContainer: AbstractStrategyOperationContainer,
        dslContext: DSLContext,
        appliedChanges: List<AppliedChanges<CommonStrategy>>
    ) {
        if (!strategyContainer.options.isFromCampaignOperation) {
            val campaignToStrategy = campaignToStrategy(appliedChanges, strategyContainer)
            val campaignAppliedChanges = toCampaignAppliedChanges(strategyContainer, campaignToStrategy)
            if (campaignAppliedChanges.isNotEmpty()) {
                commonStrategyUpdateLastChangeService.updateLastChangeForCampaignsStrategies(
                    dslContext, strategyContainer, campaignAppliedChanges
                )
            }
            updateCampaigns(strategyContainer, dslContext, campaignAppliedChanges)
        }
    }

    fun updateCampaignsInTransaction(
        dslContext: DSLContext,
        strategyContainer: AbstractStrategyOperationContainer,
        models: List<CommonStrategy>
    ) {
        val campaignToStrategy = campaignToStrategy(strategyContainer, models)

        //важно, что мы не хотим обновлять стратегию внутри кампании, если она уже сохранена корректно,
        //чтобы исключить возможные неточности в сравнении полей в appliedChanges, явно указываем откуда
        // создается стратегия
        if (strategyContainer.options.isFromCampaignOperation) {
            updateCampaignsStrategyId(dslContext, campaignToStrategy)
        } else {
            if (campaignToStrategy.isNotEmpty()) {
                commonStrategyUpdateLastChangeService.updateLastChangeForCampaignsStrategies(
                    dslContext, strategyContainer, campaignToStrategy.keys
                )
            }
            val appliedChanges = toCampaignAppliedChanges(strategyContainer, campaignToStrategy)
            updateCampaigns(strategyContainer, dslContext, appliedChanges)
        }
    }

    fun updateCampaignsOutOfTransaction(
        appliedChanges: List<AppliedChanges<out BaseStrategy>>,
        container: AbstractStrategyOperationContainer
    ) {
        val commonStrategyAppliedChanges = appliedChanges.mapNotNull { it as? AppliedChanges<CommonStrategy> }
        val campaignToStrategy = campaignToStrategy(commonStrategyAppliedChanges, container)
        updateRelatedEntitiesOutOfTransaction(
            container,
            toCampaignAppliedChanges(container, campaignToStrategy)
        )
    }

    private fun updateRelatedEntitiesOutOfTransaction(
        container: AbstractStrategyOperationContainer,
        appliedChanges: List<AppliedChanges<CampaignWithPackageStrategy>>
    ) {
        val chiefUid: Long = rbacService.getChiefByClientId(container.clientId)
        val campaignContainer = RestrictedCampaignsUpdateOperationContainer.create(
            container.shard,
            container.operator,
            container.clientId,
            container.clientUid,
            chiefUid,
            CampaignOptions.Builder().withIsFromStrategyOperation(true).build(),
            null,
            DisabledFieldsDataContainer( appliedChanges.map { c -> c.model.id } )
        )
        campaignUpdateOperationSupportFacade.updateRelatedEntitiesOutOfTransaction(
            campaignContainer,
            appliedChanges
        )
    }

    fun updateCampaignsOutOfTransaction(
        container: AbstractStrategyOperationContainer,
        models: List<CommonStrategy>
    ) {
        val campaignToStrategy = campaignToStrategy(container, models)
        updateRelatedEntitiesOutOfTransaction(
            container,
            toCampaignAppliedChanges(container, campaignToStrategy)
        )
    }

    private fun campaignToStrategy(
        strategyContainer: AbstractStrategyOperationContainer,
        models: List<CommonStrategy>
    ): Map<CampaignWithPackageStrategy, CommonStrategy> {
        return models.flatMap { strategy ->
            strategyContainer.campaigns(strategy)
                .map { campaign ->
                    campaign to strategy
                }
        }.toMap()
    }

    private fun campaignToStrategy(
        changes: List<AppliedChanges<CommonStrategy>>,
        strategyContainer: AbstractStrategyOperationContainer
    ): Map<CampaignWithPackageStrategy, CommonStrategy> {
        return changes.flatMap { appliedChanges ->
            val cidsToUpdate = getCampaignIdsToUpdate(appliedChanges)
            strategyContainer.campaigns(appliedChanges.model)
                .filter { cidsToUpdate.contains(it.id) }
                .map { it to appliedChanges.model }
        }.toMap()
    }

    private fun getCampaignIdsToUpdate(strategyChanges: AppliedChanges<CommonStrategy>): Set<Long> {
        if (!StrategyConstants.STRATEGY_PROPERTIES_TO_IGNORE_FOR_CAMPAIGN_CHANGE.containsAll(strategyChanges.actuallyChangedProps)) {
            return strategyChanges.model.cids?.toSet() ?: emptySet()
        } else {
            val oldCampaignIds = strategyChanges.getOldValue(CommonStrategy.CIDS)?.toSet() ?: emptySet()
            val newCampaignIds = strategyChanges.getNewValue(CommonStrategy.CIDS)?.toSet() ?: emptySet()

            val campaignIdsChanged = strategyChanges.changed(CommonStrategy.CIDS) && oldCampaignIds != newCampaignIds
            if (campaignIdsChanged) {
                return newCampaignIds - oldCampaignIds
            }
        }
        return emptySet()
    }

    private fun toCampaignAppliedChanges(
        strategyContainer: AbstractStrategyOperationContainer,
        campaignToStrategy: Map<CampaignWithPackageStrategy, CommonStrategy>
    ): List<AppliedChanges<CampaignWithPackageStrategy>> {
        return campaignToStrategy.map { (campaign, strategy) ->
            val changes = strategyToCampaignConverterFacade.copyStrategyToCampaignModelChanges(
                strategyContainer.now,
                strategy,
                ModelChanges(campaign.id, CampaignWithPackageStrategy::class.java),
                campaign.javaClass
            )
            revertCampaignFields(campaign, changes)
            changes.process(CampaignStatusBsSynced.NO, CommonCampaign.STATUS_BS_SYNCED)
            changes.applyTo(campaign)
        }
    }

    private fun revertCampaignFields(
        campaign: CampaignWithPackageStrategy,
        mc: ModelChanges<CampaignWithPackageStrategy>
    ) {
        val newStrategy = mc.getChangedProp(CampaignWithPackageStrategy.STRATEGY)
        val oldStrategy = campaign.strategy
        if (newStrategy != null && oldStrategy != null) {
            newStrategy.platform = oldStrategy.platform
            newStrategy.strategy = oldStrategy.strategy
        }
    }

    private fun updateCampaignsStrategyId(
        dslContext: DSLContext,
        campaignToStrategy: Map<CampaignWithPackageStrategy, CommonStrategy>
    ) {
        val appliedChanges = campaignToStrategy.map { (campaign, strategy) ->
            val changes = ModelChanges(campaign.id, CampaignWithPackageStrategy::class.java)
                .process(strategy.id, CampaignWithPackageStrategy.STRATEGY_ID)
            changes.applyTo(campaign)
        }
        if (appliedChanges.isNotEmpty()) {
            campaignModifyRepository.updateCampaignsTable(dslContext, appliedChanges)
        }
    }

    private fun updateCampaigns(
        strategyContainer: AbstractStrategyOperationContainer,
        dslContext: DSLContext,
        appliedChanges: List<AppliedChanges<CampaignWithPackageStrategy>>
    ) {
        val additionalActionsContainer = CampaignAdditionalActionsContainer()
        if (appliedChanges.isNotEmpty()) {
            val metrikaClient = metrikaClientFactory.createMetrikaClient(strategyContainer.clientId)
            val chiefUid: Long = rbacService.getChiefByClientId(strategyContainer.clientId)
            val container = RestrictedCampaignsUpdateOperationContainer.create(
                strategyContainer.shard,
                strategyContainer.operator,
                strategyContainer.clientId,
                strategyContainer.clientUid,
                chiefUid,
                CampaignOptions.Builder().withIsFromStrategyOperation(true).build(),
                metrikaClient,
                DisabledFieldsDataContainer( appliedChanges.map { c -> c.model.id } )
            )
            campaignModifyRepository.updateCampaigns(dslContext, container, appliedChanges)
            campaignUpdateOperationSupportFacade.updateRelatedEntitiesInTransaction(
                dslContext,
                container,
                appliedChanges
            )
            campaignUpdateOperationSupportFacade.addToAdditionalActionsContainer(
                additionalActionsContainer,
                container,
                appliedChanges
            )
            campaignAdditionalActionsService.processAdditionalActionsContainer(
                dslContext,
                chiefUid,
                additionalActionsContainer
            )
        }
    }
}
