package ru.yandex.direct.core.entity.strategy.service

import org.jooq.DSLContext
import org.springframework.stereotype.Service
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartData
import ru.yandex.direct.autobudget.restart.model.PackageStrategyRestartResult
import ru.yandex.direct.autobudget.restart.repository.PackageStrategyAutobudgetRestartRepository
import ru.yandex.direct.autobudget.restart.service.AutobudgetRestartService
import ru.yandex.direct.autobudget.restart.service.CampaignAutobudgetRestartContainer
import ru.yandex.direct.core.entity.campaign.CampaignAutobudgetRestartUtils
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.service.WalletHasMoneyChecker
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.service.PublicPackageStrategyAutobudgetRestartService.Companion.buildStrategyRestartData
import ru.yandex.direct.dbutil.wrapper.DslContextProvider

@Service
class PrivatePackageStrategyAutobudgetRestartService(
    private val campAutobudgetRestartService: AutobudgetRestartService,
    private val hasMoneyChecker: WalletHasMoneyChecker,
    private val metrikaCampaignRepository: MetrikaCampaignRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val strategyAutobudgetRestartRepository: PackageStrategyAutobudgetRestartRepository,
    private val dslProvider: DslContextProvider
) {
    fun recalculateAndSaveRestarts(shard: Int, strategies: List<CommonStrategy>): List<PackageStrategyRestartResult> {
        require(allIsPrivate(strategies)) { "Ожидаются только непубличные стратегии" }
        val strategyByCampaignId =
            strategies
                .filter { !it.cids.isNullOrEmpty() }
                .associateBy { it.cids.first() }
        val restarts = recalculateAndSaveRestarts(shard, strategyByCampaignId)

        return restarts.mapNotNull { container ->
            val result = container.restartResult
            strategyByCampaignId[result.cid]?.let { strategy ->
                PackageStrategyRestartResult(
                    strategyId = strategy.id,
                    restartTime = result.restartTime,
                    softRestartTime = result.softRestartTime,
                    restartReason = result.restartReason,
                    state = container.restartDbData.state
                )
            }
        }
    }

    private fun recalculateAndSaveRestarts(
        shard: Int,
        strategyByCampaignId: Map<Long, CommonStrategy>
    ): MutableList<CampaignAutobudgetRestartContainer> {
        val campaignRestartData = getCampaignRestartData(shard, strategyByCampaignId.keys)
        val campRestarts = mutableListOf<CampaignAutobudgetRestartContainer>()
        dslProvider.ppcTransaction(shard) { configuration ->
            val dslContext = configuration.dsl()
            campRestarts += campAutobudgetRestartService.calculateRestartsAndSave(dslContext, campaignRestartData)
            spreadRestartsToStrategies(dslContext, strategyByCampaignId, campRestarts)
        }
        return campRestarts
    }

    internal fun spreadRestartsToStrategies(
        dslContext: DSLContext,
        strategyByCampaignId: Map<Long, CommonStrategy>,
        restarts: List<CampaignAutobudgetRestartContainer>
    ) {
        val restartsByCampaignId = restarts.associateBy { it.restartResult.cid }
        val strategyRestarts = restartsByCampaignId.mapNotNull { (cid, restart) ->
            buildStrategyRestartData(restart.restartDbData, strategyByCampaignId[cid])
        }
        strategyAutobudgetRestartRepository.saveAutobudgetRestartData(dslContext, strategyRestarts)
    }

    private fun getCampaignRestartData(shard: Int, campaignsIds: Set<Long>): List<CampStrategyRestartData> {
        val campaigns = campaignTypedRepository.getTypedCampaigns(shard, campaignsIds)
            .mapNotNull { it as? CommonCampaign }
        val wallets =
            campaignTypedRepository.getSafely(shard, campaigns.map { it.walletId }, WalletTypedCampaign::class.java)
        val hasMoneyByCid = hasMoneyChecker.calcHasMoney(
            shard,
            campaigns,
            wallets
        )
        val campaignIdsWithCombinedGoals =
            metrikaCampaignRepository.getCampaignIdsWithCombinedGoals(shard, campaignsIds)
        return campaigns.map {
            val dto = CampaignAutobudgetRestartUtils.getStrategyDto(
                it,
                hasMoney = hasMoneyByCid[it.id]!!,
                campaignIdsWithCombinedGoals.contains(it.id)
            )
            CampStrategyRestartData(it.id, dto)
        }
    }

    private fun allIsPrivate(strategies: List<CommonStrategy>): Boolean =
        strategies.find { !isPrivate(it) } == null

    private fun isPrivate(strategy: CommonStrategy): Boolean =
        !strategy.isPublic && (strategy.cids ?: emptyList()).size <= 1
}
